/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.core.io.buffer;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.Channel;
import java.nio.channels.Channels;
import java.nio.channels.CompletionHandler;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscription;
import org.springframework.core.io.Resource;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferFactory;
import org.springframework.core.io.buffer.DefaultDataBufferFactory;
import org.springframework.core.io.buffer.PooledDataBuffer;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import reactor.core.CoreSubscriber;
import reactor.core.Disposable;
import reactor.core.publisher.BaseSubscriber;
import reactor.core.publisher.Flux;
import reactor.core.publisher.FluxSink;
import reactor.core.publisher.Mono;
import reactor.core.publisher.SynchronousSink;

public abstract class DataBufferUtils {
    private static final Consumer<DataBuffer> RELEASE_CONSUMER = DataBufferUtils::release;
    private static final DataBuffer END_FRAME = new DefaultDataBufferFactory().wrap(new byte[0]);

    public static Flux<DataBuffer> readInputStream(Callable<InputStream> inputStreamSupplier, DataBufferFactory bufferFactory, int bufferSize) {
        Assert.notNull(inputStreamSupplier, "'inputStreamSupplier' must not be null");
        return DataBufferUtils.readByteChannel(() -> Channels.newChannel((InputStream)inputStreamSupplier.call()), bufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> readByteChannel(Callable<ReadableByteChannel> channelSupplier, DataBufferFactory bufferFactory, int bufferSize) {
        Assert.notNull(channelSupplier, "'channelSupplier' must not be null");
        Assert.notNull((Object)bufferFactory, "'dataBufferFactory' must not be null");
        Assert.isTrue(bufferSize > 0, "'bufferSize' must be > 0");
        return Flux.using(channelSupplier, channel -> Flux.generate((Consumer)new ReadableByteChannelGenerator((ReadableByteChannel)channel, bufferFactory, bufferSize)), DataBufferUtils::closeChannel);
    }

    public static Flux<DataBuffer> readAsynchronousFileChannel(Callable<AsynchronousFileChannel> channelSupplier, DataBufferFactory bufferFactory, int bufferSize) {
        return DataBufferUtils.readAsynchronousFileChannel(channelSupplier, 0L, bufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> readAsynchronousFileChannel(Callable<AsynchronousFileChannel> channelSupplier, long position, DataBufferFactory bufferFactory, int bufferSize) {
        Assert.notNull(channelSupplier, "'channelSupplier' must not be null");
        Assert.notNull((Object)bufferFactory, "'dataBufferFactory' must not be null");
        Assert.isTrue(position >= 0L, "'position' must be >= 0");
        Assert.isTrue(bufferSize > 0, "'bufferSize' must be > 0");
        Flux flux = Flux.using(channelSupplier, channel -> Flux.create(sink -> {
            ReadCompletionHandler handler = new ReadCompletionHandler((AsynchronousFileChannel)channel, (FluxSink<DataBuffer>)sink, position, bufferFactory, bufferSize);
            sink.onDispose(handler::dispose);
            DataBuffer dataBuffer = bufferFactory.allocateBuffer(bufferSize);
            ByteBuffer byteBuffer = dataBuffer.asByteBuffer(0, bufferSize);
            channel.read(byteBuffer, position, dataBuffer, handler);
        }), channel -> {});
        return flux.doOnDiscard(PooledDataBuffer.class, DataBufferUtils::release);
    }

    public static Flux<DataBuffer> read(Resource resource, DataBufferFactory bufferFactory, int bufferSize) {
        return DataBufferUtils.read(resource, 0L, bufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> read(Resource resource, long position, DataBufferFactory bufferFactory, int bufferSize) {
        try {
            if (resource.isFile()) {
                File file = resource.getFile();
                return DataBufferUtils.readAsynchronousFileChannel(() -> AsynchronousFileChannel.open(file.toPath(), StandardOpenOption.READ), position, bufferFactory, bufferSize);
            }
        }
        catch (IOException file) {
        }
        Flux<DataBuffer> result = DataBufferUtils.readByteChannel(resource::readableChannel, bufferFactory, bufferSize);
        return position == 0L ? result : DataBufferUtils.skipUntilByteCount(result, position);
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source, OutputStream outputStream) {
        Assert.notNull(source, "'source' must not be null");
        Assert.notNull((Object)outputStream, "'outputStream' must not be null");
        WritableByteChannel channel = Channels.newChannel(outputStream);
        return DataBufferUtils.write(source, channel);
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source, WritableByteChannel channel) {
        Assert.notNull(source, "'source' must not be null");
        Assert.notNull((Object)channel, "'channel' must not be null");
        Flux flux = Flux.from(source);
        return Flux.create(sink -> {
            WritableByteChannelSubscriber subscriber = new WritableByteChannelSubscriber((FluxSink<DataBuffer>)sink, channel);
            sink.onDispose((Disposable)subscriber);
            flux.subscribe((CoreSubscriber)subscriber);
        });
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source, AsynchronousFileChannel channel) {
        return DataBufferUtils.write(source, channel, 0L);
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source, AsynchronousFileChannel channel, long position) {
        Assert.notNull(source, "'source' must not be null");
        Assert.notNull((Object)channel, "'channel' must not be null");
        Assert.isTrue(position >= 0L, "'position' must be >= 0");
        Flux flux = Flux.from(source);
        return Flux.create(sink -> {
            WriteCompletionHandler handler = new WriteCompletionHandler((FluxSink<DataBuffer>)sink, channel, position);
            sink.onDispose((Disposable)handler);
            flux.subscribe((CoreSubscriber)handler);
        });
    }

    static void closeChannel(@Nullable Channel channel) {
        if (channel != null && channel.isOpen()) {
            try {
                channel.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public static Flux<DataBuffer> takeUntilByteCount(Publisher<DataBuffer> publisher, long maxByteCount) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(maxByteCount >= 0L, "'maxByteCount' must be a positive number");
        AtomicLong countDown = new AtomicLong(maxByteCount);
        return Flux.from(publisher).map(buffer -> {
            long remainder = countDown.addAndGet(-buffer.readableByteCount());
            if (remainder < 0L) {
                int length = buffer.readableByteCount() + (int)remainder;
                return buffer.slice(0, length);
            }
            return buffer;
        }).takeUntil(buffer -> countDown.get() <= 0L);
    }

    public static Flux<DataBuffer> skipUntilByteCount(Publisher<DataBuffer> publisher, long maxByteCount) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(maxByteCount >= 0L, "'maxByteCount' must be a positive number");
        return Flux.defer(() -> {
            AtomicLong countDown = new AtomicLong(maxByteCount);
            return Flux.from((Publisher)publisher).skipUntil(buffer -> {
                long remainder = countDown.addAndGet(-buffer.readableByteCount());
                return remainder < 0L;
            }).map(buffer -> {
                long remainder = countDown.get();
                if (remainder < 0L) {
                    countDown.set(0L);
                    int start = buffer.readableByteCount() + (int)remainder;
                    int length = (int)(-remainder);
                    return buffer.slice(start, length);
                }
                return buffer;
            });
        }).doOnDiscard(PooledDataBuffer.class, DataBufferUtils::release);
    }

    public static <T extends DataBuffer> T retain(T dataBuffer) {
        if (dataBuffer instanceof PooledDataBuffer) {
            return (T)((PooledDataBuffer)dataBuffer).retain();
        }
        return dataBuffer;
    }

    public static boolean release(@Nullable DataBuffer dataBuffer) {
        PooledDataBuffer pooledDataBuffer;
        if (dataBuffer instanceof PooledDataBuffer && (pooledDataBuffer = (PooledDataBuffer)dataBuffer).isAllocated()) {
            return pooledDataBuffer.release();
        }
        return false;
    }

    public static Consumer<DataBuffer> releaseConsumer() {
        return RELEASE_CONSUMER;
    }

    public static Mono<DataBuffer> join(Publisher<DataBuffer> dataBuffers) {
        Assert.notNull(dataBuffers, "'dataBuffers' must not be null");
        if (dataBuffers instanceof Mono) {
            return (Mono)dataBuffers;
        }
        return Flux.from(dataBuffers).collectList().filter(list -> !list.isEmpty()).map(list -> ((DataBuffer)list.get(0)).factory().join((List<? extends DataBuffer>)list)).doOnDiscard(PooledDataBuffer.class, DataBufferUtils::release);
    }

    public static Matcher matcher(byte[] delimiter) {
        Assert.isTrue(delimiter.length > 0, "Delimiter must not be empty");
        return new KnuthMorrisPrattMatcher(delimiter);
    }

    public static Flux<DataBuffer> split(Publisher<DataBuffer> dataBuffers, byte[] delimiter) {
        return DataBufferUtils.split(dataBuffers, delimiter, true);
    }

    public static Flux<DataBuffer> split(Publisher<DataBuffer> dataBuffers, byte[] delimiter, boolean stripDelimiter) {
        Assert.notNull(dataBuffers, "DataBuffers must not be null");
        Assert.isTrue(delimiter.length > 0, "Delimiter must not be empty");
        Matcher matcher = DataBufferUtils.matcher(delimiter);
        return Flux.from(dataBuffers).flatMap(buffer -> DataBufferUtils.endFrameOnDelimiter(buffer, matcher)).bufferUntil(buffer -> buffer == END_FRAME).map(buffers -> DataBufferUtils.joinAndStrip(buffers, delimiter, stripDelimiter)).doOnDiscard(PooledDataBuffer.class, DataBufferUtils::release);
    }

    private static Flux<DataBuffer> endFrameOnDelimiter(DataBuffer dataBuffer, Matcher matcher) {
        ArrayList<DataBuffer> result = new ArrayList<DataBuffer>();
        do {
            int endIdx = matcher.match(dataBuffer);
            int readPosition = dataBuffer.readPosition();
            if (endIdx == -1) {
                result.add(DataBufferUtils.retain(dataBuffer));
                break;
            }
            int length = endIdx + 1 - readPosition;
            result.add(dataBuffer.retainedSlice(readPosition, length));
            result.add(END_FRAME);
            dataBuffer.readPosition(endIdx + 1);
        } while (dataBuffer.readableByteCount() > 0);
        DataBufferUtils.release(dataBuffer);
        return Flux.fromIterable(result);
    }

    private static DataBuffer joinAndStrip(List<DataBuffer> dataBuffers, byte[] delimiter, boolean stripDelimiter) {
        Assert.state(!dataBuffers.isEmpty(), "DataBuffers should not be empty");
        boolean endFrameFound = false;
        int lastIdx = dataBuffers.size() - 1;
        if (dataBuffers.get(lastIdx) == END_FRAME) {
            endFrameFound = true;
            dataBuffers.remove(lastIdx);
        }
        DataBuffer result = dataBuffers.get(0).factory().join(dataBuffers);
        if (stripDelimiter && endFrameFound) {
            result.writePosition(result.writePosition() - delimiter.length);
        }
        return result;
    }

    private static class KnuthMorrisPrattMatcher
    implements Matcher {
        private final byte[] delimiter;
        private final int[] table;
        private int matches = 0;

        public KnuthMorrisPrattMatcher(byte[] delimiter) {
            this.delimiter = Arrays.copyOf(delimiter, delimiter.length);
            this.table = KnuthMorrisPrattMatcher.longestSuffixPrefixTable(delimiter);
        }

        private static int[] longestSuffixPrefixTable(byte[] delimiter) {
            int[] result = new int[delimiter.length];
            result[0] = 0;
            for (int i = 1; i < delimiter.length; ++i) {
                int j = result[i - 1];
                while (j > 0 && delimiter[i] != delimiter[j]) {
                    j = result[j - 1];
                }
                if (delimiter[i] == delimiter[j]) {
                    // empty if block
                }
                result[i] = ++j;
            }
            return result;
        }

        @Override
        public int match(DataBuffer dataBuffer) {
            for (int i = dataBuffer.readPosition(); i < dataBuffer.writePosition(); ++i) {
                byte b = dataBuffer.getByte(i);
                while (this.matches > 0 && b != this.delimiter[this.matches]) {
                    this.matches = this.table[this.matches - 1];
                }
                if (b != this.delimiter[this.matches]) continue;
                ++this.matches;
                if (this.matches != this.delimiter.length) continue;
                this.matches = 0;
                return i;
            }
            return -1;
        }

        public void reset() {
            this.matches = 0;
        }

        @Override
        public byte[] delimiter() {
            return Arrays.copyOf(this.delimiter, this.delimiter.length);
        }
    }

    private static class WriteCompletionHandler
    extends BaseSubscriber<DataBuffer>
    implements CompletionHandler<Integer, ByteBuffer> {
        private final FluxSink<DataBuffer> sink;
        private final AsynchronousFileChannel channel;
        private final AtomicBoolean completed = new AtomicBoolean();
        private final AtomicReference<Throwable> error = new AtomicReference();
        private final AtomicLong position;
        private final AtomicReference<DataBuffer> dataBuffer = new AtomicReference();

        public WriteCompletionHandler(FluxSink<DataBuffer> sink, AsynchronousFileChannel channel, long position) {
            this.sink = sink;
            this.channel = channel;
            this.position = new AtomicLong(position);
        }

        protected void hookOnSubscribe(Subscription subscription) {
            this.request(1L);
        }

        protected void hookOnNext(DataBuffer value) {
            if (!this.dataBuffer.compareAndSet(null, value)) {
                throw new IllegalStateException();
            }
            ByteBuffer byteBuffer = value.asByteBuffer();
            this.channel.write(byteBuffer, this.position.get(), byteBuffer, this);
        }

        protected void hookOnError(Throwable throwable) {
            this.error.set(throwable);
            if (this.dataBuffer.get() == null) {
                this.sink.error(throwable);
            }
        }

        protected void hookOnComplete() {
            this.completed.set(true);
            if (this.dataBuffer.get() == null) {
                this.sink.complete();
            }
        }

        @Override
        public void completed(Integer written, ByteBuffer byteBuffer) {
            long pos = this.position.addAndGet(written.intValue());
            if (byteBuffer.hasRemaining()) {
                this.channel.write(byteBuffer, pos, byteBuffer, this);
                return;
            }
            this.sinkDataBuffer();
            Throwable throwable = this.error.get();
            if (throwable != null) {
                this.sink.error(throwable);
            } else if (this.completed.get()) {
                this.sink.complete();
            } else {
                this.request(1L);
            }
        }

        @Override
        public void failed(Throwable exc, ByteBuffer byteBuffer) {
            this.sinkDataBuffer();
            this.sink.error(exc);
        }

        private void sinkDataBuffer() {
            DataBuffer dataBuffer = this.dataBuffer.get();
            Assert.state(dataBuffer != null, "DataBuffer should not be null");
            this.sink.next((Object)dataBuffer);
            this.dataBuffer.set(null);
        }
    }

    private static class WritableByteChannelSubscriber
    extends BaseSubscriber<DataBuffer> {
        private final FluxSink<DataBuffer> sink;
        private final WritableByteChannel channel;

        public WritableByteChannelSubscriber(FluxSink<DataBuffer> sink, WritableByteChannel channel) {
            this.sink = sink;
            this.channel = channel;
        }

        protected void hookOnSubscribe(Subscription subscription) {
            this.request(1L);
        }

        protected void hookOnNext(DataBuffer dataBuffer) {
            try {
                ByteBuffer byteBuffer = dataBuffer.asByteBuffer();
                while (byteBuffer.hasRemaining()) {
                    this.channel.write(byteBuffer);
                }
                this.sink.next((Object)dataBuffer);
                this.request(1L);
            }
            catch (IOException ex) {
                this.sink.next((Object)dataBuffer);
                this.sink.error((Throwable)ex);
            }
        }

        protected void hookOnError(Throwable throwable) {
            this.sink.error(throwable);
        }

        protected void hookOnComplete() {
            this.sink.complete();
        }
    }

    private static class ReadCompletionHandler
    implements CompletionHandler<Integer, DataBuffer> {
        private final AsynchronousFileChannel channel;
        private final FluxSink<DataBuffer> sink;
        private final DataBufferFactory dataBufferFactory;
        private final int bufferSize;
        private final AtomicLong position;
        private final AtomicBoolean disposed = new AtomicBoolean();

        public ReadCompletionHandler(AsynchronousFileChannel channel, FluxSink<DataBuffer> sink, long position, DataBufferFactory dataBufferFactory, int bufferSize) {
            this.channel = channel;
            this.sink = sink;
            this.position = new AtomicLong(position);
            this.dataBufferFactory = dataBufferFactory;
            this.bufferSize = bufferSize;
        }

        @Override
        public void completed(Integer read, DataBuffer dataBuffer) {
            if (read != -1 && !this.disposed.get()) {
                long pos = this.position.addAndGet(read.intValue());
                dataBuffer.writePosition(read);
                this.sink.next((Object)dataBuffer);
                if (this.disposed.get()) {
                    this.complete();
                } else {
                    DataBuffer newDataBuffer = this.dataBufferFactory.allocateBuffer(this.bufferSize);
                    ByteBuffer newByteBuffer = newDataBuffer.asByteBuffer(0, this.bufferSize);
                    this.channel.read(newByteBuffer, pos, newDataBuffer, this);
                }
            } else {
                DataBufferUtils.release(dataBuffer);
                this.complete();
            }
        }

        private void complete() {
            this.sink.complete();
            DataBufferUtils.closeChannel(this.channel);
        }

        @Override
        public void failed(Throwable exc, DataBuffer dataBuffer) {
            DataBufferUtils.release(dataBuffer);
            this.sink.error(exc);
            DataBufferUtils.closeChannel(this.channel);
        }

        public void dispose() {
            this.disposed.set(true);
        }
    }

    private static class ReadableByteChannelGenerator
    implements Consumer<SynchronousSink<DataBuffer>> {
        private final ReadableByteChannel channel;
        private final DataBufferFactory dataBufferFactory;
        private final int bufferSize;

        public ReadableByteChannelGenerator(ReadableByteChannel channel, DataBufferFactory dataBufferFactory, int bufferSize) {
            this.channel = channel;
            this.dataBufferFactory = dataBufferFactory;
            this.bufferSize = bufferSize;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void accept(SynchronousSink<DataBuffer> sink) {
            boolean release = true;
            DataBuffer dataBuffer = this.dataBufferFactory.allocateBuffer(this.bufferSize);
            try {
                ByteBuffer byteBuffer = dataBuffer.asByteBuffer(0, dataBuffer.capacity());
                int read = this.channel.read(byteBuffer);
                if (read >= 0) {
                    dataBuffer.writePosition(read);
                    release = false;
                    sink.next((Object)dataBuffer);
                } else {
                    sink.complete();
                }
            }
            catch (IOException ex) {
                sink.error((Throwable)ex);
            }
            finally {
                if (release) {
                    DataBufferUtils.release(dataBuffer);
                }
            }
        }
    }

    public static interface Matcher {
        public int match(DataBuffer var1);

        public byte[] delimiter();
    }
}

