/*
 * Copyright (C) 2015 Square, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.javapoet;

import static org.springframework.javapoet.Util.checkArgument;
import static org.springframework.javapoet.Util.checkNotNull;
import static org.springframework.javapoet.Util.checkState;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Modifier;

/** A generated field declaration. */
public final class FieldSpec {
    private final TypeName type;
    private final String name;
    private final CodeBlock javadoc;
    private final List<AnnotationSpec> annotations;
    private final Set<Modifier> modifiers;
    private final CodeBlock initializer;

    private FieldSpec(Builder builder) {
        this.type = checkNotNull(builder.type, "type == null");
        this.name = checkNotNull(builder.name, "name == null");
        this.javadoc = builder.javadoc.build();
        this.annotations = Util.immutableList(builder.annotations);
        this.modifiers = Util.immutableSet(builder.modifiers);
        this.initializer = (builder.initializer == null) ? CodeBlock.builder().build() : builder.initializer;
    }

    public TypeName type() {
        return type;
    }

    public String name() {
        return name;
    }

    public CodeBlock javadoc() {
        return javadoc;
    }

    public List<AnnotationSpec> annotations() {
        return annotations;
    }

    public Set<Modifier> modifiers() {
        return modifiers;
    }

    public CodeBlock initializer() {
        return initializer;
    }

    void emit(CodeWriter codeWriter, Set<Modifier> implicitModifiers) throws IOException {
        codeWriter.emitJavadoc(javadoc);
        codeWriter.emitAnnotations(annotations, false);
        codeWriter.emitModifiers(modifiers, implicitModifiers);
        codeWriter.emit("$T $L", type, name);
        if (!initializer.isEmpty()) {
            codeWriter.emit(" = ");
            codeWriter.emit(initializer);
        }
        codeWriter.emit(";\n");
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null) {
            return false;
        }
        if (getClass() != o.getClass()) {
            return false;
        }
        return toString().equals(o.toString());
    }

    @Override
    public int hashCode() {
        return toString().hashCode();
    }

    @Override
    public String toString() {
        StringBuilder out = new StringBuilder();
        try {
            CodeWriter codeWriter = new CodeWriter(out);
            emit(codeWriter, Collections.emptySet());
            return out.toString();
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static Builder builder(TypeName type, String name, Modifier... modifiers) {
        checkNotNull(type, "type == null");
        checkArgument(SourceVersion.isName(name), "not a valid name: %s", name);
        return new Builder(type, name).addModifiers(modifiers);
    }

    public static Builder builder(Type type, String name, Modifier... modifiers) {
        return builder(TypeName.get(type), name, modifiers);
    }

    public Builder toBuilder() {
        Builder builder = new Builder(type, name);
        builder.javadoc.add(javadoc);
        builder.annotations.addAll(annotations);
        builder.modifiers.addAll(modifiers);
        builder.initializer = initializer.isEmpty() ? null : initializer;
        return builder;
    }

    public static final class Builder {
        private final TypeName type;
        private final String name;

        private final CodeBlock.Builder javadoc = CodeBlock.builder();
        private CodeBlock initializer = null;

        private final List<AnnotationSpec> annotations = new ArrayList<>();
        private final List<Modifier> modifiers = new ArrayList<>();

        private Builder(TypeName type, String name) {
            this.type = type;
            this.name = name;
        }

        public Builder addJavadoc(String format, Object... args) {
            javadoc.add(format, args);
            return this;
        }

        public Builder addJavadoc(CodeBlock block) {
            javadoc.add(block);
            return this;
        }

        public Builder addAnnotations(Iterable<AnnotationSpec> annotationSpecs) {
            checkArgument(annotationSpecs != null, "annotationSpecs == null");
            for (AnnotationSpec annotationSpec : annotationSpecs) {
                this.annotations.add(annotationSpec);
            }
            return this;
        }

        public Builder addAnnotation(AnnotationSpec annotationSpec) {
            this.annotations.add(annotationSpec);
            return this;
        }

        public Builder addAnnotation(ClassName annotation) {
            this.annotations.add(AnnotationSpec.builder(annotation).build());
            return this;
        }

        public Builder addAnnotation(Class<?> annotation) {
            return addAnnotation(ClassName.get(annotation));
        }

        public Builder addModifiers(Modifier... modifiers) {
            Collections.addAll(this.modifiers, modifiers);
            return this;
        }

        public Builder initializer(String format, Object... args) {
            return initializer(CodeBlock.of(format, args));
        }

        public Builder initializer(CodeBlock codeBlock) {
            checkState(this.initializer == null, "initializer was already set");
            this.initializer = checkNotNull(codeBlock, "codeBlock == null");
            return this;
        }

        public FieldSpec build() {
            return new FieldSpec(this);
        }
    }
}
