/*
 * Copyright 2007 Yusuke Yamamoto
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package twitter4j;

import javax.crypto.spec.SecretKeySpec;
import java.util.Arrays;
import java.util.Objects;

abstract class OAuthToken implements java.io.Serializable {

    private static final long serialVersionUID = -7841506492508140600L;
    private final String token;
    private final String tokenSecret;

    private transient SecretKeySpec secretKeySpec;
    private String[] responseStr = null;

    OAuthToken(String token, String tokenSecret) {
        if (token == null)
            throw new IllegalArgumentException("Token can't be null");
        if (tokenSecret == null)
            throw new IllegalArgumentException("TokenSecret can't be null");
        this.token = token;
        this.tokenSecret = tokenSecret;
    }

    OAuthToken(HttpResponse response) throws TwitterException {
        this(response.asString());
    }

    OAuthToken(String string) {
        responseStr = string.split("&");
        tokenSecret = getParameter("oauth_token_secret");
        token = getParameter("oauth_token");
    }

    /**
     * returns token
     * @return token
     */
    public String getToken() {
        return token;
    }

    /**
     * @return token secret
     */
    public String getTokenSecret() {
        return tokenSecret;
    }

    /*package*/ void setSecretKeySpec(SecretKeySpec secretKeySpec) {
        this.secretKeySpec = secretKeySpec;
    }

    /*package*/ SecretKeySpec getSecretKeySpec() {
        return secretKeySpec;
    }

    String getParameter(String parameter) {
        String value = null;
        for (String str : responseStr) {
            if (str.startsWith(parameter + '=')) {
                value = str.split("=")[1].trim();
                break;
            }
        }
        return value;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        OAuthToken that = (OAuthToken) o;
        return Objects.equals(token, that.token) && Objects.equals(tokenSecret, that.tokenSecret) && Objects.equals(secretKeySpec, that.secretKeySpec) && Arrays.equals(responseStr, that.responseStr);
    }

    @Override
    public int hashCode() {
        int result = token.hashCode();
        result = 31 * result + tokenSecret.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "OAuthToken{" +
                "token='" + token + '\'' +
                ", tokenSecret='" + tokenSecret + '\'' +
                ", secretKeySpec=" + secretKeySpec +
                '}';
    }
}
