/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
define('macroEditorTranslationKeys', [], [
  'title',
  'changeMacro',
  'submit',
  'descriptorRequestFailed',
  'noParameters',
  'content',
  'more'
]);

define('macroEditor', ['jquery', 'modal', 'l10n!macroEditor'], function($, $modal, translations) {
  'use strict';
  var macroDescriptors = {},

  getMacroDescriptor = function(macroId) {
    var deferred = $.Deferred();
    var macroDescriptor = macroDescriptors[macroId];
    if (macroDescriptor) {
      deferred.resolve(macroDescriptor);
    } else {
      var url = new XWiki.Document('MacroService', 'CKEditor').getURL('get', 'outputSyntax=plain');
      $.get(url, {data: 'descriptor', macroId: macroId}).done(function(macroDescriptor) {
        if (typeof macroDescriptor === 'object' && macroDescriptor !== null) {
          macroDescriptors[macroId] = macroDescriptor;
          deferred.resolve(macroDescriptor);
        } else {
          deferred.reject.apply(deferred, arguments);
        }
      }).fail(function() {
        deferred.reject.apply(deferred, arguments);
      });
    }
    return deferred.promise();
  },

  macroEditorTemplate =
    '<div>' +
      '<div class="macro-name"/>' +
      '<div class="macro-description"/>' +
      '<ul class="macro-parameters"/>' +
    '</div>',

  createMacroEditor = function(macroCall, macroDescriptor) {
    var macroEditor = $(macroEditorTemplate);
    macroEditor.find('.macro-name').text(macroDescriptor.name);
    macroEditor.find('.macro-description').text(macroDescriptor.description);
    macroEditor.find('.macro-parameters').append(addParameterSeparator(
      sortMacroParameters(macroDescriptor, macroCall)).map(maybeDisplayMacroParameter));
    macroEditor.find('.more').click(toggleMacroParameters).click();
    this.removeClass('loading').data('macroDescriptor', macroDescriptor).append(macroEditor.children());
  },

  maybeSetParameterValue = function(parameter, macroCall) {
    var parameterCall = macroCall.parameters[parameter.id.toLowerCase()];
    // Check if the macro parameter is set.
    if (parameterCall !== null && parameterCall !== undefined) {
      parameter.value = (typeof parameterCall === 'object' && typeof parameterCall.name === 'string') ?
        parameterCall.value : parameterCall;
    }
  },

  sortMacroParameters = function(macroDescriptor, macroCall) {
    var parameter, parameters = [];
    // Add the actual macro parameters (those specified in the macro descriptor).
    if (macroDescriptor.parameterDescriptorMap) {
      for (var parameterId in macroDescriptor.parameterDescriptorMap) {
        if (macroDescriptor.parameterDescriptorMap.hasOwnProperty(parameterId)) {
          parameter = $.extend({}, macroDescriptor.parameterDescriptorMap[parameterId]);
          maybeSetParameterValue(parameter, macroCall);
          parameter.name = parameter.name || parameter.id;
          parameters.push(parameter);
        }
      }
    }
    // Handle the macro content as a special macro parameter.
    if (macroDescriptor.contentDescriptor) {
      parameter = $.extend({
        id: '$content',
        name: translations.get('content')
      }, macroDescriptor.contentDescriptor);
      if (typeof macroCall.content === 'string') {
        parameter.value = macroCall.content;
      }
      parameters.push(parameter);
    }
    parameters.sort(parameterComparator);
    return parameters;
  },

  parameterComparator = function(alice, bob) {
    if (alice.mandatory === bob.mandatory) {
      var aliceHasValue = alice.hasOwnProperty('value');
      var bobHasValue = bob.hasOwnProperty('value');
      if (aliceHasValue === bobHasValue) {
        // Macro descriptor order.
        return alice.index - bob.index;
      } else {
        // Parameters with value first.
        return bobHasValue - aliceHasValue;
      }
    } else {
      // Mandatory parameters first.
      return bob.mandatory - alice.mandatory;
    }
  },

  addParameterSeparator = function(parameters) {
    var i = 0;
    // Show the mandatory parameters and those that have been set.
    while (i < parameters.length && (parameters[i].mandatory || parameters[i].hasOwnProperty('value'))) {
      i++;
    }
    // If there are no mandatory parameters and no parameter is set then show the first three parameters.
    if (i === 0 && parameters.length > 3) {
      i = 3;
    }
    if (i > 0 && i < parameters.length) {
      // Show a 'more' link to toggle the remaining parameters.
      parameters.splice(i, 0, {id: 'more'});
    } else if (parameters.length === 0) {
      // Show a message for the empty list of parameters.
      parameters.push({id: 'empty'});
    }
    return parameters;
  },

  maybeDisplayMacroParameter = function(parameter) {
    if (parameter.name) {
      return displayMacroParameter(parameter);
    } else if (parameter.id === 'more') {
      return $('<li class="more"><span class="arrow arrow-down"/> <a href="#more"/></li>')
        .find('a').text(translations.get('more')).end();
    } else if (parameter.id === 'empty') {
      return $('<li class="empty"/>').text(translations.get('noParameters'));
    }
  },

  macroParameterTemplate =
    '<li class="macro-parameter">' +
      '<div class="macro-parameter-name"/>' +
      '<div class="macro-parameter-description"/>' +
    '</li>',

  displayMacroParameter = function(parameter) {
    var output = $(macroParameterTemplate);
    output.attr('data-id', parameter.id).attr('data-type', parameter.type);
    output.find('.macro-parameter-name').text(parameter.name);
    output.find('.macro-parameter-description').text(parameter.description);
    if (parameter.mandatory) {
      output.addClass('mandatory');
    }
    output.append(displayMacroParameterField(parameter));
    return output;
  },

  booleanValue = function(value) {
    if (typeof value === 'string') {
      return value === 'true';
    } else {
      return !!value;
    }
  },

  displayMacroParameterField = function(parameter) {
    var field;
    if (parameter.id === '$content') {
      field = $('<textarea/>');
    } else if (parameter.type === 'boolean') {
      field = $(
        '<div>' +
          '<input type="checkbox" value="true"/>' +
          '<input type="hidden" value="false"/>' +
        '</div>'
      );
      field.children('input').attr('name', parameter.id);
      var checked = booleanValue(parameter.hasOwnProperty('value') ? parameter.value : parameter.defaultValue);
      field.children('input[type=checkbox]').prop('checked', checked);
    } else if (parameter.type === 'enum') {
      field = $('<select/>');
      field.append(parameter.values.map(function(value) {
        return $('<option/>').attr('value', value.id).text(value.label);
      }));
    } else {
      field = $('<input type="text"/>');
    }
    field.addClass('macro-parameter-field').filter(':input').attr('name', parameter.id)
      .val(parameter.value || parameter.defaultValue);
    return field;
  },

  toggleMacroParameters = function(event) {
    event.preventDefault();
    var toggle = $(this);
    toggle.nextAll().toggleClass('hidden');
    var arrow = toggle.find('.arrow');
    if (arrow.hasClass('arrow-down')) {
      arrow.removeClass('arrow-down').addClass('arrow-right');
    } else {
      arrow.removeClass('arrow-right').addClass('arrow-down');
    }
  },

  maybeCreateMacroEditor = function(requestNumber, macroCall, macroDescriptor) {
    // Check if the macro descriptor corresponds to the last request.
    if (this.prop('requestNumber') === requestNumber) {
      createMacroEditor.call(this, macroCall, macroDescriptor);
      this.trigger('ready');
    }
  },

  maybeShowError = function(requestNumber) {
    // Check if the error corresponds to the last request.
    if (this.prop('requestNumber') === requestNumber) {
      this.removeClass('loading').append(
        '<div class="box errormessage">' +
          translations.get('descriptorRequestFailed', '<strong/>') +
        '</div>'
      ).find('strong').text(this.attr('data-macroId'));
    }
  },

  extractFormData = function(container) {
    var data = {};
    container.find(':input').serializeArray().forEach(function(parameter) {
      var value = data[parameter.name];
      if (value === undefined) {
        data[parameter.name] = parameter.value;
      } else if ($.isArray(value)) {
        value.push(parameter.value);
      } else {
        data[parameter.name] = [value, parameter.value];
      }
    });
    return data;
  },

  toMacroCall = function(formData, macroDescriptor) {
    if (!macroDescriptor) {
      return null;
    }
    var macroCall = {
      name: macroDescriptor.id.id,
      parameters: {}
    };
    if (typeof formData.$content === 'string' && formData.$content !== '' && macroDescriptor.contentDescriptor) {
      macroCall.content = formData.$content;
    }
    for (var parameterId in formData) {
      // The parameter descriptor map keys are lower case for easy lookup (macro parameter names are case insensitive).
      var parameterDescriptor = macroDescriptor.parameterDescriptorMap[parameterId.toLowerCase()];
      if (parameterDescriptor) {
        var value = formData[parameterId];
        if ($.isArray(value)) {
          value = parameterDescriptor.type === 'boolean' ? value[0] : value.join();
        }
        var defaultValue = parameterDescriptor.defaultValue;
        if (value !== '' && (defaultValue === undefined || defaultValue === null || (defaultValue + '') !== value)) {
          macroCall.parameters[parameterId] = value;
        }
      }
    }
    return macroCall;
  },

  createMacroEditorAPI = function(macroEditor) {
    return {
      focus: function() {
        macroEditor.find(':input').not(':hidden').first().focus();
      },
      getMacroCall: function() {
        return toMacroCall(extractFormData(macroEditor), macroEditor.data('macroDescriptor'));
      },
      validate: function() {
        var macroCall = this.getMacroCall();
        var emptyMandatoryParams = macroEditor.find('.macro-parameter.mandatory ').filter(function() {
          var id = $(this).attr('data-id');
          var value = id === '$content' ? macroCall.content : macroCall.parameters[id];
          return value === undefined;
        });
        emptyMandatoryParams.first().addClass('has-error').find(':input').not(':hidden').focus();
        setTimeout(function() {
          emptyMandatoryParams.first().removeClass('has-error');
        }, 1000);
        return emptyMandatoryParams.length === 0;
      },
      update: function(macroCall, syntaxId) {
        var macroId = macroCall.name;
        if (syntaxId) {
          macroId += '/' + syntaxId;
        }
        var requestNumber = (macroEditor.prop('requestNumber') || 0) + 1;
        macroEditor.empty().addClass('loading')
          .attr('data-macroId', macroId)
          .prop('requestNumber', requestNumber);
        getMacroDescriptor(macroId).done($.proxy(maybeCreateMacroEditor, macroEditor, requestNumber, macroCall))
          .fail($.proxy(maybeShowError, macroEditor, requestNumber));
      }
    };
  },

  editMacro = $modal.createModalStep({
    'class': 'macro-editor-modal',
    title: translations.get('title'),
    content: '<div class="macro-editor xform"/>',
    acceptLabel: translations.get('submit'),
    onLoad: function() {
      var modal = this;
      var submitButton = modal.find('.modal-footer .btn-primary');
      var changeMacroButton = $('<button type="button" class="btn btn-default"/>').text(translations.get('changeMacro'))
        .insertBefore(submitButton);
      modal.on('show.bs.modal', function(event) {
        submitButton.prop('disabled', true);
      }).on('shown.bs.modal', function(event) {
        var macroEditor = modal.find('.macro-editor');
        var macroEditorAPI = macroEditor.data('macroEditorAPI');
        var input = modal.data('input');
        var macroCall = input.macroCall || {
          name: input.macroId,
          parameters: {}
        };
        // We need to obey the specified macro identifier in case the user has just changed the macro.
        macroCall.name = input.macroId || macroCall.name;
        if (!macroEditorAPI) {
          // Initialize the macro editor.
          macroEditor.on('ready', function(event) {
            macroEditorAPI.focus();
            submitButton.prop('disabled', false);
          });
          macroEditorAPI = macroEditor.xwikiMacroEditor(macroCall, input.syntaxId);
        } else {
          macroEditorAPI.update(macroCall, input.syntaxId);
        }
      });
      submitButton.click(function(event) {
        var macroEditor = modal.find('.macro-editor');
        var macroEditorAPI = macroEditor.xwikiMacroEditor();
        if (macroEditorAPI.validate()) {
          var output = modal.data('input');
          delete output.action;
          // Preserve the in-line/block mode if possible. Note that we consider the macro in-line if no value is
          // specified, because the caret is placed in an in-line context most of the time (e.g. inside a paragraph) in
          // order to allow the user to type text).
          var inline = (!output.macroCall || output.macroCall.inline !== false) &&
            macroEditor.data('macroDescriptor').supportsInlineMode;
          output.macroCall = macroEditorAPI.getMacroCall();
          output.macroCall.inline = inline;
          modal.data('output', output).modal('hide');
        }
      });
      changeMacroButton.click(function(event) {
        var macroEditorAPI = modal.find('.macro-editor').xwikiMacroEditor();
        var output = modal.data('input');
        output.macroCall = macroEditorAPI.getMacroCall();
        output.action = 'changeMacro';
        modal.data('output', output).modal('hide');
      });
    }
  });

  $.fn.xwikiMacroEditor = function(macroCall, syntaxId) {
    this.each(function() {
      var macroEditor = $(this);
      if (!macroEditor.data('macroEditorAPI')) {
        var macroEditorAPI = createMacroEditorAPI(macroEditor);
        macroEditor.data('macroEditorAPI', macroEditorAPI);
        macroEditorAPI.update(macroCall, syntaxId);
      }
    });
    return this.data('macroEditorAPI');
  };

  return editMacro;
});

/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
define('macroSelectorTranslationKeys', [], [
  'title',
  'filter.text.placeholder',
  'filter.category.all',
  'filter.category.other',
  'failedToRetrieveMacros',
  'select'
]);

define('macroSelector', ['jquery', 'modal', 'l10n!macroSelector'], function($, $modal, translations) {
  'use strict';
  var macrosBySyntax = {},
  allMacrosExcludedCategories = [],

  getMacros = function(syntaxId) {
    var deferred = $.Deferred();
    var macros = macrosBySyntax[syntaxId || ''];
    if (macros) {
      deferred.resolve(macros);
    } else {
      var url = new XWiki.Document('MacroService', 'CKEditor').getURL('get', 'outputSyntax=plain');
      $.get(url, {data: 'list', syntaxId: syntaxId}).done(function(macros) {
        // Bulletproofing: check if the returned data is json since it could some HTML representing an error
        if (typeof macros === 'object' && $.isArray(macros.list)) {
          macrosBySyntax[syntaxId || ''] = macros.list;
          allMacrosExcludedCategories = macros.options.allMacrosExcludedCategories;
          deferred.resolve(macros.list);
        } else {
          deferred.reject.apply(deferred, arguments);
        }
      }).fail(function() {
        deferred.reject.apply(deferred, arguments);
      });
    }
    return deferred.promise();
  },

  macroListTemplate = '<ul class="macro-list form-control" tabindex="0"></ul>',
  macroListItemTemplate =
    '<li data-macroCategory="" data-macroId="">' +
      '<div class="macro-name"></div>' +
      '<div class="macro-description"></div>' +
    '</li>',

  displayMacros = function(macros) {
    var list = $(macroListTemplate);
    var categories = {};
    macros.forEach(function(macro) {
      var macroCategory = macro.defaultCategory || '';
      categories[macroCategory] = 1;
      var macroListItem = $(macroListItemTemplate).attr({
        'data-macroId': macro.id.id,
        'data-macroCategory': macroCategory
      }).appendTo(list);
      macroListItem.find('.macro-name').text(macro.name);
      macroListItem.find('.macro-description').text(macro.description);
    });
    categories = Object.keys(categories).sort();
    var categoryFilter = createCategoryFilter(categories);
    var textFilter = $(document.createElement('input')).attr({
      'type': 'text',
      'class': 'macro-textFilter',
      'placeholder': translations.get('filter.text.placeholder')
    });
    var filters = $(document.createElement('div')).addClass('macro-filters input-group');
    filters.append(textFilter).append(categoryFilter);
    this.removeClass('loading').append(filters).append(list);
    // Filter the list of displayed macros to implement support for allMacrosExcludedCategories (i.e. when all macros
    // is selected, don't display macros in some given categories). More generally this makes sure that the filtering
    // is always done.
    filterMacros.call(this);
  },

  createCategoryFilter = function(categories) {
    var categoryFilter = $(
      '<div class="macro-categories input-group-btn">' +
        '<button type="button" class="btn btn-default dropdown-toggle" data-toggle="dropdown" ' +
          'aria-haspopup="true" aria-expanded="false"><span class="caret"/></button>' +
        '<ul class="dropdown-menu dropdown-menu-right">' +
          '<li><a href="#"/></li>' +
        '</ul>' +
      '</div>'
    );
    var allMacrosCategoryName = translations.get('filter.category.all');
    categoryFilter.find('.caret').before(document.createTextNode(allMacrosCategoryName + ' '));
    categoryFilter.find('a').text(allMacrosCategoryName);
    var separator = '<li role="separator" class="divider"></li>';
    var categoryList = categoryFilter.find('ul.dropdown-menu');
    if (categories.length > 0) {
      categoryList.append(separator);
    }
    var otherCategory;
    categories.forEach(function(category) {
      var item = $('<li><a href="#"></a></li>').attr('data-category', category);
      item.children('a').text(category || translations.get('filter.category.other'));
      if (category) {
        categoryList.append(item);
      } else {
        otherCategory = item;
      }
    });
    if (otherCategory) {
      if (categories.length > 1) {
        categoryList.append(separator);
      }
      categoryList.append(otherCategory);
    }
    return categoryFilter;
  },

  scrollIntoList = function(item) {
    var itemPositionTop = item.position().top;
    var list = item.parent();
    if (itemPositionTop < 0) {
      list.scrollTop(list.scrollTop() + itemPositionTop);
    } else {
      var delta = itemPositionTop + item.outerHeight() - list.height();
      if (delta > 0) {
        list.scrollTop(list.scrollTop() + delta);
      }
    }
  },

  filterMacros = function() {
    var text = $(this).find('.macro-textFilter').val().toLowerCase();
    var selectedCategory = $(this).find('.macro-categories .dropdown-toggle').attr('data-category');
    var macroSelector = $(this).closest('.macro-selector');
    macroSelector.find('.macro-list').scrollTop(0).children().each(function() {
      var name = $(this).find('.macro-name').text().toLowerCase();
      var description = $(this).find('.macro-description').text().toLowerCase();
      var category = $(this).attr('data-macroCategory');
      // We hide Macros located in some categories to exclude (e.g. internal and deprecated categories) so that they
      // are less visible to users, to provide a simpler user experience by not bloating the macro list with
      // macros that are less interesting to users.
      // Note that when "All Macros" is selected selectedCategory is undefined.
      var hide = (text && name.indexOf(text) < 0 && description.indexOf(text) < 0) ||
        (typeof selectedCategory === 'string' && category !== selectedCategory) ||
          (typeof selectedCategory !== 'string' && $.inArray(category, allMacrosExcludedCategories) !== -1);
      $(this).removeClass('selected').toggleClass('hidden', hide);
    }).not('.hidden').first().addClass('selected');
    macroSelector.trigger('change');
  },

  navigateMacroList = function(macroList, up) {
    var direction = up ? 'prev' : 'next';
    var selectedItem = macroList.children('.selected');
    if (selectedItem.size() > 0) {
      selectedItem = selectedItem[direction]();
    } else {
      selectedItem = macroList.children()[up ? 'last' : 'first']();
    }
    while(selectedItem.hasClass('hidden')) {
      selectedItem = selectedItem[direction]();
    }
    selectedItem.click();
  },

  maybeTriggerMacroSelection = function(macroSelector) {
    var selectedMacros = macroSelector.find('.macro-list .selected').map(function() {
      return $(this).attr('data-macroId');
    });
    if (selectedMacros.length > 0) {
      macroSelector.trigger('xwiki:macro:selected', selectedMacros);
    }
  },

  changeMacroCategory = function(event) {
    event.preventDefault();
    var selectedCategory = $(this).parent('li');
    var categoryFilter = $(this).closest('.macro-categories');
    var dropDownToggle = categoryFilter.find('.dropdown-toggle');
    var newCategoryId = selectedCategory.attr('data-category');
    var oldCategoryId = dropDownToggle.attr('data-category');
    if (newCategoryId !== oldCategoryId) {
      var caret = dropDownToggle.children('.caret').remove();
      dropDownToggle.text(selectedCategory.text() + ' ').append(caret);
      if (typeof newCategoryId === 'string') {
        dropDownToggle.attr('data-category', newCategoryId);
      } else {
        dropDownToggle.removeAttr('data-category');
      }
      categoryFilter.trigger('change', newCategoryId, oldCategoryId);
    }
    dropDownToggle.focus();
  },

  addMacroSelectorBehaviour = function(macroSelector) {
    macroSelector.on('click', '.macro-categories a', changeMacroCategory);
    macroSelector.on('change', '.macro-categories', $.proxy(filterMacros, macroSelector));

    var timeoutId;
    macroSelector.on('input', '.macro-textFilter', function() {
      clearTimeout(timeoutId);
      timeoutId = setTimeout($.proxy(filterMacros, macroSelector), 500);
    });

    macroSelector.on('click', '.macro-list > li', function() {
      var item = $(this);
      item.addClass('selected').siblings().removeClass('selected');
      scrollIntoList(item);
      macroSelector.trigger('change');
    });

    macroSelector.on('keydown', '.macro-textFilter, .macro-list', function(event) {
      if (event.which === 38 || event.which === 40) {
        navigateMacroList(macroSelector.find('.macro-list'), event.which === 38);
        event.preventDefault();
      } else if (event.which === 13) {
        maybeTriggerMacroSelection(macroSelector);
      }
    });

    macroSelector.on('dblclick', '.macro-list', function() {
      maybeTriggerMacroSelection(macroSelector);
    });
  },

  maybeDisplayMacros = function(requestNumber, macros) {
    // Check if the list of macros corresponds to the last request.
    if (this.prop('requestNumber') === requestNumber) {
      displayMacros.call(this, macros);
      this.trigger('ready');
    }
  },

  maybeShowError = function(requestNumber) {
    // Check if the error corresponds to the last request.
    if (this.prop('requestNumber') === requestNumber) {
      var errorMessage = $('<div class="box errormessage"/>').text(translations.get('failedToRetrieveMacros'));
      this.removeClass('loading').append(errorMessage);
    }
  },

  createMacroSelectorAPI = function(macroSelector) {
    return {
      filter: function(text, category) {
        macroSelector.find('.macro-textFilter').val(text);
        macroSelector.find('.macro-categories .dropdown-toggle').attr('data-category', category);
        filterMacros.call(macroSelector[0]);
      },
      getSelectedMacro: function() {
        return macroSelector.find('.macro-list > li.selected').attr('data-macroId');
      },
      reset: function(macroId) {
        this.filter('');
        this.select(macroId);
      },
      select: function(macroId) {
        macroSelector.find('.macro-list > li').filter(function() {
          return $(this).attr('data-macroId') === macroId;
        }).click();
      },
      update: function(syntaxId) {
        syntaxId = syntaxId || macroSelector.attr('data-syntaxId');
        var requestNumber = (macroSelector.prop('requestNumber') || 0) + 1;
        macroSelector.empty().addClass('loading')
          .attr('data-syntaxId', syntaxId)
          .prop('requestNumber', requestNumber);
        getMacros(syntaxId).done($.proxy(maybeDisplayMacros, macroSelector, requestNumber))
          .fail($.proxy(maybeShowError, macroSelector, requestNumber));
      }
    };
  },

  selectMacro = $modal.createModalStep({
    'class': 'macro-selector-modal',
    title: translations.get('title'),
    content: '<div class="macro-selector loading"/>',
    acceptLabel: translations.get('select'),
    onLoad: function() {
      var modal = this;
      var selectButton = modal.find('.modal-footer .btn-primary');
      modal.on('shown.bs.modal', function(event) {
        var input = modal.data('input') || {};
        var macroSelector = modal.find('.macro-selector');
        var macroSelectorAPI = macroSelector.data('macroSelectorAPI');
        if (!macroSelectorAPI) {
          // Create the macro selector.
          macroSelector.on('ready', function() {
            macroSelectorAPI.select(input.macroId);
            macroSelector.find('.macro-textFilter').focus();
          }).on('change', function() {
            selectButton.prop('disabled', !macroSelectorAPI.getSelectedMacro());
          }).on('xwiki:macro:selected', function(event, macroIds) {
            selectButton.click();
          }).attr('data-syntaxId', input.syntaxId);
          macroSelectorAPI = macroSelector.xwikiMacroSelector();
        } else if (macroSelector.attr('data-syntaxId') !== input.syntaxId) {
          // Update the list of macros.
          macroSelectorAPI.update(input.syntaxId);
        } else {
          macroSelectorAPI.reset(input.macroId);
          macroSelector.find('.macro-textFilter').focus();
        }
      });
      selectButton.click(function() {
        var macroSelectorAPI = modal.find('.macro-selector').xwikiMacroSelector();
        var output = modal.data('input') || {};
        output.macroId = macroSelectorAPI.getSelectedMacro();
        modal.data('output', output).modal('hide');
      });
    }
  });

  $.fn.xwikiMacroSelector = function() {
    this.each(function() {
      var macroSelector = $(this);
      if (!macroSelector.data('macroSelectorAPI')) {
        var macroSelectorAPI = createMacroSelectorAPI(macroSelector);
        macroSelector.data('macroSelectorAPI', macroSelectorAPI);
        addMacroSelectorBehaviour(macroSelector);
        if (macroSelector.hasClass('loading')) {
          macroSelectorAPI.update();
        } else {
          macroSelector.trigger('ready');
        }
      }
    });
    return this.data('macroSelectorAPI');
  };

  return selectMacro;
});

/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
define('macroWizard', ['macroSelector', 'macroEditor'], function(selectMacro, editMacro) {
  'use strict';
  var selectOtherMacroOrFinish = function(data) {
    if (data.action === 'changeMacro') {
      delete data.action;
      data.macroId = data.macroCall && data.macroCall.name;
      return insertMacroWizard(data);
    } else {
      return data.macroCall;
    }
  },

  insertMacroWizard = function(data) {
    return selectMacro(data).then(editMacro).then(selectOtherMacroOrFinish);
  },

  editMacroWizard = function(macroCall, syntaxId) {
    return editMacro({
      macroCall: macroCall,
      syntaxId: syntaxId
    }).then(selectOtherMacroOrFinish);
  };

  return function(macroCall, syntaxId) {
    if (typeof macroCall === 'object' && macroCall !== null &&
        typeof macroCall.name === 'string' && macroCall.name !== '') {
      macroCall.parameters = macroCall.parameters || {};
      syntaxId = (typeof syntaxId === 'string' && syntaxId) || XWiki.docsyntax;
      return editMacroWizard(macroCall, syntaxId);
    } else {
      // The macro wizard can be called passing just the syntax as the first parameter.
      syntaxId = (typeof macroCall === 'string' && macroCall) ||
        (typeof syntaxId === 'string' && syntaxId) || XWiki.docsyntax;
      var data = {syntaxId: syntaxId};
      if (typeof macroCall === 'object' && macroCall !== null) {
        // We can pass default macro parameter values to the Insert Macro Wizard.
        data.macroCall = macroCall;
      }
      return insertMacroWizard(data);
    }
  };
});

