/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.resource.temporary;

import org.xwiki.model.reference.EntityReference;
import org.xwiki.resource.AbstractResourceReference;
import org.xwiki.resource.ResourceType;
import org.xwiki.stability.Unstable;

/**
 * Points to a temporary Resource that's been generated by some XWiki process (For example the Formula Macro generates
 * images of rendered formulas, the Chart Macro generates images of rendered chart data, etc).
 *
 * @version $Id: ff0fd2279498f714a3a0339a12f22c8fbd61997b $
 * @since 6.1M2
 */
@Unstable
public class TemporaryResourceReference extends AbstractResourceReference
{
    /**
     * Represents a Temporary Resource Type.
     */
    public static final ResourceType TYPE = new ResourceType("tmp");

    private EntityReference owningEntityReference;

    private String resourceName;

    private String moduleId;

    /**
     * @param moduleId see {@link #getModuleId()}
     * @param resourceName see {@link #getResourceName()}
     * @param owningEntityReference see {@link #getOwningEntityReference()}
     */
    public TemporaryResourceReference(String moduleId, String resourceName, EntityReference owningEntityReference)
    {
        setType(TYPE);
        this.moduleId = moduleId;
        this.resourceName = resourceName;
        this.owningEntityReference = owningEntityReference;
    }

    /**
     * @param moduleId see {@link #getModuleId()}
     * @param resourceName see {@link #getResourceName()}
     */
    public TemporaryResourceReference(String moduleId, String resourceName)
    {
        this(moduleId, resourceName, null);
    }

    /**
     * @return the reference to tne entity owning the current temporary resource. This can be used for example to verify
     *         that the user asking for the temporary resource has the permission to view the owning entity before
     *         letting him access the temporary resource.
     */
    public EntityReference getOwningEntityReference()
    {
        return this.owningEntityReference;
    }

    /**
     * @return the name of the temporary resource (eg the temporary file name of a generated image)
     */
    public String getResourceName()
    {
        return this.resourceName;
    }

    /**
     * @return the module id, a free name (used as a namespace) allowing several components to generate temporary
     *         resources for the same Entity
     */
    public String getModuleId()
    {
        return this.moduleId;
    }
}
