/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.wiki.internal.provisioning;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Provider;
import javax.inject.Singleton;

import org.slf4j.Logger;
import org.xwiki.component.annotation.Component;
import org.xwiki.job.event.status.JobProgressManager;
import org.xwiki.model.reference.DocumentReference;
import org.xwiki.model.reference.DocumentReferenceResolver;
import org.xwiki.model.reference.WikiReference;
import org.xwiki.query.Query;
import org.xwiki.query.QueryException;
import org.xwiki.query.QueryManager;
import org.xwiki.wiki.manager.WikiManagerException;
import org.xwiki.wiki.provisioning.WikiCopier;

import com.xpn.xwiki.XWiki;
import com.xpn.xwiki.XWikiContext;
import com.xpn.xwiki.XWikiException;

/**
 * Default implementation for {@link WikiCopier}.
 * 
 * @version $Id: 27fbaabadbaafabdc0c0e44b78a049c89e850985 $
 * @since 7.0M2
 */
@Component
@Singleton
public class DefaultWikiCopier implements WikiCopier
{
    @Inject
    private QueryManager queryManager;

    @Inject
    private Provider<XWikiContext> xcontextProvider;

    @Inject
    @Named("current")
    private DocumentReferenceResolver<String> documentReferenceResolver;

    @Inject
    private JobProgressManager progress;

    @Inject
    private Logger logger;

    @Override
    public void copyDocuments(String fromWikiId, String toWikiId, boolean withHistory) throws WikiManagerException
    {
        XWikiContext context = xcontextProvider.get();
        XWiki xwiki = context.getWiki();

        this.progress.pushLevelProgress(2, this);

        try {
            // Get documents

            this.progress.startStep(this, "Get documents to copy");

            Query query = queryManager.createQuery("select distinct doc.fullName from Document as doc", Query.XWQL);
            query.setWiki(fromWikiId);
            List<String> documentFullnames = query.execute();

            this.progress.endStep(this);

            // Copy documents

            this.progress.startStep(this, "Copy documents");
            this.progress.pushLevelProgress(documentFullnames.size(), this);

            WikiReference fromWikiReference = new WikiReference(fromWikiId);

            try {
                for (String documentFullName : documentFullnames) {
                    this.progress.startStep(this);

                    DocumentReference origDocReference =
                        documentReferenceResolver.resolve(documentFullName, fromWikiReference);
                    DocumentReference newDocReference = origDocReference.setWikiReference(new WikiReference(toWikiId));

                    logger.info("Copying document [{}] to [{}].", origDocReference, newDocReference);
                    xwiki.copyDocument(origDocReference, newDocReference, null, !withHistory, true, context);
                    logger.info("Done copying document [{}] to [{}].", origDocReference, newDocReference);

                    this.progress.endStep(this);
                }
            } finally {
                this.progress.popLevelProgress(this);
                this.progress.endStep(this);
            }
        } catch (QueryException e) {
            WikiManagerException thrownException =
                new WikiManagerException("Unable to get the list of wiki documents to copy.", e);
            logger.error(thrownException.getMessage(), thrownException);
            throw thrownException;
        } catch (XWikiException e) {
            WikiManagerException thrownException = new WikiManagerException("Failed to copy documents.", e);
            logger.error(thrownException.getMessage(), thrownException);
            throw thrownException;
        } finally {
            this.progress.popLevelProgress(this);
        }
    }

    @Override
    public void copyDeletedDocuments(String fromWikiId, String toWikiId) throws WikiManagerException
    {
        throw new WikiManagerException("This method is not implemented yet");
    }
}
