/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.gwt.wysiwyg.client.plugin.image;

import org.xwiki.gwt.dom.client.DOMUtils;
import org.xwiki.gwt.dom.client.Document;
import org.xwiki.gwt.dom.client.DocumentFragment;
import org.xwiki.gwt.dom.client.Element;
import org.xwiki.gwt.dom.client.InnerHTMLListener;
import org.xwiki.gwt.dom.client.Text;

import com.google.gwt.dom.client.Node;
import com.google.gwt.dom.client.NodeList;

/**
 * Analyzes the passed subtree and detects the image structures and transforms them in minimal HTML elements with
 * metafragments attached, so that the editor operates with minimal HTML.
 * 
 * @version $Id: d3014115822f28473fdb281631a31292e9486a2b $
 */
public class ImageMetaDataExtractor implements InnerHTMLListener
{
    @Override
    public void onInnerHTMLChange(Element parent)
    {
        // look up all images in this subtree
        NodeList<com.google.gwt.dom.client.Element> imgs = parent.getElementsByTagName("img");
        for (int i = 0; i < imgs.getLength(); i++) {
            Element img = (Element) imgs.getItem(i);
            processElement(img);
        }
    }

    /**
     * Processes the passed image element: tests if it is a valid image generated by the image plugin and creates the
     * metafragment, removing the neighbouring comments.
     * 
     * @param img the image element to process.
     */
    private void processElement(Element img)
    {
        // Check previous sibling to be image start comment
        Node previousSibling = img.getPreviousSibling();
        boolean foundComments =
            previousSibling != null && previousSibling.getNodeType() == DOMUtils.COMMENT_NODE
                && previousSibling.getNodeValue().startsWith("startimage");
        // Check next sibling to be image end comment
        Node nextSibling = img.getNextSibling();
        foundComments &=
            nextSibling != null && nextSibling.getNodeType() == DOMUtils.COMMENT_NODE
                && nextSibling.getNodeValue().startsWith("stopimage");
        if (!foundComments) {
            return;
        }
        // This is a valid image html fragment, transform this element in a metafragment element
        DocumentFragment metaFragment = ((Document) img.getOwnerDocument()).createDocumentFragment();
        metaFragment.appendChild(previousSibling);
        Text placeholder = (Text) ((Document) img.getOwnerDocument()).createTextNode(Element.INNER_HTML_PLACEHOLDER);
        metaFragment.appendChild(placeholder);
        metaFragment.appendChild(nextSibling);
        img.setMetaData(metaFragment);
    }
}
