/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.rendering.internal.macro.content;

import java.io.StringReader;
import java.util.List;

import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;

import org.xwiki.component.annotation.Component;
import org.xwiki.component.manager.ComponentLookupException;
import org.xwiki.component.manager.ComponentManager;
import org.xwiki.rendering.block.Block;
import org.xwiki.rendering.macro.AbstractMacro;
import org.xwiki.rendering.macro.MacroExecutionException;
import org.xwiki.rendering.macro.content.ContentMacroParameters;
import org.xwiki.rendering.macro.descriptor.DefaultContentDescriptor;
import org.xwiki.rendering.parser.ParseException;
import org.xwiki.rendering.parser.Parser;
import org.xwiki.rendering.syntax.Syntax;
import org.xwiki.rendering.transformation.MacroTransformationContext;

/**
 * Allows writing content in any syntax (wiki syntax, HTML, etc). This allows to have to main content in a given syntax
 * and mix content written in another syntax in it.
 *
 * @version $Id: 857c76de1fbb2112e8046355667409338d0ac1f3 $
 * @since 4.2M3
 */
@Component
@Named("content")
@Singleton
public class ContentMacro extends AbstractMacro<ContentMacroParameters>
{
    /**
     * The description of the macro.
     */
    private static final String DESCRIPTION = "Allows writing content in any wiki markup";

    /**
     * The description of the macro content.
     */
    private static final String CONTENT_DESCRIPTION = "The content to execute";

    /**
     * Used to find the Parser corresponding to the user-specified syntax for the Macro.
     */
    @Inject
    private ComponentManager componentManager;

    /**
     * Create and initialize the descriptor of the macro.
     */
    public ContentMacro()
    {
        super("Content", DESCRIPTION, new DefaultContentDescriptor(CONTENT_DESCRIPTION), ContentMacroParameters.class);
        setDefaultCategory(DEFAULT_CATEGORY_CONTENT);
    }

    @Override
    public boolean supportsInlineMode()
    {
        return false;
    }

    @Override
    public List<Block> execute(ContentMacroParameters parameters, String content, MacroTransformationContext context)
        throws MacroExecutionException
    {
        try {
            return getSyntaxParser(parameters.getSyntax()).parse(new StringReader(content)).getChildren();
        } catch (ParseException e) {
            throw new MacroExecutionException(
                String.format("Failed to parse macro content in syntax [%s]", parameters.getSyntax()), e);
        }
    }

    /**
     * Get the parser for the passed Syntax.
     *
     * @param syntax the Syntax for which to find the Parser
     * @return the matching Parser that can be used to parse content in the passed Syntax
     * @throws MacroExecutionException if there's no Parser in the system for the passed Syntax
     */
    protected Parser getSyntaxParser(Syntax syntax) throws MacroExecutionException
    {
        try {
            return this.componentManager.getInstance(Parser.class, syntax.toIdString());
        } catch (ComponentLookupException e) {
            throw new MacroExecutionException(String.format("Cannot find Parser for syntax [%s]", syntax.toIdString()));
        }
    }
}
