/**
 * <h1>CloudFront Origins for the CDK CloudFront Library</h1>
 * <p>
 * This library contains convenience methods for defining origins for a CloudFront distribution. You can use this library to create origins from
 * S3 buckets, Elastic Load Balancing v2 load balancers, or any other domain name.
 * <p>
 * <h2>S3 Bucket</h2>
 * <p>
 * An S3 bucket can be added as an origin. If the bucket is configured as a website endpoint, the distribution can use S3 redirects and S3 custom error
 * documents.
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new S3Origin(myBucket)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The above will treat the bucket differently based on if <code>IBucket.isWebsite</code> is set or not. If the bucket is configured as a website, the bucket is
 * treated as an HTTP origin, and the built-in S3 redirects and error pages can be used. Otherwise, the bucket is handled as a bucket origin and
 * CloudFront's redirect and error handling will be used. In the latter case, the Origin will create an origin access identity and grant it access to the
 * underlying bucket. This can be used in conjunction with a bucket that is not public to require that your users access your content using CloudFront
 * URLs and not S3 URLs directly. Alternatively, a custom origin access identity can be passed to the S3 origin in the properties.
 * <p>
 * <h3>Adding Custom Headers</h3>
 * <p>
 * You can configure CloudFront to add custom headers to the requests that it sends to your origin. These custom headers enable you to send and gather information from your origin that you don’t get with typical viewer requests. These headers can even be customized for each origin. CloudFront supports custom headers for both for custom and Amazon S3 origins.
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(S3Origin.Builder.create(myBucket)
 *                 .customHeaders(Map.of(
 *                         "Foo", "bar"))
 *                 .build()).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>ELBv2 Load Balancer</h2>
 * <p>
 * An Elastic Load Balancing (ELB) v2 load balancer may be used as an origin. In order for a load balancer to serve as an origin, it must be publicly
 * accessible (<code>internetFacing</code> is true). Both Application and Network load balancers are supported.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * Vpc vpc;
 * 
 * // Create an application load balancer in a VPC. 'internetFacing' must be 'true'
 * // for CloudFront to access the load balancer and use it as an origin.
 * ApplicationLoadBalancer lb = ApplicationLoadBalancer.Builder.create(this, "LB")
 *         .vpc(vpc)
 *         .internetFacing(true)
 *         .build();
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new LoadBalancerV2Origin(lb)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The origin can also be customized to respond on different ports, have different connection properties, etc.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.elasticloadbalancingv2.*;
 * 
 * ApplicationLoadBalancer loadBalancer;
 * 
 * LoadBalancerV2Origin origin = LoadBalancerV2Origin.Builder.create(loadBalancer)
 *         .connectionAttempts(3)
 *         .connectionTimeout(Duration.seconds(5))
 *         .readTimeout(Duration.seconds(45))
 *         .keepaliveTimeout(Duration.seconds(45))
 *         .protocolPolicy(OriginProtocolPolicy.MATCH_VIEWER)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that the <code>readTimeout</code> and <code>keepaliveTimeout</code> properties can extend their values over 60 seconds only if a limit increase request for CloudFront origin response timeout
 * quota has been approved in the target account; otherwise, values over 60 seconds will produce an error at deploy time. Consider that this value is
 * still limited to a maximum value of 180 seconds, which is a hard limit for that quota.
 * <p>
 * <h2>From an HTTP endpoint</h2>
 * <p>
 * Origins can also be created from any other HTTP endpoint, given the domain name, and optionally, other origin properties.
 * <p>
 * <blockquote><pre>
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new HttpOrigin("www.example.com")).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * See the documentation of <code>aws-cdk-lib/aws-cloudfront</code> for more information.
 * <p>
 * <h2>Failover Origins (Origin Groups)</h2>
 * <p>
 * You can set up CloudFront with origin failover for scenarios that require high availability.
 * To get started, you create an origin group with two origins: a primary and a secondary.
 * If the primary origin is unavailable, or returns specific HTTP response status codes that indicate a failure,
 * CloudFront automatically switches to the secondary origin.
 * You achieve that behavior in the CDK using the <code>OriginGroup</code> class:
 * <p>
 * <blockquote><pre>
 * Bucket myBucket = new Bucket(this, "myBucket");
 * Distribution.Builder.create(this, "myDist")
 *         .defaultBehavior(BehaviorOptions.builder()
 *                 .origin(OriginGroup.Builder.create()
 *                         .primaryOrigin(new S3Origin(myBucket))
 *                         .fallbackOrigin(new HttpOrigin("www.example.com"))
 *                         // optional, defaults to: 500, 502, 503 and 504
 *                         .fallbackStatusCodes(List.of(404))
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>From an API Gateway REST API</h2>
 * <p>
 * Origins can be created from an API Gateway REST API. It is recommended to use a
 * <a href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-endpoint-types.html">regional API</a> in this case. The origin path will automatically be set as the stage name.
 * <p>
 * <blockquote><pre>
 * RestApi api;
 * 
 * Distribution.Builder.create(this, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new RestApiOrigin(api)).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to use a different origin path, you can specify it in the <code>originPath</code> property.
 * <p>
 * <blockquote><pre>
 * RestApi api;
 * 
 * Distribution.Builder.create(this, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder().origin(RestApiOrigin.Builder.create(api).originPath("/custom-origin-path").build()).build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>From a Lambda Function URL</h2>
 * <p>
 * Lambda Function URLs enable direct invocation of Lambda functions via HTTP(S), without intermediaries. They can be set as CloudFront origins for streamlined function execution behind a CDN, leveraging caching and custom domains.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * Function fn;
 * 
 * FunctionUrl fnUrl = fn.addFunctionUrl(FunctionUrlOptions.builder().authType(FunctionUrlAuthType.NONE).build());
 * 
 * Distribution.Builder.create(this, "Distribution")
 *         .defaultBehavior(BehaviorOptions.builder().origin(new FunctionUrlOrigin(fnUrl)).build())
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.cloudfront.origins;
