/**
 * <h1>Amazon Lambda Node.js Library</h1>
 * <p>
 * This library provides constructs for Node.js Lambda functions.
 * <p>
 * <h2>Node.js Function</h2>
 * <p>
 * The <code>NodejsFunction</code> construct creates a Lambda function with automatic transpiling and bundling
 * of TypeScript or Javascript code. This results in smaller Lambda packages that contain only the
 * code and dependencies needed to run the function.
 * <p>
 * It uses <a href="https://esbuild.github.io/">esbuild</a> under the hood.
 * <p>
 * <h2>Reference project architecture</h2>
 * <p>
 * The <code>NodejsFunction</code> allows you to define your CDK and runtime dependencies in a single
 * package.json and to collocate your runtime code with your infrastructure code:
 * <p>
 * <blockquote><pre>
 * .
 * ├── lib
 * │   ├── my-construct.api.ts # Lambda handler for API
 * │   ├── my-construct.auth.ts # Lambda handler for Auth
 * │   └── my-construct.ts # CDK construct with two Lambda functions
 * ├── package-lock.json # single lock file
 * ├── package.json # CDK and runtime dependencies defined in a single package.json
 * └── tsconfig.json
 * </pre></blockquote>
 * <p>
 * By default, the construct will use the name of the defining file and the construct's
 * id to look up the entry file. In <code>my-construct.ts</code> above we have:
 * <p>
 * <blockquote><pre>
 * // automatic entry look up
 * NodejsFunction apiHandler = new NodejsFunction(this, "api");
 * NodejsFunction authHandler = new NodejsFunction(this, "auth");
 * </pre></blockquote>
 * <p>
 * Alternatively, an entry file and handler can be specified:
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "MyFunction")
 *         .entry("/path/to/my/file.ts") // accepts .js, .jsx, .cjs, .mjs, .ts, .tsx, .cts and .mts files
 *         .handler("myExportedFunc")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The handler value will be automatically prefixed with the bundled output file name, <code>index.</code>,
 * unless the handler value contains a <code>.</code> character, in which case the handler value is used as-is to
 * allow for values needed by some Lambda extensions.
 * <p>
 * For monorepos, the reference architecture becomes:
 * <p>
 * <blockquote><pre>
 * .
 * ├── packages
 * │   ├── cool-package
 * │   │   ├── lib
 * │   │   │   ├── cool-construct.api.ts
 * │   │   │   ├── cool-construct.auth.ts
 * │   │   │   └── cool-construct.ts
 * │   │   ├── package.json # CDK and runtime dependencies for cool-package
 * │   │   └── tsconfig.json
 * │   └── super-package
 * │       ├── lib
 * │       │   ├── super-construct.handler.ts
 * │       │   └── super-construct.ts
 * │       ├── package.json # CDK and runtime dependencies for super-package
 * │       └── tsconfig.json
 * ├── package-lock.json # single lock file
 * ├── package.json # root dependencies
 * └── tsconfig.json
 * </pre></blockquote>
 * <p>
 * <h2>Customizing the underlying Lambda function</h2>
 * <p>
 * All properties of <code>lambda.Function</code> can be used to customize the underlying <code>lambda.Function</code>.
 * <p>
 * See also the <a href="https://github.com/aws/aws-cdk/tree/main/packages/aws-cdk-lib/aws-lambda">AWS Lambda construct library</a>.
 * <p>
 * The <code>NodejsFunction</code> construct automatically <a href="https://docs.aws.amazon.com/sdk-for-javascript/v2/developer-guide/node-reusing-connections.html">reuses existing connections</a>
 * when working with the AWS SDK for JavaScript. Set the <code>awsSdkConnectionReuse</code> prop to <code>false</code> to disable it.
 * <p>
 * <h2>Runtime</h2>
 * <p>
 * When the <code>&#64;aws-cdk/aws-lambda-nodejs:useLatestRuntimeVersion</code> feature flag is enabled, the <code>NODEJS_LATEST</code> runtime
 * will be used by default. This runtime will be updated to use the latest Node.js version currently available in lambda.
 * Since this runtime can change from version to version, you should ensure that all of your dependencies are included
 * during packaging and avoid relying on depdendencies being globally installed. See <a href="#externals">externals</a> for details.
 * <p>
 * <strong>When using <code>NODEJS_LATEST</code> runtime make sure that all of your dependencies are included during bundling, or as layers.
 * Usage of globally installed packages in the lambda environment may cause your function to break in future versions. If
 * you need to rely on packages pre-installed in the lambda environment, you must explicitly set your runtime.</strong>
 * <p>
 * This can be set via <code>lambda.Runtime</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.Runtime;
 * 
 * 
 * NodejsFunction.Builder.create(this, "my-function")
 *         .runtime(Runtime.NODEJS_18_X)
 *         .build();
 * </pre></blockquote>
 * <p>
 * With the <code>&#64;aws-cdk/aws-lambda-nodejs:useLatestRuntimeVersion</code> disabled, the runtime will default to <code>NODEJ_16_X</code>.
 * <p>
 * <h2>Lock file</h2>
 * <p>
 * The <code>NodejsFunction</code> requires a dependencies lock file (<code>yarn.lock</code>, <code>pnpm-lock.yaml</code> or
 * <code>package-lock.json</code>). When bundling in a Docker container, the path containing this lock file is
 * used as the source (<code>/asset-input</code>) for the volume mounted in the container.
 * <p>
 * By default, the construct will try to automatically determine your project lock file.
 * Alternatively, you can specify the <code>depsLockFilePath</code> prop manually. In this
 * case you need to ensure that this path includes <code>entry</code> and any module/dependencies
 * used by your function. Otherwise bundling will fail.
 * <p>
 * <h2>Local bundling</h2>
 * <p>
 * If <code>esbuild</code> is available it will be used to bundle your code in your environment. Otherwise,
 * bundling will happen in a <a href="https://gallery.ecr.aws/sam/build-nodejs18.x">Lambda compatible Docker container</a>
 * with the Docker platform based on the target architecture of the Lambda function.
 * <p>
 * For macOS the recommended approach is to install <code>esbuild</code> as Docker volume performance is really poor.
 * <p>
 * <code>esbuild</code> can be installed with:
 * <p>
 * <blockquote><pre>
 * $ npm install --save-dev esbuild&#64;0
 * </pre></blockquote>
 * <p>
 * OR
 * <p>
 * <blockquote><pre>
 * $ yarn add --dev esbuild&#64;0
 * </pre></blockquote>
 * <p>
 * If you're using a monorepo layout, the <code>esbuild</code> dependency needs to be installed in the "root" <code>package.json</code> file,
 * not in the workspace. From the reference architecture described <a href="#reference-project-architecture">above</a>, the <code>esbuild</code>
 * dev dependency needs to be in <code>./package.json</code>, not <code>packages/cool-package/package.json</code> or
 * <code>packages/super-package/package.json</code>.
 * <p>
 * To force bundling in a Docker container even if <code>esbuild</code> is available in your environment,
 * set <code>bundling.forceDockerBundling</code> to <code>true</code>. This is useful if your function relies on node
 * modules that should be installed (<code>nodeModules</code> prop, see <a href="#install-modules">below</a>) in a Lambda
 * compatible environment. This is usually the case with modules using native dependencies.
 * <p>
 * <h2>Working with modules</h2>
 * <p>
 * <h3>Externals</h3>
 * <p>
 * When the <code>NODEJS_LATEST</code> runtime is used, no modules are excluded from bundling by default. This is because the runtime
 * will change as new NodeJs versions become available in lambda, which may change what packages are vended as part of the
 * environment.
 * <p>
 * When passing a runtime that is known to include a version of the aws sdk, it will be excluded by default. For example, when
 * passing <code>NODEJS_16_X</code>, <code>aws-sdk</code> is excluded. When passing <code>NODEJS_18_X</code>,  all <code>&#64;aws-sdk/*</code> packages are excluded.
 * <p>
 * This can be configured by specifying <code>bundling.externalModules</code>:
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .externalModules(List.of("&#64;aws-sdk/*", "cool-module"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Install modules</h3>
 * <p>
 * By default, all node modules referenced in your Lambda code will be bundled by <code>esbuild</code>.
 * Use the <code>nodeModules</code> prop under <code>bundling</code> to specify a list of modules that should not be
 * bundled but instead included in the <code>node_modules</code> folder of the Lambda package. This is useful
 * when working with native dependencies or when <code>esbuild</code> fails to bundle a module.
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .nodeModules(List.of("native-module", "other-module"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The modules listed in <code>nodeModules</code> must be present in the <code>package.json</code>'s dependencies or
 * installed. The same version will be used for installation. The lock file (<code>yarn.lock</code>,
 * <code>pnpm-lock.yaml</code> or <code>package-lock.json</code>) will be used along with the right installer (<code>yarn</code>,
 * <code>pnpm</code> or <code>npm</code>).
 * <p>
 * When working with <code>nodeModules</code> using native dependencies, you might want to force bundling in a
 * Docker container even if <code>esbuild</code> is available in your environment. This can be done by setting
 * <code>bundling.forceDockerBundling</code> to <code>true</code>.
 * <p>
 * <h2>Configuring <code>esbuild</code></h2>
 * <p>
 * The <code>NodejsFunction</code> construct exposes <a href="https://esbuild.github.io/api/#build-api">esbuild options</a>
 * via properties under <code>bundling</code>:
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .minify(true) // minify code, defaults to false
 *                 .sourceMap(true) // include source map, defaults to false
 *                 .sourceMapMode(SourceMapMode.INLINE) // defaults to SourceMapMode.DEFAULT
 *                 .sourcesContent(false) // do not include original source into source map, defaults to true
 *                 .target("es2020") // target environment for the generated JavaScript code
 *                 .loader(Map.of( // Use the 'dataurl' loader for '.png' files
 *                         ".png", "dataurl"))
 *                 .define(Map.of( // Replace strings during build time
 *                         "process.env.API_KEY", JSON.stringify("xxx-xxxx-xxx"),
 *                         "process.env.PRODUCTION", JSON.stringify(true),
 *                         "process.env.NUMBER", JSON.stringify(123)))
 *                 .logLevel(LogLevel.ERROR) // defaults to LogLevel.WARNING
 *                 .keepNames(true) // defaults to false
 *                 .tsconfig("custom-tsconfig.json") // use custom-tsconfig.json instead of default,
 *                 .metafile(true) // include meta file, defaults to false
 *                 .banner("/* comments *&#47;") // requires esbuild &gt;= 0.9.0, defaults to none
 *                 .footer("/* comments *&#47;") // requires esbuild &gt;= 0.9.0, defaults to none
 *                 .charset(Charset.UTF8) // do not escape non-ASCII characters, defaults to Charset.ASCII
 *                 .format(OutputFormat.ESM) // ECMAScript module output format, defaults to OutputFormat.CJS (OutputFormat.ESM requires Node.js &gt;= 14)
 *                 .mainFields(List.of("module", "main")) // prefer ECMAScript versions of dependencies
 *                 .inject(List.of("./my-shim.js", "./other-shim.js")) // allows to automatically replace a global variable with an import from another file
 *                 .esbuildArgs(Map.of( // Pass additional arguments to esbuild
 *                         "--log-limit", "0",
 *                         "--splitting", true))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Command hooks</h2>
 * <p>
 * It is possible to run additional commands by specifying the <code>commandHooks</code> prop:
 * <p>
 * <blockquote><pre>
 * // This example only available in TypeScript
 * // Run additional props via `commandHooks`
 * new nodejs.NodejsFunction(this, 'my-handler-with-commands', {
 *   bundling: {
 *     commandHooks: {
 *       beforeBundling(inputDir: string, outputDir: string): string[] {
 *         return [
 *           `echo hello &gt; ${inputDir}/a.txt`,
 *           `cp ${inputDir}/a.txt ${outputDir}`,
 *         ];
 *       },
 *       afterBundling(inputDir: string, outputDir: string): string[] {
 *         return [`cp ${inputDir}/b.txt ${outputDir}/txt`];
 *       },
 *       beforeInstall() {
 *         return [];
 *       },
 *       // ...
 *     },
 *     // ...
 *   },
 * });
 * </pre></blockquote>
 * <p>
 * The following hooks are available:
 * <p>
 * <ul>
 * <li><code>beforeBundling</code>: runs before all bundling commands</li>
 * <li><code>beforeInstall</code>: runs before node modules installation</li>
 * <li><code>afterBundling</code>: runs after all bundling commands</li>
 * </ul>
 * <p>
 * They all receive the directory containing the lock file (<code>inputDir</code>) and the
 * directory where the bundled asset will be output (<code>outputDir</code>). They must return
 * an array of commands to run. Commands are chained with <code>&amp;&amp;</code>.
 * <p>
 * The commands will run in the environment in which bundling occurs: inside the
 * container for Docker bundling or on the host OS for local bundling.
 * <p>
 * <h2>Pre Compilation with TSC</h2>
 * <p>
 * In some cases, <code>esbuild</code> may not yet support some newer features of the typescript language, such as,
 * <a href="https://www.typescriptlang.org/tsconfig#emitDecoratorMetadata"><code>emitDecoratorMetadata</code></a>.
 * In such cases, it is possible to run pre-compilation using <code>tsc</code> by setting the <code>preCompilation</code> flag.
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .preCompilation(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note: A <a href="https://www.typescriptlang.org/docs/handbook/tsconfig-json.html"><code>tsconfig.json</code> file</a> is required
 * <p>
 * <h2>Customizing Docker bundling</h2>
 * <p>
 * Use <code>bundling.environment</code> to define environments variables when <code>esbuild</code> runs:
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .environment(Map.of(
 *                         "NODE_ENV", "production"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>bundling.buildArgs</code> to pass build arguments when building the Docker bundling image:
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .buildArgs(Map.of(
 *                         "HTTPS_PROXY", "https://127.0.0.1:3001"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>bundling.dockerImage</code> to use a custom Docker bundling image:
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .dockerImage(DockerImage.fromBuild("/path/to/Dockerfile"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * This image should have <code>esbuild</code> installed <strong>globally</strong>. If you plan to use <code>nodeModules</code> it
 * should also have <code>npm</code>, <code>yarn</code> or <code>pnpm</code> depending on the lock file you're using.
 * <p>
 * Use the <a href="https://github.com/aws/aws-cdk/blob/main/packages/aws-cdk-lib/aws-lambda-nodejs/lib/Dockerfile">default image provided by <code>aws-cdk-lib/aws-lambda-nodejs</code></a>
 * as a source of inspiration.
 * <p>
 * You can set additional Docker options to configure the build environment:
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .network("host")
 *                 .securityOpt("no-new-privileges")
 *                 .user("user:group")
 *                 .volumesFrom(List.of("777f7dc92da7"))
 *                 .volumes(List.of(DockerVolume.builder().hostPath("/host-path").containerPath("/container-path").build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Asset hash</h2>
 * <p>
 * By default the asset hash will be calculated based on the bundled output (<code>AssetHashType.OUTPUT</code>).
 * <p>
 * Use the <code>assetHash</code> prop to pass a custom hash:
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .assetHash("my-custom-hash")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you chose to customize the hash, you will need to make sure it is updated every time the asset
 * changes, or otherwise it is possible that some deployments will not be invalidated.
 * <p>
 * <h2>Docker based bundling in complex Docker configurations</h2>
 * <p>
 * By default the input and output of Docker based bundling is handled via bind mounts.
 * In situtations where this does not work, like Docker-in-Docker setups or when using a remote Docker socket, you can configure an alternative, but slower, variant that also works in these situations.
 * <p>
 * <blockquote><pre>
 * NodejsFunction.Builder.create(this, "my-handler")
 *         .bundling(BundlingOptions.builder()
 *                 .bundlingFileAccess(BundlingFileAccess.VOLUME_COPY)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.lambda.nodejs;
