/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Initiates the authentication response.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InitiateAuthResponse extends CognitoIdentityProviderResponse implements
        ToCopyableBuilder<InitiateAuthResponse.Builder, InitiateAuthResponse> {
    private static final SdkField<String> CHALLENGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChallengeName").getter(getter(InitiateAuthResponse::challengeNameAsString))
            .setter(setter(Builder::challengeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChallengeName").build()).build();

    private static final SdkField<String> SESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Session")
            .getter(getter(InitiateAuthResponse::session)).setter(setter(Builder::session))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Session").build()).build();

    private static final SdkField<Map<String, String>> CHALLENGE_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ChallengeParameters")
            .getter(getter(InitiateAuthResponse::challengeParameters))
            .setter(setter(Builder::challengeParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChallengeParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<AuthenticationResultType> AUTHENTICATION_RESULT_FIELD = SdkField
            .<AuthenticationResultType> builder(MarshallingType.SDK_POJO).memberName("AuthenticationResult")
            .getter(getter(InitiateAuthResponse::authenticationResult)).setter(setter(Builder::authenticationResult))
            .constructor(AuthenticationResultType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationResult").build())
            .build();

    private static final SdkField<List<String>> AVAILABLE_CHALLENGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AvailableChallenges")
            .getter(getter(InitiateAuthResponse::availableChallengesAsStrings))
            .setter(setter(Builder::availableChallengesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailableChallenges").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CHALLENGE_NAME_FIELD,
            SESSION_FIELD, CHALLENGE_PARAMETERS_FIELD, AUTHENTICATION_RESULT_FIELD, AVAILABLE_CHALLENGES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String challengeName;

    private final String session;

    private final Map<String, String> challengeParameters;

    private final AuthenticationResultType authenticationResult;

    private final List<String> availableChallenges;

    private InitiateAuthResponse(BuilderImpl builder) {
        super(builder);
        this.challengeName = builder.challengeName;
        this.session = builder.session;
        this.challengeParameters = builder.challengeParameters;
        this.authenticationResult = builder.authenticationResult;
        this.availableChallenges = builder.availableChallenges;
    }

    /**
     * <p>
     * The name of an additional authentication challenge that you must respond to.
     * </p>
     * <p>
     * Possible challenges include the following:
     * </p>
     * <note>
     * <p>
     * All of the following challenges require <code>USERNAME</code> and, when the app client has a client secret,
     * <code>SECRET_HASH</code> in the parameters.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <code>WEB_AUTHN</code>: Respond to the challenge with the results of a successful authentication with a WebAuthn
     * authenticator, or passkey. Examples of WebAuthn authenticators include biometric devices and security keys.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PASSWORD</code>: Respond with <code>USER_PASSWORD_AUTH</code> parameters: <code>USERNAME</code> (required),
     * <code>PASSWORD</code> (required), <code>SECRET_HASH</code> (required if the app client is configured with a
     * client secret), <code>DEVICE_KEY</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PASSWORD_SRP</code>: Respond with <code>USER_SRP_AUTH</code> parameters: <code>USERNAME</code> (required),
     * <code>SRP_A</code> (required), <code>SECRET_HASH</code> (required if the app client is configured with a client
     * secret), <code>DEVICE_KEY</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SELECT_CHALLENGE</code>: Respond to the challenge with <code>USERNAME</code> and an <code>ANSWER</code>
     * that matches one of the challenge types in the <code>AvailableChallenges</code> response parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMS_MFA</code>: Respond with an <code>SMS_MFA_CODE</code> that your user pool delivered in an SMS message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EMAIL_OTP</code>: Respond with an <code>EMAIL_OTP_CODE</code> that your user pool delivered in an email
     * message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
     * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOM_CHALLENGE</code>: This is returned if your custom authentication flow determines that the user
     * should pass another challenge before tokens are issued. The parameters of the challenge are determined by your
     * Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEVICE_SRP_AUTH</code>: Respond with the initial parameters of device SRP authentication. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
     * >Signing in with a device</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEVICE_PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
     * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
     * >Signing in with a device</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_PASSWORD_REQUIRED</code>: For users who are required to change their passwords after successful first
     * login. Respond to this challenge with <code>NEW_PASSWORD</code> and any required attributes that Amazon Cognito
     * returned in the <code>requiredAttributes</code> parameter. You can also set values for attributes that aren't
     * required by your user pool and that your app client can write.
     * </p>
     * <p>
     * Amazon Cognito only returns this challenge for users who have temporary passwords. When you create passwordless
     * users, you must provide values for all required attributes.
     * </p>
     * <note>
     * <p>
     * In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that already
     * has a value. In <code>AdminRespondToAuthChallenge</code> or <code>RespondToAuthChallenge</code>, set a value for
     * any keys that Amazon Cognito returned in the <code>requiredAttributes</code> parameter, then use the
     * <code>AdminUpdateUserAttributes</code> or <code>UpdateUserAttributes</code> API operation to modify the value of
     * any additional attributes.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>MFA_SETUP</code>: For users who are required to setup an MFA factor before they can sign in. The MFA types
     * activated for the user pool will be listed in the challenge parameters <code>MFAS_CAN_SETUP</code> value.
     * </p>
     * <p>
     * To set up time-based one-time password (TOTP) MFA, use the session returned in this challenge from
     * <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> as an input to <code>AssociateSoftwareToken</code>.
     * Then, use the session returned by <code>VerifySoftwareToken</code> as an input to
     * <code>RespondToAuthChallenge</code> or <code>AdminRespondToAuthChallenge</code> with challenge name
     * <code>MFA_SETUP</code> to complete sign-in.
     * </p>
     * <p>
     * To set up SMS or email MFA, collect a <code>phone_number</code> or <code>email</code> attribute for the user.
     * Then restart the authentication flow with an <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #challengeName}
     * will return {@link ChallengeNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #challengeNameAsString}.
     * </p>
     * 
     * @return The name of an additional authentication challenge that you must respond to.</p>
     *         <p>
     *         Possible challenges include the following:
     *         </p>
     *         <note>
     *         <p>
     *         All of the following challenges require <code>USERNAME</code> and, when the app client has a client
     *         secret, <code>SECRET_HASH</code> in the parameters.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>WEB_AUTHN</code>: Respond to the challenge with the results of a successful authentication with a
     *         WebAuthn authenticator, or passkey. Examples of WebAuthn authenticators include biometric devices and
     *         security keys.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PASSWORD</code>: Respond with <code>USER_PASSWORD_AUTH</code> parameters: <code>USERNAME</code>
     *         (required), <code>PASSWORD</code> (required), <code>SECRET_HASH</code> (required if the app client is
     *         configured with a client secret), <code>DEVICE_KEY</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PASSWORD_SRP</code>: Respond with <code>USER_SRP_AUTH</code> parameters: <code>USERNAME</code>
     *         (required), <code>SRP_A</code> (required), <code>SECRET_HASH</code> (required if the app client is
     *         configured with a client secret), <code>DEVICE_KEY</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SELECT_CHALLENGE</code>: Respond to the challenge with <code>USERNAME</code> and an
     *         <code>ANSWER</code> that matches one of the challenge types in the <code>AvailableChallenges</code>
     *         response parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMS_MFA</code>: Respond with an <code>SMS_MFA_CODE</code> that your user pool delivered in an SMS
     *         message.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EMAIL_OTP</code>: Respond with an <code>EMAIL_OTP_CODE</code> that your user pool delivered in an
     *         email message.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
     *         <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOM_CHALLENGE</code>: This is returned if your custom authentication flow determines that the
     *         user should pass another challenge before tokens are issued. The parameters of the challenge are
     *         determined by your Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEVICE_SRP_AUTH</code>: Respond with the initial parameters of device SRP authentication. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
     *         >Signing in with a device</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEVICE_PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
     *         <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations.
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
     *         >Signing in with a device</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_PASSWORD_REQUIRED</code>: For users who are required to change their passwords after successful
     *         first login. Respond to this challenge with <code>NEW_PASSWORD</code> and any required attributes that
     *         Amazon Cognito returned in the <code>requiredAttributes</code> parameter. You can also set values for
     *         attributes that aren't required by your user pool and that your app client can write.
     *         </p>
     *         <p>
     *         Amazon Cognito only returns this challenge for users who have temporary passwords. When you create
     *         passwordless users, you must provide values for all required attributes.
     *         </p>
     *         <note>
     *         <p>
     *         In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that
     *         already has a value. In <code>AdminRespondToAuthChallenge</code> or <code>RespondToAuthChallenge</code>,
     *         set a value for any keys that Amazon Cognito returned in the <code>requiredAttributes</code> parameter,
     *         then use the <code>AdminUpdateUserAttributes</code> or <code>UpdateUserAttributes</code> API operation to
     *         modify the value of any additional attributes.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>MFA_SETUP</code>: For users who are required to setup an MFA factor before they can sign in. The
     *         MFA types activated for the user pool will be listed in the challenge parameters
     *         <code>MFAS_CAN_SETUP</code> value.
     *         </p>
     *         <p>
     *         To set up time-based one-time password (TOTP) MFA, use the session returned in this challenge from
     *         <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> as an input to
     *         <code>AssociateSoftwareToken</code>. Then, use the session returned by <code>VerifySoftwareToken</code>
     *         as an input to <code>RespondToAuthChallenge</code> or <code>AdminRespondToAuthChallenge</code> with
     *         challenge name <code>MFA_SETUP</code> to complete sign-in.
     *         </p>
     *         <p>
     *         To set up SMS or email MFA, collect a <code>phone_number</code> or <code>email</code> attribute for the
     *         user. Then restart the authentication flow with an <code>InitiateAuth</code> or
     *         <code>AdminInitiateAuth</code> request.
     *         </p>
     *         </li>
     * @see ChallengeNameType
     */
    public final ChallengeNameType challengeName() {
        return ChallengeNameType.fromValue(challengeName);
    }

    /**
     * <p>
     * The name of an additional authentication challenge that you must respond to.
     * </p>
     * <p>
     * Possible challenges include the following:
     * </p>
     * <note>
     * <p>
     * All of the following challenges require <code>USERNAME</code> and, when the app client has a client secret,
     * <code>SECRET_HASH</code> in the parameters.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * <code>WEB_AUTHN</code>: Respond to the challenge with the results of a successful authentication with a WebAuthn
     * authenticator, or passkey. Examples of WebAuthn authenticators include biometric devices and security keys.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PASSWORD</code>: Respond with <code>USER_PASSWORD_AUTH</code> parameters: <code>USERNAME</code> (required),
     * <code>PASSWORD</code> (required), <code>SECRET_HASH</code> (required if the app client is configured with a
     * client secret), <code>DEVICE_KEY</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PASSWORD_SRP</code>: Respond with <code>USER_SRP_AUTH</code> parameters: <code>USERNAME</code> (required),
     * <code>SRP_A</code> (required), <code>SECRET_HASH</code> (required if the app client is configured with a client
     * secret), <code>DEVICE_KEY</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SELECT_CHALLENGE</code>: Respond to the challenge with <code>USERNAME</code> and an <code>ANSWER</code>
     * that matches one of the challenge types in the <code>AvailableChallenges</code> response parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SMS_MFA</code>: Respond with an <code>SMS_MFA_CODE</code> that your user pool delivered in an SMS message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EMAIL_OTP</code>: Respond with an <code>EMAIL_OTP_CODE</code> that your user pool delivered in an email
     * message.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
     * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOM_CHALLENGE</code>: This is returned if your custom authentication flow determines that the user
     * should pass another challenge before tokens are issued. The parameters of the challenge are determined by your
     * Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEVICE_SRP_AUTH</code>: Respond with the initial parameters of device SRP authentication. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
     * >Signing in with a device</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEVICE_PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
     * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
     * >Signing in with a device</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_PASSWORD_REQUIRED</code>: For users who are required to change their passwords after successful first
     * login. Respond to this challenge with <code>NEW_PASSWORD</code> and any required attributes that Amazon Cognito
     * returned in the <code>requiredAttributes</code> parameter. You can also set values for attributes that aren't
     * required by your user pool and that your app client can write.
     * </p>
     * <p>
     * Amazon Cognito only returns this challenge for users who have temporary passwords. When you create passwordless
     * users, you must provide values for all required attributes.
     * </p>
     * <note>
     * <p>
     * In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that already
     * has a value. In <code>AdminRespondToAuthChallenge</code> or <code>RespondToAuthChallenge</code>, set a value for
     * any keys that Amazon Cognito returned in the <code>requiredAttributes</code> parameter, then use the
     * <code>AdminUpdateUserAttributes</code> or <code>UpdateUserAttributes</code> API operation to modify the value of
     * any additional attributes.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>MFA_SETUP</code>: For users who are required to setup an MFA factor before they can sign in. The MFA types
     * activated for the user pool will be listed in the challenge parameters <code>MFAS_CAN_SETUP</code> value.
     * </p>
     * <p>
     * To set up time-based one-time password (TOTP) MFA, use the session returned in this challenge from
     * <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> as an input to <code>AssociateSoftwareToken</code>.
     * Then, use the session returned by <code>VerifySoftwareToken</code> as an input to
     * <code>RespondToAuthChallenge</code> or <code>AdminRespondToAuthChallenge</code> with challenge name
     * <code>MFA_SETUP</code> to complete sign-in.
     * </p>
     * <p>
     * To set up SMS or email MFA, collect a <code>phone_number</code> or <code>email</code> attribute for the user.
     * Then restart the authentication flow with an <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #challengeName}
     * will return {@link ChallengeNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #challengeNameAsString}.
     * </p>
     * 
     * @return The name of an additional authentication challenge that you must respond to.</p>
     *         <p>
     *         Possible challenges include the following:
     *         </p>
     *         <note>
     *         <p>
     *         All of the following challenges require <code>USERNAME</code> and, when the app client has a client
     *         secret, <code>SECRET_HASH</code> in the parameters.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>WEB_AUTHN</code>: Respond to the challenge with the results of a successful authentication with a
     *         WebAuthn authenticator, or passkey. Examples of WebAuthn authenticators include biometric devices and
     *         security keys.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PASSWORD</code>: Respond with <code>USER_PASSWORD_AUTH</code> parameters: <code>USERNAME</code>
     *         (required), <code>PASSWORD</code> (required), <code>SECRET_HASH</code> (required if the app client is
     *         configured with a client secret), <code>DEVICE_KEY</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PASSWORD_SRP</code>: Respond with <code>USER_SRP_AUTH</code> parameters: <code>USERNAME</code>
     *         (required), <code>SRP_A</code> (required), <code>SECRET_HASH</code> (required if the app client is
     *         configured with a client secret), <code>DEVICE_KEY</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SELECT_CHALLENGE</code>: Respond to the challenge with <code>USERNAME</code> and an
     *         <code>ANSWER</code> that matches one of the challenge types in the <code>AvailableChallenges</code>
     *         response parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SMS_MFA</code>: Respond with an <code>SMS_MFA_CODE</code> that your user pool delivered in an SMS
     *         message.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EMAIL_OTP</code>: Respond with an <code>EMAIL_OTP_CODE</code> that your user pool delivered in an
     *         email message.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
     *         <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOM_CHALLENGE</code>: This is returned if your custom authentication flow determines that the
     *         user should pass another challenge before tokens are issued. The parameters of the challenge are
     *         determined by your Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEVICE_SRP_AUTH</code>: Respond with the initial parameters of device SRP authentication. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
     *         >Signing in with a device</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEVICE_PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
     *         <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations.
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
     *         >Signing in with a device</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_PASSWORD_REQUIRED</code>: For users who are required to change their passwords after successful
     *         first login. Respond to this challenge with <code>NEW_PASSWORD</code> and any required attributes that
     *         Amazon Cognito returned in the <code>requiredAttributes</code> parameter. You can also set values for
     *         attributes that aren't required by your user pool and that your app client can write.
     *         </p>
     *         <p>
     *         Amazon Cognito only returns this challenge for users who have temporary passwords. When you create
     *         passwordless users, you must provide values for all required attributes.
     *         </p>
     *         <note>
     *         <p>
     *         In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that
     *         already has a value. In <code>AdminRespondToAuthChallenge</code> or <code>RespondToAuthChallenge</code>,
     *         set a value for any keys that Amazon Cognito returned in the <code>requiredAttributes</code> parameter,
     *         then use the <code>AdminUpdateUserAttributes</code> or <code>UpdateUserAttributes</code> API operation to
     *         modify the value of any additional attributes.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>MFA_SETUP</code>: For users who are required to setup an MFA factor before they can sign in. The
     *         MFA types activated for the user pool will be listed in the challenge parameters
     *         <code>MFAS_CAN_SETUP</code> value.
     *         </p>
     *         <p>
     *         To set up time-based one-time password (TOTP) MFA, use the session returned in this challenge from
     *         <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> as an input to
     *         <code>AssociateSoftwareToken</code>. Then, use the session returned by <code>VerifySoftwareToken</code>
     *         as an input to <code>RespondToAuthChallenge</code> or <code>AdminRespondToAuthChallenge</code> with
     *         challenge name <code>MFA_SETUP</code> to complete sign-in.
     *         </p>
     *         <p>
     *         To set up SMS or email MFA, collect a <code>phone_number</code> or <code>email</code> attribute for the
     *         user. Then restart the authentication flow with an <code>InitiateAuth</code> or
     *         <code>AdminInitiateAuth</code> request.
     *         </p>
     *         </li>
     * @see ChallengeNameType
     */
    public final String challengeNameAsString() {
        return challengeName;
    }

    /**
     * <p>
     * The session identifier that links a challenge response to the initial authentication request. If the user must
     * pass another challenge, Amazon Cognito returns a session ID and challenge parameters.
     * </p>
     * 
     * @return The session identifier that links a challenge response to the initial authentication request. If the user
     *         must pass another challenge, Amazon Cognito returns a session ID and challenge parameters.
     */
    public final String session() {
        return session;
    }

    /**
     * For responses, this returns true if the service returned a value for the ChallengeParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasChallengeParameters() {
        return challengeParameters != null && !(challengeParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The required parameters of the <code>ChallengeName</code> challenge.
     * </p>
     * <p>
     * All challenges require <code>USERNAME</code>. They also require <code>SECRET_HASH</code> if your app client has a
     * client secret.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChallengeParameters} method.
     * </p>
     * 
     * @return The required parameters of the <code>ChallengeName</code> challenge.</p>
     *         <p>
     *         All challenges require <code>USERNAME</code>. They also require <code>SECRET_HASH</code> if your app
     *         client has a client secret.
     */
    public final Map<String, String> challengeParameters() {
        return challengeParameters;
    }

    /**
     * <p>
     * The result of a successful and complete authentication request. This result is only returned if the user doesn't
     * need to pass another challenge. If they must pass another challenge before they get tokens, Amazon Cognito
     * returns a challenge in <code>ChallengeName</code>, <code>ChallengeParameters</code>, and <code>Session</code>
     * response parameters.
     * </p>
     * 
     * @return The result of a successful and complete authentication request. This result is only returned if the user
     *         doesn't need to pass another challenge. If they must pass another challenge before they get tokens,
     *         Amazon Cognito returns a challenge in <code>ChallengeName</code>, <code>ChallengeParameters</code>, and
     *         <code>Session</code> response parameters.
     */
    public final AuthenticationResultType authenticationResult() {
        return authenticationResult;
    }

    /**
     * <p>
     * This response parameter lists the available authentication challenges that users can select from in <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
     * >choice-based authentication</a>. For example, they might be able to choose between passkey authentication, a
     * one-time password from an SMS message, and a traditional password.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailableChallenges} method.
     * </p>
     * 
     * @return This response parameter lists the available authentication challenges that users can select from in <a
     *         href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
     *         >choice-based authentication</a>. For example, they might be able to choose between passkey
     *         authentication, a one-time password from an SMS message, and a traditional password.
     */
    public final List<ChallengeNameType> availableChallenges() {
        return AvailableChallengeListTypeCopier.copyStringToEnum(availableChallenges);
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailableChallenges property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailableChallenges() {
        return availableChallenges != null && !(availableChallenges instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * This response parameter lists the available authentication challenges that users can select from in <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
     * >choice-based authentication</a>. For example, they might be able to choose between passkey authentication, a
     * one-time password from an SMS message, and a traditional password.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailableChallenges} method.
     * </p>
     * 
     * @return This response parameter lists the available authentication challenges that users can select from in <a
     *         href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
     *         >choice-based authentication</a>. For example, they might be able to choose between passkey
     *         authentication, a one-time password from an SMS message, and a traditional password.
     */
    public final List<String> availableChallengesAsStrings() {
        return availableChallenges;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(challengeNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(session());
        hashCode = 31 * hashCode + Objects.hashCode(hasChallengeParameters() ? challengeParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(authenticationResult());
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailableChallenges() ? availableChallengesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InitiateAuthResponse)) {
            return false;
        }
        InitiateAuthResponse other = (InitiateAuthResponse) obj;
        return Objects.equals(challengeNameAsString(), other.challengeNameAsString())
                && Objects.equals(session(), other.session()) && hasChallengeParameters() == other.hasChallengeParameters()
                && Objects.equals(challengeParameters(), other.challengeParameters())
                && Objects.equals(authenticationResult(), other.authenticationResult())
                && hasAvailableChallenges() == other.hasAvailableChallenges()
                && Objects.equals(availableChallengesAsStrings(), other.availableChallengesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InitiateAuthResponse").add("ChallengeName", challengeNameAsString())
                .add("Session", session() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ChallengeParameters", hasChallengeParameters() ? challengeParameters() : null)
                .add("AuthenticationResult", authenticationResult())
                .add("AvailableChallenges", hasAvailableChallenges() ? availableChallengesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ChallengeName":
            return Optional.ofNullable(clazz.cast(challengeNameAsString()));
        case "Session":
            return Optional.ofNullable(clazz.cast(session()));
        case "ChallengeParameters":
            return Optional.ofNullable(clazz.cast(challengeParameters()));
        case "AuthenticationResult":
            return Optional.ofNullable(clazz.cast(authenticationResult()));
        case "AvailableChallenges":
            return Optional.ofNullable(clazz.cast(availableChallengesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ChallengeName", CHALLENGE_NAME_FIELD);
        map.put("Session", SESSION_FIELD);
        map.put("ChallengeParameters", CHALLENGE_PARAMETERS_FIELD);
        map.put("AuthenticationResult", AUTHENTICATION_RESULT_FIELD);
        map.put("AvailableChallenges", AVAILABLE_CHALLENGES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InitiateAuthResponse, T> g) {
        return obj -> g.apply((InitiateAuthResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderResponse.Builder, SdkPojo,
            CopyableBuilder<Builder, InitiateAuthResponse> {
        /**
         * <p>
         * The name of an additional authentication challenge that you must respond to.
         * </p>
         * <p>
         * Possible challenges include the following:
         * </p>
         * <note>
         * <p>
         * All of the following challenges require <code>USERNAME</code> and, when the app client has a client secret,
         * <code>SECRET_HASH</code> in the parameters.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <code>WEB_AUTHN</code>: Respond to the challenge with the results of a successful authentication with a
         * WebAuthn authenticator, or passkey. Examples of WebAuthn authenticators include biometric devices and
         * security keys.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PASSWORD</code>: Respond with <code>USER_PASSWORD_AUTH</code> parameters: <code>USERNAME</code>
         * (required), <code>PASSWORD</code> (required), <code>SECRET_HASH</code> (required if the app client is
         * configured with a client secret), <code>DEVICE_KEY</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PASSWORD_SRP</code>: Respond with <code>USER_SRP_AUTH</code> parameters: <code>USERNAME</code>
         * (required), <code>SRP_A</code> (required), <code>SECRET_HASH</code> (required if the app client is configured
         * with a client secret), <code>DEVICE_KEY</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SELECT_CHALLENGE</code>: Respond to the challenge with <code>USERNAME</code> and an <code>ANSWER</code>
         * that matches one of the challenge types in the <code>AvailableChallenges</code> response parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMS_MFA</code>: Respond with an <code>SMS_MFA_CODE</code> that your user pool delivered in an SMS
         * message.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EMAIL_OTP</code>: Respond with an <code>EMAIL_OTP_CODE</code> that your user pool delivered in an email
         * message.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
         * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOM_CHALLENGE</code>: This is returned if your custom authentication flow determines that the user
         * should pass another challenge before tokens are issued. The parameters of the challenge are determined by
         * your Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEVICE_SRP_AUTH</code>: Respond with the initial parameters of device SRP authentication. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
         * >Signing in with a device</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEVICE_PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
         * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
         * >Signing in with a device</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_PASSWORD_REQUIRED</code>: For users who are required to change their passwords after successful
         * first login. Respond to this challenge with <code>NEW_PASSWORD</code> and any required attributes that Amazon
         * Cognito returned in the <code>requiredAttributes</code> parameter. You can also set values for attributes
         * that aren't required by your user pool and that your app client can write.
         * </p>
         * <p>
         * Amazon Cognito only returns this challenge for users who have temporary passwords. When you create
         * passwordless users, you must provide values for all required attributes.
         * </p>
         * <note>
         * <p>
         * In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that
         * already has a value. In <code>AdminRespondToAuthChallenge</code> or <code>RespondToAuthChallenge</code>, set
         * a value for any keys that Amazon Cognito returned in the <code>requiredAttributes</code> parameter, then use
         * the <code>AdminUpdateUserAttributes</code> or <code>UpdateUserAttributes</code> API operation to modify the
         * value of any additional attributes.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>MFA_SETUP</code>: For users who are required to setup an MFA factor before they can sign in. The MFA
         * types activated for the user pool will be listed in the challenge parameters <code>MFAS_CAN_SETUP</code>
         * value.
         * </p>
         * <p>
         * To set up time-based one-time password (TOTP) MFA, use the session returned in this challenge from
         * <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> as an input to
         * <code>AssociateSoftwareToken</code>. Then, use the session returned by <code>VerifySoftwareToken</code> as an
         * input to <code>RespondToAuthChallenge</code> or <code>AdminRespondToAuthChallenge</code> with challenge name
         * <code>MFA_SETUP</code> to complete sign-in.
         * </p>
         * <p>
         * To set up SMS or email MFA, collect a <code>phone_number</code> or <code>email</code> attribute for the user.
         * Then restart the authentication flow with an <code>InitiateAuth</code> or <code>AdminInitiateAuth</code>
         * request.
         * </p>
         * </li>
         * </ul>
         * 
         * @param challengeName
         *        The name of an additional authentication challenge that you must respond to.</p>
         *        <p>
         *        Possible challenges include the following:
         *        </p>
         *        <note>
         *        <p>
         *        All of the following challenges require <code>USERNAME</code> and, when the app client has a client
         *        secret, <code>SECRET_HASH</code> in the parameters.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>WEB_AUTHN</code>: Respond to the challenge with the results of a successful authentication with
         *        a WebAuthn authenticator, or passkey. Examples of WebAuthn authenticators include biometric devices
         *        and security keys.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PASSWORD</code>: Respond with <code>USER_PASSWORD_AUTH</code> parameters: <code>USERNAME</code>
         *        (required), <code>PASSWORD</code> (required), <code>SECRET_HASH</code> (required if the app client is
         *        configured with a client secret), <code>DEVICE_KEY</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PASSWORD_SRP</code>: Respond with <code>USER_SRP_AUTH</code> parameters: <code>USERNAME</code>
         *        (required), <code>SRP_A</code> (required), <code>SECRET_HASH</code> (required if the app client is
         *        configured with a client secret), <code>DEVICE_KEY</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SELECT_CHALLENGE</code>: Respond to the challenge with <code>USERNAME</code> and an
         *        <code>ANSWER</code> that matches one of the challenge types in the <code>AvailableChallenges</code>
         *        response parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMS_MFA</code>: Respond with an <code>SMS_MFA_CODE</code> that your user pool delivered in an
         *        SMS message.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EMAIL_OTP</code>: Respond with an <code>EMAIL_OTP_CODE</code> that your user pool delivered in
         *        an email message.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
         *        <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP
         *        calculations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOM_CHALLENGE</code>: This is returned if your custom authentication flow determines that the
         *        user should pass another challenge before tokens are issued. The parameters of the challenge are
         *        determined by your Lambda function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEVICE_SRP_AUTH</code>: Respond with the initial parameters of device SRP authentication. For
         *        more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
         *        >Signing in with a device</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEVICE_PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
         *        <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP
         *        calculations. For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
         *        >Signing in with a device</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_PASSWORD_REQUIRED</code>: For users who are required to change their passwords after
         *        successful first login. Respond to this challenge with <code>NEW_PASSWORD</code> and any required
         *        attributes that Amazon Cognito returned in the <code>requiredAttributes</code> parameter. You can also
         *        set values for attributes that aren't required by your user pool and that your app client can write.
         *        </p>
         *        <p>
         *        Amazon Cognito only returns this challenge for users who have temporary passwords. When you create
         *        passwordless users, you must provide values for all required attributes.
         *        </p>
         *        <note>
         *        <p>
         *        In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that
         *        already has a value. In <code>AdminRespondToAuthChallenge</code> or
         *        <code>RespondToAuthChallenge</code>, set a value for any keys that Amazon Cognito returned in the
         *        <code>requiredAttributes</code> parameter, then use the <code>AdminUpdateUserAttributes</code> or
         *        <code>UpdateUserAttributes</code> API operation to modify the value of any additional attributes.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>MFA_SETUP</code>: For users who are required to setup an MFA factor before they can sign in. The
         *        MFA types activated for the user pool will be listed in the challenge parameters
         *        <code>MFAS_CAN_SETUP</code> value.
         *        </p>
         *        <p>
         *        To set up time-based one-time password (TOTP) MFA, use the session returned in this challenge from
         *        <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> as an input to
         *        <code>AssociateSoftwareToken</code>. Then, use the session returned by
         *        <code>VerifySoftwareToken</code> as an input to <code>RespondToAuthChallenge</code> or
         *        <code>AdminRespondToAuthChallenge</code> with challenge name <code>MFA_SETUP</code> to complete
         *        sign-in.
         *        </p>
         *        <p>
         *        To set up SMS or email MFA, collect a <code>phone_number</code> or <code>email</code> attribute for
         *        the user. Then restart the authentication flow with an <code>InitiateAuth</code> or
         *        <code>AdminInitiateAuth</code> request.
         *        </p>
         *        </li>
         * @see ChallengeNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChallengeNameType
         */
        Builder challengeName(String challengeName);

        /**
         * <p>
         * The name of an additional authentication challenge that you must respond to.
         * </p>
         * <p>
         * Possible challenges include the following:
         * </p>
         * <note>
         * <p>
         * All of the following challenges require <code>USERNAME</code> and, when the app client has a client secret,
         * <code>SECRET_HASH</code> in the parameters.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * <code>WEB_AUTHN</code>: Respond to the challenge with the results of a successful authentication with a
         * WebAuthn authenticator, or passkey. Examples of WebAuthn authenticators include biometric devices and
         * security keys.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PASSWORD</code>: Respond with <code>USER_PASSWORD_AUTH</code> parameters: <code>USERNAME</code>
         * (required), <code>PASSWORD</code> (required), <code>SECRET_HASH</code> (required if the app client is
         * configured with a client secret), <code>DEVICE_KEY</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PASSWORD_SRP</code>: Respond with <code>USER_SRP_AUTH</code> parameters: <code>USERNAME</code>
         * (required), <code>SRP_A</code> (required), <code>SECRET_HASH</code> (required if the app client is configured
         * with a client secret), <code>DEVICE_KEY</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SELECT_CHALLENGE</code>: Respond to the challenge with <code>USERNAME</code> and an <code>ANSWER</code>
         * that matches one of the challenge types in the <code>AvailableChallenges</code> response parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SMS_MFA</code>: Respond with an <code>SMS_MFA_CODE</code> that your user pool delivered in an SMS
         * message.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EMAIL_OTP</code>: Respond with an <code>EMAIL_OTP_CODE</code> that your user pool delivered in an email
         * message.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
         * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOM_CHALLENGE</code>: This is returned if your custom authentication flow determines that the user
         * should pass another challenge before tokens are issued. The parameters of the challenge are determined by
         * your Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEVICE_SRP_AUTH</code>: Respond with the initial parameters of device SRP authentication. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
         * >Signing in with a device</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEVICE_PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
         * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP calculations. For
         * more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
         * >Signing in with a device</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_PASSWORD_REQUIRED</code>: For users who are required to change their passwords after successful
         * first login. Respond to this challenge with <code>NEW_PASSWORD</code> and any required attributes that Amazon
         * Cognito returned in the <code>requiredAttributes</code> parameter. You can also set values for attributes
         * that aren't required by your user pool and that your app client can write.
         * </p>
         * <p>
         * Amazon Cognito only returns this challenge for users who have temporary passwords. When you create
         * passwordless users, you must provide values for all required attributes.
         * </p>
         * <note>
         * <p>
         * In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that
         * already has a value. In <code>AdminRespondToAuthChallenge</code> or <code>RespondToAuthChallenge</code>, set
         * a value for any keys that Amazon Cognito returned in the <code>requiredAttributes</code> parameter, then use
         * the <code>AdminUpdateUserAttributes</code> or <code>UpdateUserAttributes</code> API operation to modify the
         * value of any additional attributes.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>MFA_SETUP</code>: For users who are required to setup an MFA factor before they can sign in. The MFA
         * types activated for the user pool will be listed in the challenge parameters <code>MFAS_CAN_SETUP</code>
         * value.
         * </p>
         * <p>
         * To set up time-based one-time password (TOTP) MFA, use the session returned in this challenge from
         * <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> as an input to
         * <code>AssociateSoftwareToken</code>. Then, use the session returned by <code>VerifySoftwareToken</code> as an
         * input to <code>RespondToAuthChallenge</code> or <code>AdminRespondToAuthChallenge</code> with challenge name
         * <code>MFA_SETUP</code> to complete sign-in.
         * </p>
         * <p>
         * To set up SMS or email MFA, collect a <code>phone_number</code> or <code>email</code> attribute for the user.
         * Then restart the authentication flow with an <code>InitiateAuth</code> or <code>AdminInitiateAuth</code>
         * request.
         * </p>
         * </li>
         * </ul>
         * 
         * @param challengeName
         *        The name of an additional authentication challenge that you must respond to.</p>
         *        <p>
         *        Possible challenges include the following:
         *        </p>
         *        <note>
         *        <p>
         *        All of the following challenges require <code>USERNAME</code> and, when the app client has a client
         *        secret, <code>SECRET_HASH</code> in the parameters.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>WEB_AUTHN</code>: Respond to the challenge with the results of a successful authentication with
         *        a WebAuthn authenticator, or passkey. Examples of WebAuthn authenticators include biometric devices
         *        and security keys.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PASSWORD</code>: Respond with <code>USER_PASSWORD_AUTH</code> parameters: <code>USERNAME</code>
         *        (required), <code>PASSWORD</code> (required), <code>SECRET_HASH</code> (required if the app client is
         *        configured with a client secret), <code>DEVICE_KEY</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PASSWORD_SRP</code>: Respond with <code>USER_SRP_AUTH</code> parameters: <code>USERNAME</code>
         *        (required), <code>SRP_A</code> (required), <code>SECRET_HASH</code> (required if the app client is
         *        configured with a client secret), <code>DEVICE_KEY</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SELECT_CHALLENGE</code>: Respond to the challenge with <code>USERNAME</code> and an
         *        <code>ANSWER</code> that matches one of the challenge types in the <code>AvailableChallenges</code>
         *        response parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SMS_MFA</code>: Respond with an <code>SMS_MFA_CODE</code> that your user pool delivered in an
         *        SMS message.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EMAIL_OTP</code>: Respond with an <code>EMAIL_OTP_CODE</code> that your user pool delivered in
         *        an email message.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
         *        <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP
         *        calculations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOM_CHALLENGE</code>: This is returned if your custom authentication flow determines that the
         *        user should pass another challenge before tokens are issued. The parameters of the challenge are
         *        determined by your Lambda function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEVICE_SRP_AUTH</code>: Respond with the initial parameters of device SRP authentication. For
         *        more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
         *        >Signing in with a device</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEVICE_PASSWORD_VERIFIER</code>: Respond with <code>PASSWORD_CLAIM_SIGNATURE</code>,
         *        <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, and <code>TIMESTAMP</code> after client-side SRP
         *        calculations. For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html#user-pools-remembered-devices-signing-in-with-a-device"
         *        >Signing in with a device</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_PASSWORD_REQUIRED</code>: For users who are required to change their passwords after
         *        successful first login. Respond to this challenge with <code>NEW_PASSWORD</code> and any required
         *        attributes that Amazon Cognito returned in the <code>requiredAttributes</code> parameter. You can also
         *        set values for attributes that aren't required by your user pool and that your app client can write.
         *        </p>
         *        <p>
         *        Amazon Cognito only returns this challenge for users who have temporary passwords. When you create
         *        passwordless users, you must provide values for all required attributes.
         *        </p>
         *        <note>
         *        <p>
         *        In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that
         *        already has a value. In <code>AdminRespondToAuthChallenge</code> or
         *        <code>RespondToAuthChallenge</code>, set a value for any keys that Amazon Cognito returned in the
         *        <code>requiredAttributes</code> parameter, then use the <code>AdminUpdateUserAttributes</code> or
         *        <code>UpdateUserAttributes</code> API operation to modify the value of any additional attributes.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>MFA_SETUP</code>: For users who are required to setup an MFA factor before they can sign in. The
         *        MFA types activated for the user pool will be listed in the challenge parameters
         *        <code>MFAS_CAN_SETUP</code> value.
         *        </p>
         *        <p>
         *        To set up time-based one-time password (TOTP) MFA, use the session returned in this challenge from
         *        <code>InitiateAuth</code> or <code>AdminInitiateAuth</code> as an input to
         *        <code>AssociateSoftwareToken</code>. Then, use the session returned by
         *        <code>VerifySoftwareToken</code> as an input to <code>RespondToAuthChallenge</code> or
         *        <code>AdminRespondToAuthChallenge</code> with challenge name <code>MFA_SETUP</code> to complete
         *        sign-in.
         *        </p>
         *        <p>
         *        To set up SMS or email MFA, collect a <code>phone_number</code> or <code>email</code> attribute for
         *        the user. Then restart the authentication flow with an <code>InitiateAuth</code> or
         *        <code>AdminInitiateAuth</code> request.
         *        </p>
         *        </li>
         * @see ChallengeNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChallengeNameType
         */
        Builder challengeName(ChallengeNameType challengeName);

        /**
         * <p>
         * The session identifier that links a challenge response to the initial authentication request. If the user
         * must pass another challenge, Amazon Cognito returns a session ID and challenge parameters.
         * </p>
         * 
         * @param session
         *        The session identifier that links a challenge response to the initial authentication request. If the
         *        user must pass another challenge, Amazon Cognito returns a session ID and challenge parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder session(String session);

        /**
         * <p>
         * The required parameters of the <code>ChallengeName</code> challenge.
         * </p>
         * <p>
         * All challenges require <code>USERNAME</code>. They also require <code>SECRET_HASH</code> if your app client
         * has a client secret.
         * </p>
         * 
         * @param challengeParameters
         *        The required parameters of the <code>ChallengeName</code> challenge.</p>
         *        <p>
         *        All challenges require <code>USERNAME</code>. They also require <code>SECRET_HASH</code> if your app
         *        client has a client secret.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder challengeParameters(Map<String, String> challengeParameters);

        /**
         * <p>
         * The result of a successful and complete authentication request. This result is only returned if the user
         * doesn't need to pass another challenge. If they must pass another challenge before they get tokens, Amazon
         * Cognito returns a challenge in <code>ChallengeName</code>, <code>ChallengeParameters</code>, and
         * <code>Session</code> response parameters.
         * </p>
         * 
         * @param authenticationResult
         *        The result of a successful and complete authentication request. This result is only returned if the
         *        user doesn't need to pass another challenge. If they must pass another challenge before they get
         *        tokens, Amazon Cognito returns a challenge in <code>ChallengeName</code>,
         *        <code>ChallengeParameters</code>, and <code>Session</code> response parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationResult(AuthenticationResultType authenticationResult);

        /**
         * <p>
         * The result of a successful and complete authentication request. This result is only returned if the user
         * doesn't need to pass another challenge. If they must pass another challenge before they get tokens, Amazon
         * Cognito returns a challenge in <code>ChallengeName</code>, <code>ChallengeParameters</code>, and
         * <code>Session</code> response parameters.
         * </p>
         * This is a convenience method that creates an instance of the {@link AuthenticationResultType.Builder}
         * avoiding the need to create one manually via {@link AuthenticationResultType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AuthenticationResultType.Builder#build()} is called immediately
         * and its result is passed to {@link #authenticationResult(AuthenticationResultType)}.
         * 
         * @param authenticationResult
         *        a consumer that will call methods on {@link AuthenticationResultType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authenticationResult(AuthenticationResultType)
         */
        default Builder authenticationResult(Consumer<AuthenticationResultType.Builder> authenticationResult) {
            return authenticationResult(AuthenticationResultType.builder().applyMutation(authenticationResult).build());
        }

        /**
         * <p>
         * This response parameter lists the available authentication challenges that users can select from in <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
         * >choice-based authentication</a>. For example, they might be able to choose between passkey authentication, a
         * one-time password from an SMS message, and a traditional password.
         * </p>
         * 
         * @param availableChallenges
         *        This response parameter lists the available authentication challenges that users can select from in <a
         *        href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
         *        >choice-based authentication</a>. For example, they might be able to choose between passkey
         *        authentication, a one-time password from an SMS message, and a traditional password.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableChallengesWithStrings(Collection<String> availableChallenges);

        /**
         * <p>
         * This response parameter lists the available authentication challenges that users can select from in <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
         * >choice-based authentication</a>. For example, they might be able to choose between passkey authentication, a
         * one-time password from an SMS message, and a traditional password.
         * </p>
         * 
         * @param availableChallenges
         *        This response parameter lists the available authentication challenges that users can select from in <a
         *        href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
         *        >choice-based authentication</a>. For example, they might be able to choose between passkey
         *        authentication, a one-time password from an SMS message, and a traditional password.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableChallengesWithStrings(String... availableChallenges);

        /**
         * <p>
         * This response parameter lists the available authentication challenges that users can select from in <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
         * >choice-based authentication</a>. For example, they might be able to choose between passkey authentication, a
         * one-time password from an SMS message, and a traditional password.
         * </p>
         * 
         * @param availableChallenges
         *        This response parameter lists the available authentication challenges that users can select from in <a
         *        href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
         *        >choice-based authentication</a>. For example, they might be able to choose between passkey
         *        authentication, a one-time password from an SMS message, and a traditional password.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableChallenges(Collection<ChallengeNameType> availableChallenges);

        /**
         * <p>
         * This response parameter lists the available authentication challenges that users can select from in <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
         * >choice-based authentication</a>. For example, they might be able to choose between passkey authentication, a
         * one-time password from an SMS message, and a traditional password.
         * </p>
         * 
         * @param availableChallenges
         *        This response parameter lists the available authentication challenges that users can select from in <a
         *        href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/authentication-flows-selection-sdk.html#authentication-flows-selection-choice"
         *        >choice-based authentication</a>. For example, they might be able to choose between passkey
         *        authentication, a one-time password from an SMS message, and a traditional password.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableChallenges(ChallengeNameType... availableChallenges);
    }

    static final class BuilderImpl extends CognitoIdentityProviderResponse.BuilderImpl implements Builder {
        private String challengeName;

        private String session;

        private Map<String, String> challengeParameters = DefaultSdkAutoConstructMap.getInstance();

        private AuthenticationResultType authenticationResult;

        private List<String> availableChallenges = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InitiateAuthResponse model) {
            super(model);
            challengeName(model.challengeName);
            session(model.session);
            challengeParameters(model.challengeParameters);
            authenticationResult(model.authenticationResult);
            availableChallengesWithStrings(model.availableChallenges);
        }

        public final String getChallengeName() {
            return challengeName;
        }

        public final void setChallengeName(String challengeName) {
            this.challengeName = challengeName;
        }

        @Override
        public final Builder challengeName(String challengeName) {
            this.challengeName = challengeName;
            return this;
        }

        @Override
        public final Builder challengeName(ChallengeNameType challengeName) {
            this.challengeName(challengeName == null ? null : challengeName.toString());
            return this;
        }

        public final String getSession() {
            return session;
        }

        public final void setSession(String session) {
            this.session = session;
        }

        @Override
        public final Builder session(String session) {
            this.session = session;
            return this;
        }

        public final Map<String, String> getChallengeParameters() {
            if (challengeParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return challengeParameters;
        }

        public final void setChallengeParameters(Map<String, String> challengeParameters) {
            this.challengeParameters = ChallengeParametersTypeCopier.copy(challengeParameters);
        }

        @Override
        public final Builder challengeParameters(Map<String, String> challengeParameters) {
            this.challengeParameters = ChallengeParametersTypeCopier.copy(challengeParameters);
            return this;
        }

        public final AuthenticationResultType.Builder getAuthenticationResult() {
            return authenticationResult != null ? authenticationResult.toBuilder() : null;
        }

        public final void setAuthenticationResult(AuthenticationResultType.BuilderImpl authenticationResult) {
            this.authenticationResult = authenticationResult != null ? authenticationResult.build() : null;
        }

        @Override
        public final Builder authenticationResult(AuthenticationResultType authenticationResult) {
            this.authenticationResult = authenticationResult;
            return this;
        }

        public final Collection<String> getAvailableChallenges() {
            if (availableChallenges instanceof SdkAutoConstructList) {
                return null;
            }
            return availableChallenges;
        }

        public final void setAvailableChallenges(Collection<String> availableChallenges) {
            this.availableChallenges = AvailableChallengeListTypeCopier.copy(availableChallenges);
        }

        @Override
        public final Builder availableChallengesWithStrings(Collection<String> availableChallenges) {
            this.availableChallenges = AvailableChallengeListTypeCopier.copy(availableChallenges);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availableChallengesWithStrings(String... availableChallenges) {
            availableChallengesWithStrings(Arrays.asList(availableChallenges));
            return this;
        }

        @Override
        public final Builder availableChallenges(Collection<ChallengeNameType> availableChallenges) {
            this.availableChallenges = AvailableChallengeListTypeCopier.copyEnumToString(availableChallenges);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availableChallenges(ChallengeNameType... availableChallenges) {
            availableChallenges(Arrays.asList(availableChallenges));
            return this;
        }

        @Override
        public InitiateAuthResponse build() {
            return new InitiateAuthResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
