/*
 * Copyright (C) 2018 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.core.widget

import android.text.Editable
import android.text.TextWatcher
import android.widget.TextView

/**
 * Add an action which will be invoked before the text changed.
 *
 * @return the [TextWatcher] added to the TextView
 */
public inline fun TextView.doBeforeTextChanged(
    crossinline action: (text: CharSequence?, start: Int, count: Int, after: Int) -> Unit
): TextWatcher = addTextChangedListener(beforeTextChanged = action)

/**
 * Add an action which will be invoked when the text is changing.
 *
 * @return the [TextWatcher] added to the TextView
 */
public inline fun TextView.doOnTextChanged(
    crossinline action: (text: CharSequence?, start: Int, before: Int, count: Int) -> Unit
): TextWatcher = addTextChangedListener(onTextChanged = action)

/**
 * Add an action which will be invoked after the text changed.
 *
 * @return the [TextWatcher] added to the TextView
 */
public inline fun TextView.doAfterTextChanged(
    crossinline action: (text: Editable?) -> Unit
): TextWatcher = addTextChangedListener(afterTextChanged = action)

/**
 * Add a text changed listener to this TextView using the provided actions
 *
 * @return the [TextWatcher] added to the TextView
 */
public inline fun TextView.addTextChangedListener(
    crossinline beforeTextChanged:
        (text: CharSequence?, start: Int, count: Int, after: Int) -> Unit =
        { _, _, _, _ ->
        },
    crossinline onTextChanged: (text: CharSequence?, start: Int, before: Int, count: Int) -> Unit =
        { _, _, _, _ ->
        },
    crossinline afterTextChanged: (text: Editable?) -> Unit = {},
): TextWatcher {
    val textWatcher =
        object : TextWatcher {
            override fun afterTextChanged(s: Editable?) {
                afterTextChanged.invoke(s)
            }

            override fun beforeTextChanged(
                text: CharSequence?,
                start: Int,
                count: Int,
                after: Int,
            ) {
                beforeTextChanged.invoke(text, start, count, after)
            }

            override fun onTextChanged(text: CharSequence?, start: Int, before: Int, count: Int) {
                onTextChanged.invoke(text, start, before, count)
            }
        }
    addTextChangedListener(textWatcher)

    return textWatcher
}
