/*
 * Copyright 2015, Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.support.test.rule.logging;

import android.os.Build;
import android.os.Trace;
import android.support.annotation.VisibleForTesting;
import android.support.test.annotation.Beta;

/**
 * When applied to a test class this {@link org.junit.rules.TestRule} calls
 * {@code Trace.beginSection()} before test method execution. The TestRule then calls
 * {@code Trace.endSection()} after test method execution. The TestRule enables Systrace/Atrace
 * output to have discrete information about when/which test methods were run (which helps during
 * performance test debugging).
 * <p/>
 * If the API level is less than 18 (Jelly Bean) this rule will do nothing since {@code Trace} isn't
 * supported.
 * <p/>
 * Usage:
 * <code>
 * \@Rule
 * public EnableTestTracingRule mEnableTestTracing = new EnableTestTracingRule();
 * <code>
 */
@Beta
public class EnableTestTracingRule extends LoggingBaseRule {

    @VisibleForTesting
    boolean mTraceActive = false;

    @Override
    public void afterTest() {
        if (getAndroidRuntimeVersion() >= Build.VERSION_CODES.JELLY_BEAN) {
            mTraceActive = false;
            Trace.endSection();
        }
    }

    @Override
    public void beforeTest() {
        if (getAndroidRuntimeVersion() >= Build.VERSION_CODES.JELLY_BEAN) {
            mTraceActive = true;
            Trace.beginSection(getTestName());
        }
    }

    @Override
    String getDefaultLogFileName() {
        return null;
    }
}
