/*
 * Copyright (C) 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.support.test.rule.logging;

import android.os.Build;
import android.os.Trace;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.test.annotation.Beta;
import android.util.Log;

import org.junit.Rule;

import java.io.File;
import java.io.IOException;

import static android.support.test.rule.logging.RuleLoggingUtils.startProcess;
import static android.support.test.rule.logging.RuleLoggingUtils.writeErrorToFileAndLogcat;

/**
 * When applied to a test class this {@code Rule} attempts to clear the Logcat buffer on the device
 * before test method execution. The {@link Rule} then collects everything in the
 * Logcat buffer after test method execution and logs it to a file on external storage. The default
 * file location is maintained at {@link RuleLoggingUtils#getTestDir(String, String, int)}. The
 * TestRule performs these actions through calling the device's logcat command.
 * <p/>
 * Test methods can perform additional clearing and capturing of the Logcat buffer through the
 * {@link #clearLogcat()} and {@link #dumpLogcat(String)} methods.
 * <p/>
 * Usage:
 * <code>
 * \@Rule
 * public LogLogcatRule mLogLogcatRule = new LogLogcatRule();
 * </code>
 */
@Beta
public class LogLogcatRule extends LoggingBaseRule {

    private static final String TAG = "LogLogcatRule";

    private static final String DEFAULT_LOG_FILE_NAME = "logcat.log";

    /**
     * {@inheritDoc}
     */
    public LogLogcatRule() {
    }

    /**
     * {@inheritDoc}
     */
    public LogLogcatRule(@NonNull File logFileOutputDirectory, @Nullable String fileName) {
        super(logFileOutputDirectory, fileName);
    }

    private static void dumpLogcat(File logFile) {
        Process process = null;
        try {
            if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
                Trace.beginSection("Recording logcat");
            }

            process = startProcess(new String[] { "logcat", "-d", "-f",
                    logFile.getAbsolutePath() });
            process.waitFor();
        } catch (IOException | InterruptedException exception) {
            writeErrorToFileAndLogcat(logFile, TAG, "Error recording logcat output.", exception);
        } finally {
            if (process != null) {
                process.destroy();
            }
            if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
                Trace.endSection();
            }
        }
    }

    /**
     * Clear the current logcat buffer. This can be called from test method code, but will alter the
     * results of the {@link Rule}'s final log of the buffer at the end of the test method.
     */
    public static void clearLogcat() {
        Process process = null;
        try {
            process = RuleLoggingUtils.startProcess(new String[] { "logcat", "-c" });
            process.waitFor();
            if (process.exitValue() != 0) {
                throw new IllegalStateException("Error while clearing logcat, exitValue=" +
                        process.exitValue());
            }
        } catch (IllegalStateException | IOException | InterruptedException exception) {
            Log.e(TAG, "Exception clearing logcat.", exception);
        } finally {
            if (process != null) {
                process.destroy();
            }
        }

        // This log statement ensures the logcat buffer is re-initialized with the normal buffer
        // headers. Do not remove it as it ensures that a dump of the logcat will not be completely
        // empty which could cause confusion when looking at the log file.
        Log.i(TAG, "Logcat cleared by test rule");
    }

    /**
     * Extract logcat buffer to a file after test run.
     */
    @Override
    public void afterTest() {
        dumpLogcat(getLogFileName());
    }

    /**
     * Clear logcat buffer prior to test run.
     */
    @Override
    public void beforeTest() {
        clearLogcat();
    }

    /**
     * Dump the current state of the logcat buffer to a file in the test method's logging directory.
     * This method can be used when writing tests to record logcat at different points in time.
     *
     * @param logFileName where Logcat output is sent
     * @return the {@link File} where output was logged
     */
    public File dumpLogcat(String logFileName) {
        File logFile;
        if (null == getLogFileOutputDirectory()) {
            logFile = RuleLoggingUtils.getTestFile(getTestClass(), getTestName(), logFileName,
                    getTestRunNumber());
        } else {
            logFile = new File(getLogFileOutputDirectory(), logFileName);
        }
        dumpLogcat(logFile);
        return logFile;
    }

    public String getDefaultLogFileName() {
        return DEFAULT_LOG_FILE_NAME;
    }
}