/*
 * Decompiled with CFR 0.152.
 */
package com.android.builder.utils;

import com.android.builder.utils.ExceptionConsumer;
import com.android.builder.utils.ExceptionRunnable;
import com.android.builder.utils.SynchronizedFile;
import com.android.utils.FileUtils;
import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.collect.Maps;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.google.common.io.FileWriteMode;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Stream;

public class FileCache {
    private final File cacheDirectory;
    private final LockingScope lockingScope;
    private final AtomicInteger missCount = new AtomicInteger(0);
    private final AtomicInteger hitCount = new AtomicInteger(0);

    private FileCache(File cacheDirectory, LockingScope lockingScope) {
        this.cacheDirectory = cacheDirectory;
        this.lockingScope = lockingScope;
    }

    public static FileCache getInstanceWithMultiProcessLocking(File cacheDirectory) {
        return new FileCache(cacheDirectory, LockingScope.MULTI_PROCESS);
    }

    public static FileCache getInstanceWithSingleProcessLocking(File cacheDirectory) {
        return new FileCache(cacheDirectory, LockingScope.SINGLE_PROCESS);
    }

    public File getCacheDirectory() {
        return this.cacheDirectory;
    }

    public QueryResult createFile(File outputFile, Inputs inputs, ExceptionRunnable fileCreator) throws ExecutionException, IOException {
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)outputFile, (File)this.cacheDirectory) ? 1 : 0) != 0, (Object)String.format("Output file/directory '%1$s' must not be located in the cache directory '%2$s'", outputFile.getAbsolutePath(), this.cacheDirectory.getAbsolutePath()));
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)this.cacheDirectory, (File)outputFile) ? 1 : 0) != 0, (Object)String.format("Output directory '%1$s' must not contain the cache directory '%2$s'", outputFile.getAbsolutePath(), this.cacheDirectory.getAbsolutePath()));
        Preconditions.checkArgument((!FileUtils.isSameFile((File)outputFile, (File)this.cacheDirectory) ? 1 : 0) != 0, (Object)String.format("Output directory must not be the same as the cache directory '%1$s'", this.cacheDirectory.getAbsolutePath()));
        File cacheEntryDir = this.getCacheEntryDir(inputs);
        File cachedFile = FileCache.getCachedFile(cacheEntryDir);
        Callable<Void> actionIfCacheHit = () -> {
            FileUtils.deletePath((File)outputFile);
            com.google.common.io.Files.createParentDirs((File)outputFile);
            if (cachedFile.exists()) {
                FileCache.copyFileOrDirectory(cachedFile, outputFile);
            }
            return null;
        };
        Callable<Void> actionIfCacheMissedOrCorrupted = () -> {
            FileUtils.deletePath((File)outputFile);
            com.google.common.io.Files.createParentDirs((File)outputFile);
            try {
                fileCreator.run();
            }
            catch (Exception exception) {
                throw new FileCreatorException(exception);
            }
            if (outputFile.exists()) {
                FileCache.copyFileOrDirectory(outputFile, cachedFile);
            }
            return null;
        };
        return this.queryCacheEntry(inputs, cacheEntryDir, actionIfCacheHit, actionIfCacheMissedOrCorrupted);
    }

    public QueryResult createFileInCacheIfAbsent(Inputs inputs, ExceptionConsumer<File> fileCreator) throws ExecutionException, IOException {
        File cacheEntryDir = this.getCacheEntryDir(inputs);
        File cachedFile = FileCache.getCachedFile(cacheEntryDir);
        Callable<Void> actionIfCacheMissedOrCorrupted = () -> {
            try {
                fileCreator.accept(cachedFile);
            }
            catch (Exception exception) {
                throw new FileCreatorException(exception);
            }
            return null;
        };
        QueryResult queryResult = this.queryCacheEntry(inputs, cacheEntryDir, () -> null, actionIfCacheMissedOrCorrupted);
        return new QueryResult(queryResult.getQueryEvent(), queryResult.getCauseOfCorruption(), cachedFile);
    }

    private QueryResult queryCacheEntry(Inputs inputs, File cacheEntryDir, Callable<Void> actionIfCacheHit, Callable<Void> actionIfCacheMissedOrCorrupted) throws ExecutionException, IOException {
        if (this.lockingScope == LockingScope.MULTI_PROCESS) {
            Preconditions.checkNotNull((Object)this.cacheDirectory.getCanonicalFile().getParentFile(), (Object)"Cache directory must not be the root directory");
            FileUtils.mkdirs((File)this.cacheDirectory.getCanonicalFile().getParentFile());
        }
        try {
            return this.getSynchronizedFile(this.cacheDirectory).read(sameCacheDirectory -> {
                FileUtils.mkdirs((File)this.cacheDirectory);
                QueryResult queryResult = this.getSynchronizedFile(cacheEntryDir).read(sameCacheEntryDir -> {
                    QueryResult result = FileCache.checkCacheEntry(inputs, cacheEntryDir);
                    if (result.getQueryEvent().equals((Object)QueryEvent.HIT)) {
                        this.hitCount.incrementAndGet();
                        actionIfCacheHit.call();
                    }
                    return result;
                });
                if (queryResult.getQueryEvent().equals((Object)QueryEvent.HIT)) {
                    return queryResult;
                }
                return this.getSynchronizedFile(cacheEntryDir).write(sameCacheEntryDir -> {
                    QueryResult result = FileCache.checkCacheEntry(inputs, cacheEntryDir);
                    if (result.getQueryEvent().equals((Object)QueryEvent.HIT)) {
                        this.hitCount.incrementAndGet();
                        actionIfCacheHit.call();
                        return result;
                    }
                    if (result.getQueryEvent().equals((Object)QueryEvent.CORRUPTED)) {
                        FileUtils.deletePath((File)cacheEntryDir);
                    }
                    this.missCount.incrementAndGet();
                    FileUtils.mkdirs((File)cacheEntryDir);
                    actionIfCacheMissedOrCorrupted.call();
                    com.google.common.io.Files.asCharSink((File)FileCache.getInputsFile(cacheEntryDir), (Charset)StandardCharsets.UTF_8, (FileWriteMode[])new FileWriteMode[0]).write((CharSequence)inputs.toString());
                    return result;
                });
            });
        }
        catch (ExecutionException exception) {
            for (Throwable exceptionInCausalChain : Throwables.getCausalChain((Throwable)exception)) {
                if (!(exceptionInCausalChain instanceof FileCreatorException)) continue;
                throw exception;
            }
            for (Throwable exceptionInCausalChain : Throwables.getCausalChain((Throwable)exception)) {
                if (!(exceptionInCausalChain instanceof IOException)) continue;
                throw new IOException(exception);
            }
            throw new RuntimeException(exception);
        }
    }

    public boolean cacheEntryExists(Inputs inputs) throws IOException {
        if (this.lockingScope == LockingScope.MULTI_PROCESS) {
            Preconditions.checkNotNull((Object)this.cacheDirectory.getCanonicalFile().getParentFile(), (Object)"Cache directory must not be the root directory");
            FileUtils.mkdirs((File)this.cacheDirectory.getCanonicalFile().getParentFile());
        }
        try {
            QueryResult queryResult = this.getSynchronizedFile(this.cacheDirectory).read(sameCacheDirectory -> {
                FileUtils.mkdirs((File)this.cacheDirectory);
                return this.getSynchronizedFile(this.getCacheEntryDir(inputs)).read(cacheEntryDir -> FileCache.checkCacheEntry(inputs, cacheEntryDir));
            });
            return queryResult.getQueryEvent().equals((Object)QueryEvent.HIT);
        }
        catch (ExecutionException exception) {
            for (Throwable exceptionInCausalChain : Throwables.getCausalChain((Throwable)exception)) {
                if (!(exceptionInCausalChain instanceof IOException)) continue;
                throw new IOException(exception);
            }
            throw new RuntimeException(exception);
        }
    }

    private static QueryResult checkCacheEntry(Inputs inputs, File cacheEntryDir) {
        String inputsInCacheEntry;
        if (!cacheEntryDir.exists()) {
            return new QueryResult(QueryEvent.MISSED);
        }
        File inputsFile = FileCache.getInputsFile(cacheEntryDir);
        if (!inputsFile.exists()) {
            return new QueryResult(QueryEvent.CORRUPTED, (Throwable)new IllegalStateException(String.format("Inputs file '%s' does not exist", inputsFile.getAbsolutePath())));
        }
        try {
            inputsInCacheEntry = com.google.common.io.Files.asCharSource((File)inputsFile, (Charset)StandardCharsets.UTF_8).read();
        }
        catch (IOException e) {
            return new QueryResult(QueryEvent.CORRUPTED, (Throwable)e);
        }
        if (!inputs.toString().equals(inputsInCacheEntry)) {
            return new QueryResult(QueryEvent.CORRUPTED, (Throwable)new IllegalStateException(String.format("Expected contents '%s' but found '%s' in inputs file '%s'", inputs.toString(), inputsInCacheEntry, inputsFile.getAbsolutePath())));
        }
        return new QueryResult(QueryEvent.HIT);
    }

    private File getCacheEntryDir(Inputs inputs) {
        return new File(this.cacheDirectory, inputs.getKey());
    }

    private static File getCachedFile(File cacheEntryDir) {
        return new File(cacheEntryDir, "output");
    }

    private static File getInputsFile(File cacheEntryDir) {
        return new File(cacheEntryDir, "inputs");
    }

    public File getFileInCache(Inputs inputs) {
        return FileCache.getCachedFile(this.getCacheEntryDir(inputs));
    }

    public void deleteOldCacheEntries(long lastTimestamp) {
        if (this.lockingScope == LockingScope.MULTI_PROCESS && !FileUtils.parentDirExists((File)this.cacheDirectory)) {
            return;
        }
        try {
            this.getSynchronizedFile(this.cacheDirectory).write(sameCacheDirectory -> {
                if (!this.cacheDirectory.exists()) {
                    return null;
                }
                for (File fileInDir : (File[])Verify.verifyNotNull((Object)this.cacheDirectory.listFiles())) {
                    File cacheEntryDir;
                    if (!fileInDir.isDirectory() || !FileCache.getInputsFile(fileInDir).isFile() || (cacheEntryDir = fileInDir).lastModified() > lastTimestamp) continue;
                    FileUtils.deletePath((File)cacheEntryDir);
                    if (this.lockingScope != LockingScope.MULTI_PROCESS) continue;
                    FileUtils.deleteIfExists((File)SynchronizedFile.getLockFile(cacheEntryDir));
                }
                return null;
            });
        }
        catch (ExecutionException exception) {
            throw new RuntimeException(exception);
        }
    }

    public void delete() throws IOException {
        if (this.lockingScope == LockingScope.MULTI_PROCESS && !FileUtils.parentDirExists((File)this.cacheDirectory)) {
            return;
        }
        try {
            this.getSynchronizedFile(this.cacheDirectory).write(sameCacheDirectory -> {
                FileUtils.deletePath((File)this.cacheDirectory);
                return null;
            });
        }
        catch (ExecutionException exception) {
            for (Throwable exceptionInCausalChain : Throwables.getCausalChain((Throwable)exception)) {
                if (!(exceptionInCausalChain instanceof IOException)) continue;
                throw new IOException(exception);
            }
            throw new RuntimeException(exception);
        }
    }

    private SynchronizedFile getSynchronizedFile(File fileToSynchronize) {
        if (this.lockingScope == LockingScope.MULTI_PROCESS) {
            Preconditions.checkArgument((boolean)FileUtils.parentDirExists((File)fileToSynchronize), (Object)("Parent directory of " + fileToSynchronize.getAbsolutePath() + " does not exist"));
            return SynchronizedFile.getInstanceWithMultiProcessLocking(fileToSynchronize);
        }
        return SynchronizedFile.getInstanceWithSingleProcessLocking(fileToSynchronize);
    }

    int getMisses() {
        return this.missCount.get();
    }

    int getHits() {
        return this.hitCount.get();
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("cacheDirectory", (Object)this.cacheDirectory).add("lockingScope", (Object)this.lockingScope).toString();
    }

    private static void copyFileOrDirectory(File from, File to) throws IOException {
        Preconditions.checkArgument((boolean)from.exists(), (Object)("Source path " + from.getAbsolutePath() + " does not exist"));
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)from, (File)to) ? 1 : 0) != 0);
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)to, (File)from) ? 1 : 0) != 0);
        Preconditions.checkArgument((!FileUtils.isSameFile((File)from, (File)to) ? 1 : 0) != 0);
        if (from.isFile()) {
            com.google.common.io.Files.createParentDirs((File)to);
            FileUtils.copyFile((File)from, (File)to);
        } else if (from.isDirectory()) {
            FileUtils.deletePath((File)to);
            FileUtils.copyDirectory((File)from, (File)to);
        }
    }

    public static CacheSession newSession() {
        return new CacheSession(){
            private final ConcurrentHashMap<File, String> pathHashes = new ConcurrentHashMap();

            @Override
            String getDirectoryHash(File directory) {
                return this.pathHashes.computeIfAbsent(directory, dir -> Inputs.Builder.getDirectoryHash(dir));
            }

            @Override
            String getRegularFileHash(File regularFile) {
                return this.pathHashes.computeIfAbsent(regularFile, file -> Inputs.Builder.getFileHash(file));
            }
        };
    }

    public static abstract class CacheSession {
        private CacheSession() {
        }

        abstract String getDirectoryHash(File var1);

        abstract String getRegularFileHash(File var1);
    }

    public static enum QueryEvent {
        HIT,
        MISSED,
        CORRUPTED;

    }

    public static final class QueryResult {
        private final QueryEvent queryEvent;
        private final Throwable causeOfCorruption;
        private final File cachedFile;

        QueryResult(QueryEvent queryEvent, Throwable causeOfCorruption, File cachedFile) {
            Preconditions.checkState((queryEvent.equals((Object)QueryEvent.CORRUPTED) && causeOfCorruption != null || !queryEvent.equals((Object)QueryEvent.CORRUPTED) && causeOfCorruption == null ? 1 : 0) != 0);
            this.queryEvent = queryEvent;
            this.causeOfCorruption = causeOfCorruption;
            this.cachedFile = cachedFile;
        }

        private QueryResult(QueryEvent queryEvent, Throwable causeOfCorruption) {
            this(queryEvent, causeOfCorruption, (File)null);
        }

        private QueryResult(QueryEvent queryEvent) {
            this(queryEvent, (Throwable)null, (File)null);
        }

        public QueryEvent getQueryEvent() {
            return this.queryEvent;
        }

        public Throwable getCauseOfCorruption() {
            return this.causeOfCorruption;
        }

        public File getCachedFile() {
            return this.cachedFile;
        }
    }

    public static enum DirectoryProperties {
        HASH,
        PATH_HASH;

    }

    public static enum FileProperties {
        HASH,
        PATH_HASH,
        PATH_SIZE_TIMESTAMP;

    }

    public static enum Command {
        TEST,
        PREDEX_LIBRARY,
        GENERATE_MOCKABLE_JAR,
        PREDEX_LIBRARY_TO_DEX_ARCHIVE,
        DESUGAR_LIBRARY,
        EXTRACT_AAPT2_JNI,
        EXTRACT_DESUGAR_JAR,
        FIX_STACK_FRAMES;

    }

    public static final class Inputs {
        private static final String COMMAND = "COMMAND";
        private final Command command;
        private final LinkedHashMap<String, String> parameters;

        private Inputs(Builder builder) {
            this.command = builder.command;
            this.parameters = Maps.newLinkedHashMap((Map)builder.parameters);
        }

        public String toString() {
            return "COMMAND=" + this.command.name() + System.lineSeparator() + Joiner.on((String)System.lineSeparator()).withKeyValueSeparator("=").join(this.parameters);
        }

        public String getKey() {
            return Hashing.sha256().hashUnencodedChars((CharSequence)this.toString()).toString();
        }

        public static final class Builder {
            private final Command command;
            private final CacheSession session;
            private final LinkedHashMap<String, String> parameters = Maps.newLinkedHashMap();

            public Builder(Command command) {
                this(command, new CacheSession(){

                    @Override
                    String getDirectoryHash(File directory) {
                        return Builder.getDirectoryHash(directory);
                    }

                    @Override
                    String getRegularFileHash(File regularFile) {
                        return Builder.getFileHash(regularFile);
                    }
                });
            }

            public Builder(Command command, CacheSession session) {
                this.command = command;
                this.session = session;
            }

            public Builder putString(String name, String value) {
                Preconditions.checkState((!this.parameters.containsKey(name) ? 1 : 0) != 0, (String)"Input parameter %s already exists", (Object)name);
                this.parameters.put(name, value);
                return this;
            }

            public Builder putBoolean(String name, boolean value) {
                return this.putString(name, String.valueOf(value));
            }

            public Builder putLong(String name, long value) {
                return this.putString(name, String.valueOf(value));
            }

            public Builder putFile(String name, File file, FileProperties fileProperties) {
                Preconditions.checkArgument((boolean)file.isFile(), (Object)(file + " is not a file."));
                switch (fileProperties) {
                    case HASH: {
                        this.putString(name, this.session.getRegularFileHash(file));
                        break;
                    }
                    case PATH_HASH: {
                        this.putString(name + ".path", file.getPath());
                        this.putString(name + ".hash", this.session.getRegularFileHash(file));
                        break;
                    }
                    case PATH_SIZE_TIMESTAMP: {
                        this.putString(name + ".path", file.getPath());
                        this.putLong(name + ".size", file.length());
                        this.putLong(name + ".timestamp", file.lastModified());
                        break;
                    }
                    default: {
                        throw new RuntimeException("Unknown enum " + (Object)((Object)fileProperties));
                    }
                }
                return this;
            }

            public Builder putDirectory(String name, File directory, DirectoryProperties directoryProperties) {
                Preconditions.checkArgument((boolean)directory.isDirectory(), (Object)(directory + " is not a directory."));
                switch (directoryProperties) {
                    case HASH: {
                        this.putString(name, this.session.getDirectoryHash(directory));
                        break;
                    }
                    case PATH_HASH: {
                        this.putString(name + ".path", directory.getPath());
                        this.putString(name + ".hash", this.session.getDirectoryHash(directory));
                        break;
                    }
                    default: {
                        throw new RuntimeException("Unknown enum " + (Object)((Object)directoryProperties));
                    }
                }
                return this;
            }

            static String getFileHash(File file) {
                try {
                    return Hashing.sha256().hashBytes(Files.readAllBytes(file.toPath())).toString();
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }

            static String getDirectoryHash(File directory) {
                Hasher hasher = Hashing.sha256().newHasher();
                try (Stream<Path> entries = Files.walk(directory.toPath(), new FileVisitOption[0]);){
                    Stream<Path> sortedEntries = entries.filter(e -> !FileUtils.isSameFile((File)e.toFile(), (File)directory)).sorted();
                    sortedEntries.forEach(entry -> {
                        hasher.putUnencodedChars((CharSequence)"$$$DIRECTORY_ENTRY_RELATIVE_PATH$$$");
                        hasher.putUnencodedChars((CharSequence)directory.toPath().relativize((Path)entry).toString());
                        if (Files.isRegularFile(entry, new LinkOption[0])) {
                            hasher.putUnencodedChars((CharSequence)"$$$DIRECTORY_ENTRY_FILE_CONTENTS$$$");
                            try {
                                hasher.putBytes(Files.readAllBytes(entry));
                            }
                            catch (IOException e) {
                                throw new UncheckedIOException(e);
                            }
                        }
                    });
                }
                catch (IOException e2) {
                    throw new UncheckedIOException(e2);
                }
                return hasher.hash().toString();
            }

            public Inputs build() {
                Preconditions.checkState((!this.parameters.isEmpty() ? 1 : 0) != 0, (Object)"Inputs must not be empty.");
                return new Inputs(this);
            }
        }
    }

    private static final class FileCreatorException
    extends ExecutionException {
        public FileCreatorException(Exception exception) {
            super(exception);
        }
    }

    private static enum LockingScope {
        MULTI_PROCESS,
        SINGLE_PROCESS;

    }
}

