/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.external.cmake.server;

import com.android.annotations.NonNull;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.Arrays;
import java.util.List;

/** Utilility class for Cmake server. */
public class ServerUtils {
    /**
     * Parses the compile commands json to create the compilation database.
     *
     * @param buildDirectory - build directory where compileCommandsFilename is present
     * @param compileCommandsFilename - json file with compile commands info generated by Cmake
     * @return list of compilation database present in the json file
     * @throws IOException I/O failure
     */
    @NonNull
    public static List<CompileCommand> getCompilationDatabase(
            @NonNull File buildDirectory, @NonNull String compileCommandsFilename)
            throws IOException, JsonSyntaxException {
        File compileCommandsFile = new File(buildDirectory, compileCommandsFilename);
        if (!compileCommandsFile.isFile()) {
            throw new RuntimeException(String.format("File %s not found", compileCommandsFile));
        }
        return getCompilationDatabase(compileCommandsFile);
    }

    /**
     * Parses the compile commands json to create the compilation database given the compile
     * commands json file.
     *
     * @param compileCommandsFile - json file with compile commands info generated by Cmake
     * @return list of compilation database present in the json file
     * @throws IOException I/O failure
     */
    @NonNull
    public static List<CompileCommand> getCompilationDatabase(@NonNull File compileCommandsFile)
            throws IOException, JsonSyntaxException {
        final String text =
                new String(
                        Files.readAllBytes(compileCommandsFile.toPath()), StandardCharsets.UTF_8);
        Gson gson = new GsonBuilder().create();

        return Arrays.asList(gson.fromJson(text, CompileCommand[].class));
    }
}
