/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.tasks;

import static com.android.build.VariantOutput.OutputType.FULL_SPLIT;
import static com.android.build.VariantOutput.OutputType.MAIN;
import static com.android.build.gradle.internal.publishing.AndroidArtifacts.ArtifactScope.ALL;
import static com.android.build.gradle.internal.publishing.AndroidArtifacts.ArtifactType.LINT;
import static com.android.build.gradle.internal.publishing.AndroidArtifacts.ConsumedConfigType.RUNTIME_CLASSPATH;
import static com.android.build.gradle.internal.scope.TaskOutputHolder.AnchorOutputType.ALL_CLASSES;
import static com.android.build.gradle.internal.scope.TaskOutputHolder.TaskOutputType.LIBRARY_MANIFEST;
import static com.android.build.gradle.internal.scope.TaskOutputHolder.TaskOutputType.LINT_JAR;
import static com.android.build.gradle.internal.scope.TaskOutputHolder.TaskOutputType.MANIFEST_MERGE_REPORT;
import static com.android.build.gradle.internal.scope.TaskOutputHolder.TaskOutputType.MERGED_MANIFESTS;

import com.android.annotations.NonNull;
import com.android.annotations.Nullable;
import com.android.build.gradle.internal.dsl.LintOptions;
import com.android.build.gradle.internal.scope.BuildOutput;
import com.android.build.gradle.internal.scope.BuildOutputs;
import com.android.build.gradle.internal.scope.GlobalScope;
import com.android.build.gradle.internal.scope.TaskConfigAction;
import com.android.build.gradle.internal.scope.TaskOutputHolder;
import com.android.build.gradle.internal.scope.VariantScope;
import com.android.build.gradle.internal.tasks.AndroidBuilderTask;
import com.android.build.gradle.options.BooleanOption;
import com.android.builder.model.Version;
import com.android.builder.utils.FileCache;
import com.android.sdklib.BuildToolInfo;
import com.android.tools.lint.gradle.api.ReflectiveLintRunner;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Streams;
import java.io.File;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.gradle.api.Project;
import org.gradle.api.file.ConfigurableFileCollection;
import org.gradle.api.file.FileCollection;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.api.plugins.JavaBasePlugin;
import org.gradle.api.tasks.InputFiles;
import org.gradle.tooling.provider.model.ToolingModelBuilderRegistry;
import org.jetbrains.annotations.NotNull;

public abstract class LintBaseTask extends AndroidBuilderTask {
    public static final String LINT_CLASS_PATH = "lintClassPath";

    protected static final Logger LOG = Logging.getLogger(LintBaseTask.class);

    @Nullable FileCollection lintClassPath;

    /** Lint classpath */
    @InputFiles
    @Nullable
    public FileCollection getLintClassPath() {
        return lintClassPath;
    }

    @Nullable protected LintOptions lintOptions;
    @Nullable protected File sdkHome;
    protected ToolingModelBuilderRegistry toolingRegistry;
    @Nullable protected File reportsDir;

    @Nullable
    public LintOptions getLintOptions() {
        return lintOptions;
    }

    protected void runLint(LintBaseTaskDescriptor descriptor) {
        FileCollection lintClassPath = getLintClassPath();
        if (lintClassPath != null) {
            new ReflectiveLintRunner().runLint(getProject().getGradle(),
                    descriptor, lintClassPath.getFiles());
        }
    }

    protected abstract class LintBaseTaskDescriptor extends
            com.android.tools.lint.gradle.api.LintExecutionRequest {
        @Override
        @Nullable
        public File getSdkHome() {
            return sdkHome;
        }

        @NonNull
        @Override
        public ToolingModelBuilderRegistry getToolingRegistry() {
            return toolingRegistry;
        }

        @Nullable
        @Override
        public LintOptions getLintOptions() {
            return lintOptions;
        }

        @Override
        @Nullable
        public File getReportsDir() {
            return reportsDir;
        }

        @NonNull
        @Override
        public Project getProject() {
            return LintBaseTask.this.getProject();
        }

        @NonNull
        @Override
        public BuildToolInfo getBuildTools() {
            return LintBaseTask.this.getBuildTools();
        }

        @Override
        public void warn(@NonNull String message, @NonNull Object... args) {
            LOG.warn(message, args);
        }

        @NotNull
        @Override
        public String getGradlePluginVersion() {
            return Version.ANDROID_GRADLE_PLUGIN_VERSION;
        }
    }

    @Nullable
    protected static FileCache getUserIntermediatesCache(GlobalScope globalScope) {
        if (globalScope
                .getProjectOptions()
                .get(BooleanOption.ENABLE_INTERMEDIATE_ARTIFACTS_CACHE)) {
            return globalScope.getBuildCache();
        } else {
            return null;
        }
    }

    public static class VariantInputs implements com.android.tools.lint.gradle.api.VariantInputs {
        @NonNull private final String name;
        @NonNull private final FileCollection localLintJarCollection;
        @NonNull private final FileCollection dependencyLintJarCollection;
        @NonNull private final FileCollection mergedManifest;
        @Nullable private final FileCollection mergedManifestReport;
        private List<File> lintRuleJars;

        private final ConfigurableFileCollection allInputs;

        public VariantInputs(@NonNull VariantScope variantScope) {
            name = variantScope.getFullVariantName();
            allInputs = variantScope.getGlobalScope().getProject().files();

            allInputs.from(
                    localLintJarCollection = variantScope.getGlobalScope().getOutput(LINT_JAR));
            allInputs.from(
                    dependencyLintJarCollection =
                            variantScope.getArtifactFileCollection(RUNTIME_CLASSPATH, ALL, LINT));

            if (variantScope.hasOutput(MERGED_MANIFESTS)) {
                mergedManifest = variantScope.getOutput(MERGED_MANIFESTS);
            } else if (variantScope.hasOutput(LIBRARY_MANIFEST)) {
                mergedManifest = variantScope.getOutput(LIBRARY_MANIFEST);
            } else {
                throw new RuntimeException(
                        "VariantInputs initialized with no merged manifest on: "
                                + variantScope.getVariantConfiguration().getType());
            }
            allInputs.from(mergedManifest);

            if (variantScope.hasOutput(MANIFEST_MERGE_REPORT)) {
                allInputs.from(
                        mergedManifestReport = variantScope.getOutput(MANIFEST_MERGE_REPORT));
            } else {
                throw new RuntimeException(
                        "VariantInputs initialized with no merged manifest report on: "
                                + variantScope.getVariantConfiguration().getType());
            }

            // these inputs are only there to ensure that the lint task runs after these build
            // intermediates are built.
            allInputs.from(variantScope.getOutput(ALL_CLASSES));
        }

        @NonNull
        public FileCollection getAllInputs() {
            return allInputs;
        }

        @Override
        @NonNull
        public String getName() {
            return name;
        }

        /** the lint rule jars */
        @Override
        @NonNull
        public List<File> getRuleJars() {
            if (lintRuleJars == null) {
                lintRuleJars =
                        Streams.concat(
                                        dependencyLintJarCollection.getFiles().stream(),
                                        localLintJarCollection.getFiles().stream())
                                .filter(File::isFile)
                                .collect(Collectors.toList());
            }

            return lintRuleJars;
        }

        /** the merged manifest of the current module */
        @Override
        @NonNull
        public File getMergedManifest() {
            File file = mergedManifest.getSingleFile();
            if (file.isFile()) {
                return file;
            }

            Collection<BuildOutput> manifests =
                    BuildOutputs.load(
                            ImmutableList.of(TaskOutputHolder.TaskOutputType.MERGED_MANIFESTS),
                            file);

            if (manifests.isEmpty()) {
                throw new RuntimeException("Can't find any manifest in folder: " + file);
            }

            // first search for a main manifest
            Optional<File> mainManifest =
                    manifests
                            .stream()
                            .filter(buildOutput -> buildOutput.getApkInfo().getType() == MAIN)
                            .map(BuildOutput::getOutputFile)
                            .findFirst();
            if (mainManifest.isPresent()) {
                return mainManifest.get();
            }

            // else search for a full_split with no filters.
            Optional<File> universalSplit =
                    manifests
                            .stream()
                            .filter(
                                    output ->
                                            output.getApkInfo().getType() == FULL_SPLIT
                                                    && output.getFilters().isEmpty())
                            .map(BuildOutput::getOutputFile)
                            .findFirst();

            // return the universal Manifest, or a random one if not found.
            return universalSplit.orElseGet(() -> manifests.iterator().next().getOutputFile());
        }

        @Override
        @Nullable
        public File getManifestMergeReport() {
            if (mergedManifestReport == null) {
                return null;
            }

            return mergedManifestReport.getSingleFile();
        }
    }

    public abstract static class BaseConfigAction<T extends LintBaseTask>
            implements TaskConfigAction<T> {

        @NonNull private final GlobalScope globalScope;

        public BaseConfigAction(@NonNull GlobalScope globalScope) {
            this.globalScope = globalScope;
        }

        @NonNull
        protected GlobalScope getGlobalScope() {
            return globalScope;
        }

        @Override
        public void execute(@NonNull T lintTask) {
            lintTask.setGroup(JavaBasePlugin.VERIFICATION_GROUP);
            lintTask.lintOptions = globalScope.getExtension().getLintOptions();
            File sdkFolder = globalScope.getSdkHandler().getSdkFolder();
            if (sdkFolder != null) {
                lintTask.sdkHome = sdkFolder;
            }

            lintTask.toolingRegistry = globalScope.getToolingRegistry();
            lintTask.reportsDir = globalScope.getReportsDir();
            lintTask.setAndroidBuilder(globalScope.getAndroidBuilder());

            lintTask.lintClassPath = globalScope.getProject().getConfigurations()
                    .getByName(LINT_CLASS_PATH);
        }
    }
}
