/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.tasks

import com.android.build.api.component.impl.AnnotationProcessorImpl
import com.android.build.gradle.internal.component.ComponentCreationConfig
import com.android.build.gradle.internal.publishing.AndroidArtifacts
import com.android.build.gradle.internal.scope.InternalArtifactType
import com.android.build.gradle.internal.services.BuiltInKotlinServices
import com.android.build.gradle.internal.utils.KgpVersion
import com.android.build.gradle.internal.utils.maybeRegister
import com.android.build.gradle.internal.utils.setDisallowChanges
import org.gradle.api.Project
import org.gradle.api.tasks.TaskProvider
import org.jetbrains.kotlin.gradle.dsl.KaptExtensionConfig
import org.jetbrains.kotlin.gradle.tasks.Kapt

internal const val KOTLIN_GROUP = "org.jetbrains.kotlin"
internal const val KAPT_ARTIFACT = "kotlin-annotation-processing-gradle"
internal const val KOTLIN_STDLIB = "kotlin-stdlib"
private const val KAPT_WORKERS_CONFIGURATION = "kotlinKaptWorkerDependencies"

class KaptCreationAction(
    creationConfig: ComponentCreationConfig,
    project: Project,
    private val kotlinServices: BuiltInKotlinServices,
    private val kaptExtension: KaptExtensionConfig
) : KotlinTaskCreationAction<Kapt>(creationConfig) {

    private val kotlinJvmFactory = kotlinServices.kotlinBaseApiPlugin

    private val kaptWorkersDependencies = run {
        project.configurations.maybeRegister(KAPT_WORKERS_CONFIGURATION) {
            isVisible = false
            isCanBeConsumed = false
            dependencies.add(project.dependencies.create("$KOTLIN_GROUP:$KAPT_ARTIFACT:${kotlinServices.kgpVersion}"))
            dependencies.add(project.dependencies.create("$KOTLIN_GROUP:$KOTLIN_STDLIB:${kotlinServices.kgpVersion}"))
        }
    }

    override val taskName: String = creationConfig.computeTaskNameInternal("kapt", "Kotlin")

    override fun getTaskProvider(): TaskProvider<out Kapt> {
        if (kotlinServices.kgpVersion >= KgpVersion.KGP_2_1_0) {
            return kotlinJvmFactory.registerKaptTask(taskName, kaptExtension)
        }
        return kotlinJvmFactory.registerKaptTask(taskName)
    }

    override fun handleProvider(task: TaskProvider<out Kapt>) {
        val artifacts = creationConfig.artifacts

        artifacts.setInitialProvider(task) { it.destinationDir }
            .atLocation(creationConfig.paths.kaptSourceOutputDir)
            .on(InternalArtifactType.BUILT_IN_KAPT_GENERATED_JAVA_SOURCES)
        artifacts.setInitialProvider(task) { it.kotlinSourcesDestinationDir }
            .atLocation(creationConfig.paths.kaptKotlinSourceOutputDir)
            .on(InternalArtifactType.BUILT_IN_KAPT_GENERATED_KOTLIN_SOURCES)
        artifacts.setInitialProvider(task) { it.incAptCache }
            .on(InternalArtifactType.BUILT_IN_KAPT_INCREMENTAL_CACHE)
        artifacts.setInitialProvider(task) { it.classesDir }
            .on(InternalArtifactType.BUILT_IN_KAPT_CLASSES_DIR)
    }

    override fun configureTask(task: Kapt) {
        task.sourceSetName.set(creationConfig.name)

        creationConfig.sources.java {
            task.source.from(it.all)
        }
        task.stubsDir.set(creationConfig.artifacts.get(InternalArtifactType.BUILT_IN_KAPT_STUBS))
        // BUILT_IN_KAPT_STUBS must be added to task.source too because task.source is intended to
        // contain the Java source code generated by the related KaptGenerateStubs task.
        task.source.from(creationConfig.artifacts.get(InternalArtifactType.BUILT_IN_KAPT_STUBS))
        // Never add jdk classes to classpath with Android as android.jar should be used
        task.addJdkClassesToClasspath.set(false)
        task.classpath.setFrom(
            creationConfig.global.bootClasspath,
            creationConfig.getJavaClasspath(
                AndroidArtifacts.ConsumedConfigType.COMPILE_CLASSPATH,
                AndroidArtifacts.ArtifactType.CLASSES_JAR,
                null
            ),
        )
        val externalKaptDeps = creationConfig.variantDependencies.getArtifactFileCollection(
            AndroidArtifacts.ConsumedConfigType.ANNOTATION_PROCESSOR,
            AndroidArtifacts.ArtifactScope.EXTERNAL,
            AndroidArtifacts.ArtifactType.PROCESSED_JAR
        )
        task.kaptClasspath.from(
            creationConfig.variantDependencies.getArtifactFileCollection(
                AndroidArtifacts.ConsumedConfigType.ANNOTATION_PROCESSOR,
                AndroidArtifacts.ArtifactScope.PROJECT,
                AndroidArtifacts.ArtifactType.JAR
            ),
            externalKaptDeps
        )
        // This does not create a circular dependency as this input is @Internal. These classes are
        // needed in order to support incremental annotation processing with KAPT.
        task.compiledSources.from(task.project.provider {
            listOf(
                creationConfig.artifacts.get(InternalArtifactType.JAVAC).get().asFile,
                creationConfig.artifacts.get(InternalArtifactType.BUILT_IN_KOTLINC).get().asFile,
            )
        })
        task.kaptJars.from(kaptWorkersDependencies)
        task.defaultJavaSourceCompatibility
            .set(creationConfig.global.compileOptions.sourceCompatibility.toString())

        // Add annotation processing options
        val processorOptions = creationConfig.javaCompilation.annotationProcessor
        task.annotationProcessorOptionProviders.add(
            listOf(
                CommandLineArgumentProviderAdapter(
                    (processorOptions as AnnotationProcessorImpl).finalListOfClassNames,
                    processorOptions.arguments
                )
            )
        )
        task.annotationProcessorOptionProviders.add(processorOptions.argumentProviders)
        task.includeCompileClasspath.setDisallowChanges(false)
    }
}
