/*
 * Copyright (C) 2007 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.android.resources;

import com.android.annotations.NonNull;
import com.android.annotations.Nullable;
import java.util.HashMap;
import java.util.Map;

/**
 * Enum representing a type of compiled resource.
 */
public enum ResourceType {
    ANIM("anim", "Animation"),
    ANIMATOR("animator", "Animator"),
    ARRAY("array", "Array", "string-array", "integer-array"),
    ATTR("attr", "Attr"),
    BOOL("bool", "Boolean"),
    COLOR("color", "Color"),
    DECLARE_STYLEABLE("declare-styleable", "Declare Styleable"),
    DIMEN("dimen", "Dimension"),
    DRAWABLE("drawable", "Drawable"),
    FONT("font", "Font"),
    FRACTION("fraction", "Fraction"),
    ID("id", "ID"),
    INTEGER("integer", "Integer"),
    INTERPOLATOR("interpolator", "Interpolator"),
    LAYOUT("layout", "Layout"),
    MENU("menu", "Menu"),
    MIPMAP("mipmap", "Mip Map"),
    NAVIGATION("navigation", "Navigation"),
    PLURALS("plurals", "Plurals"),
    RAW("raw", "Raw"),
    STRING("string", "String"),
    STYLE("style", "Style"),
    STYLEABLE("styleable", "Styleable"),
    TRANSITION("transition", "Transition"),
    XML("xml", "XML"),

    /**
     * This is not actually used. Only there because they get parsed and since we want to detect new
     * resource type, we need to have this one exist.
     */
    PUBLIC("public", "###"),

    /**
     * This type is used for elements dynamically generated by the parsing of aapt:attr nodes. The
     * "aapt:attr" allow to inline resources as part of a different resource, for example, a
     * drawable as part of a layout. When the parser, encounters one of this nodes, it will generate
     * a synthetic _aaptattr reference.
     */
    AAPT("_aapt", "Aapt Attribute"),

    /** Represents item tags inside a style definition. */
    STYLE_ITEM("item", "Style Item"),

    /**
     * Not an actual resource type from AAPT. Used to provide sample data values in the tools
     * namespace
     */
    SAMPLE_DATA("sample", "Sample data");

    private final String mName;
    private final String mDisplayName;
    private final String[] mAlternateXmlNames;

    ResourceType(String name, String displayName, String... alternateXmlNames) {
        mName = name;
        mDisplayName = displayName;
        mAlternateXmlNames = alternateXmlNames;
    }

    private static final Map<String, ResourceType> sNameToType;

    static {
        ResourceType[] values = ResourceType.values();
        int size = 0;
        for (ResourceType type : values) {
            ++size;
            if (type.mAlternateXmlNames != null) {
                size += (type.mAlternateXmlNames.length);
            }
        }
        sNameToType = new HashMap<>(2 * size);
        for (ResourceType type : values) {
            sNameToType.put(type.getName(), type);
            if (type.mAlternateXmlNames != null) {
                for (String alternateName : type.mAlternateXmlNames) {
                    sNameToType.put(alternateName, type);
                }
            }
        }
    }

    /**
     * Returns the resource type name, as used by XML files.
     */
    @NonNull
    public String getName() {
        return mName;
    }

    /**
     * Returns a translated display name for the resource type.
     */
    @NonNull
    public String getDisplayName() {
        return mDisplayName;
    }

    /**
     * Returns the enum by its name as it appears in the XML or the R class.
     * @param name name of the resource
     * @return the matching {@link ResourceType} or <code>null</code> if no match was found.
     */
    @Nullable
    public static ResourceType getEnum(@NonNull String name) {
        return sNameToType.get(name);
    }

    @Override
    public String toString() {
        return getName();
    }
}
