/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.cluster.service;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.jira.cluster.ClusterManager;
import com.atlassian.jira.cluster.ClusterSettings;
import com.atlassian.jira.cluster.Node;
import com.atlassian.jira.cluster.service.NodeTimeHelper;
import com.atlassian.jira.cluster.service.analytics.ClusterStateChangedEvent;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OfflineNodesScannerService {
    private static final Logger log = LoggerFactory.getLogger(OfflineNodesScannerService.class);
    static final Duration WARNING_MESSAGE_PERIOD = Duration.ofMinutes(5L);
    private static Clock clock = Clock.systemUTC();
    private final ClusterManager clusterManager;
    private final EventPublisher eventPublisher;
    private final NodeTimeHelper nodeTimeHelper;
    private final Duration retentionPeriod;
    private volatile Instant nextWarningMessageRun;

    public OfflineNodesScannerService(ClusterManager clusterManager, EventPublisher eventPublisher, NodeTimeHelper nodeTimeHelper) {
        this.clusterManager = clusterManager;
        this.eventPublisher = eventPublisher;
        this.nodeTimeHelper = nodeTimeHelper;
        this.nextWarningMessageRun = Instant.now(clock);
        this.retentionPeriod = ClusterSettings.getClusterRetentionPeriod();
        log.info("{} Service has been registered with retention period {}", (Object)"[CLUSTER-STATE]", (Object)this.retentionPeriod);
    }

    void removeExpiredOfflineNodes() {
        this.notifyAboutNodesInOfflineState(this.retentionPeriod);
        log.info("{} Service is starting to check the cluster state with retention period {}", (Object)"[CLUSTER-STATE]", (Object)this.retentionPeriod);
        List<String> nodesRemoved = this.clusterManager.removeOfflineNodesIfOlderThan(this.retentionPeriod);
        if (!nodesRemoved.isEmpty()) {
            log.warn("{} {} nodes removed from the cluster: {}.", new Object[]{"[CLUSTER-STATE]", nodesRemoved.size(), nodesRemoved});
        } else {
            String currentClusterState = this.getCurrentClusterStateData();
            log.info("{} Service did not find any stale OFFLINE nodes. {} ", (Object)"[CLUSTER-STATE]", (Object)currentClusterState);
        }
        this.sendAnalytics(nodesRemoved.size());
    }

    private void notifyAboutNodesInOfflineState(Duration retentionPeriod) {
        Instant now = Instant.now(clock);
        if (now.isAfter(this.nextWarningMessageRun)) {
            this.clusterManager.getAllNodes().stream().filter(node -> this.clusterManager.isNodeOffline(node.getNodeId())).forEach(node -> this.logAboutNodeInOfflineState((Node)node, retentionPeriod));
            this.nextWarningMessageRun = now.plusMillis(WARNING_MESSAGE_PERIOD.toMillis());
        }
    }

    private String getCurrentClusterStateData() {
        int numberOfNodes = this.clusterManager.getAllNodes().size();
        int numberOfActiveNodes = this.clusterManager.findLiveNodes().size();
        int numberOfActiveAndNotAliveNodes = this.clusterManager.findActiveAndNotAliveNodes().size();
        int numberOfOfflineNodes = this.clusterManager.findOfflineNodes().size();
        return String.format("Current cluster state: {numberOfNodes=%d, numberOfActiveNodes=%d, numberOfActiveNotAliveNodes=%d, numberOfOfflineNodes=%d}", numberOfNodes, numberOfActiveNodes, numberOfActiveAndNotAliveNodes, numberOfOfflineNodes);
    }

    private void logAboutNodeInOfflineState(Node node, Duration retentionPeriod) {
        Duration nextRemoveAction = this.nodeTimeHelper.getEstimatedRetentionTime(node, retentionPeriod, clock);
        Duration nodeInState = this.nodeTimeHelper.getTimeOfNodeBeingInCurrentState(node, clock);
        log.info("{} Node {} is in the OFFLINE state for {}. It will be removed from the cluster approximately after {}", new Object[]{"[CLUSTER-STATE]", node.getNodeId(), nodeInState, nextRemoveAction});
    }

    private void sendAnalytics(int numberOfAffectedNodes) {
        if (numberOfAffectedNodes > 0) {
            String actionName = ClusterStateChangedEvent.ActionNames.NODES_REMOVED.getActionName();
            ClusterStateChangedEvent event = new ClusterStateChangedEvent(actionName, numberOfAffectedNodes);
            this.eventPublisher.publish((Object)event);
        }
    }

    @VisibleForTesting
    static void setClock(Clock clock) {
        OfflineNodesScannerService.clock = clock;
    }
}

