/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.cluster.dbr;

import com.atlassian.jira.cluster.dbr.DBRMessage;
import com.atlassian.jira.cluster.dbr.DBRMessageDataSerialized;
import com.atlassian.jira.cluster.dbr.DBRMessageDataSerializer;
import com.atlassian.jira.cluster.dbr.DBRMessageUpdateData;
import com.atlassian.jira.cluster.dbr.DBRMessageUpdateWithRelatedData;
import com.atlassian.jira.cluster.dbr.DBRReceiverStats;
import com.atlassian.jira.cluster.dbr.DBRStatsListener;
import com.atlassian.jira.cluster.dbr.transport.DBRTransportManager;
import com.atlassian.jira.index.QueueingIndexPriority;
import com.atlassian.jira.index.ha.NodeReindexService;
import com.atlassian.jira.issue.index.IndexDirectoryFactory;
import com.atlassian.jira.issue.index.InternalIndexingService;
import com.atlassian.jira.issue.index.exception.CannotGetIndexLockException;
import com.atlassian.jira.util.index.IndexLifecycleManager;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.gson.Gson;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DBRReceiver {
    private static final Logger log = LoggerFactory.getLogger(DBRReceiver.class);
    private final DBRReceiverStats.TotalAndSnapshotDBRReceiverStats stats = new DBRReceiverStats.TotalAndSnapshotDBRReceiverStats(new DBRStatsListener());
    private final InternalIndexingService internalIndexingService;
    private final IndexLifecycleManager indexLifecycleManager;
    private final DBRMessageDataSerializer dbrMessageDataSerializer;
    private final NodeReindexService nodeReindexService;

    public DBRReceiver(DBRTransportManager dbrTransportManager, IndexLifecycleManager indexLifecycleManager, InternalIndexingService internalIndexingService, DBRMessageDataSerializer dbrMessageDataSerializer, NodeReindexService nodeReindexService) {
        this.internalIndexingService = internalIndexingService;
        this.indexLifecycleManager = indexLifecycleManager;
        this.dbrMessageDataSerializer = dbrMessageDataSerializer;
        this.nodeReindexService = nodeReindexService;
        dbrTransportManager.register(this::receive);
    }

    void receive(DBRMessage dbrMessage) {
        this.stats.onReceiveDBRMessage(dbrMessage);
        log.trace("[DBR] [RECEIVER] Received a dbr message: {}", (Object)dbrMessage);
        if (this.shouldSkipDBRMessage(dbrMessage)) {
            return;
        }
        if (dbrMessage.operation == DBRMessage.Operation.UPDATE) {
            this.processDBRMessageUpdate(dbrMessage);
        } else {
            this.processDBRMessageUpdateWithRelated(dbrMessage);
        }
    }

    private boolean shouldSkipDBRMessage(DBRMessage dbrMessage) {
        Preconditions.checkNotNull((Object)dbrMessage);
        Preconditions.checkNotNull(dbrMessage.getDbrMessageDataSerialized());
        Preconditions.checkNotNull((Object)dbrMessage.getDbrMessageDataSerialized().getData());
        if (!this.indexLifecycleManager.isIndexAvailable()) {
            this.stats.onSkipDBRMessageWhenIndexNotAvailable(dbrMessage);
            log.trace("[DBR] [RECEIVER] Skipped a dbr message because indexing isn't available at this moment. DBR message: {}", (Object)dbrMessage);
            return true;
        }
        if (this.nodeReindexService.isPaused()) {
            this.stats.onSkipDBRMessageWhenIndexReplicationPaused(dbrMessage);
            log.trace("[DBR] [RECEIVER] Skipped a dbr message because index replication is paused. DBR message: {}", (Object)dbrMessage);
            return true;
        }
        return false;
    }

    private void processDBRMessageUpdate(DBRMessage dbrMessage) {
        Preconditions.checkArgument((dbrMessage.getOperation() == DBRMessage.Operation.UPDATE ? 1 : 0) != 0);
        try {
            IndexDirectoryFactory.Name index = dbrMessage.getIndex();
            DBRMessageDataSerialized dbrMessageUpdateData = dbrMessage.getDbrMessageDataSerialized();
            Stopwatch deserializeStopwatch = Stopwatch.createStarted();
            DBRMessageUpdateData deserialized = (DBRMessageUpdateData)this.dbrMessageDataSerializer.deserialize(dbrMessageUpdateData);
            log.trace("[DBR] [RECEIVER] Deserialized a dbr update message: {}", (Object)dbrMessage);
            long deserializeDuration = deserializeStopwatch.stop().elapsed(TimeUnit.MILLISECONDS);
            Stopwatch indexStopwatch = Stopwatch.createStarted();
            QueueingIndexPriority.runWithIndexingPriority(() -> this.internalIndexingService.conditionalUpdateWithVersion(index, deserialized.getDocument()), -100);
            log.trace("[DBR] [RECEIVER] Indexed a dbr update message: {}", (Object)dbrMessage);
            long indexDuration = indexStopwatch.stop().elapsed(TimeUnit.MILLISECONDS);
            this.stats.onProcessDBRMessageUpdate(index, dbrMessageUpdateData.getData().length, deserializeDuration, indexDuration);
        }
        catch (CannotGetIndexLockException t) {
            log.trace("[DBR] [RECEIVER] Skipped a dbr message because indexing isn't available at this moment. DBR message: {}. Exception: {}", (Object)dbrMessage, (Object)t);
            this.stats.onSkipDBRMessageWhenIndexNotAvailable(dbrMessage);
        }
        catch (Exception t) {
            log.error("[DBR] [RECEIVER] Error when processing a dbr update message: {}", (Object)dbrMessage, (Object)t);
            this.stats.onProcessDBRMessageUpdateError(t);
        }
    }

    private void processDBRMessageUpdateWithRelated(DBRMessage dbrMessage) {
        Preconditions.checkArgument((dbrMessage.getOperation() == DBRMessage.Operation.UPDATE_WITH_RELATED ? 1 : 0) != 0);
        try {
            DBRMessageDataSerialized dbrMessageUpdateData = dbrMessage.getDbrMessageDataSerialized();
            Stopwatch deserializeStopwatch = Stopwatch.createStarted();
            DBRMessageUpdateWithRelatedData deserialized = (DBRMessageUpdateWithRelatedData)this.dbrMessageDataSerializer.deserialize(dbrMessageUpdateData);
            log.trace("[DBR] [RECEIVER] Deserialized a dbr update with related message: {}", (Object)dbrMessage);
            long deserializeDuration = deserializeStopwatch.stop().elapsed(TimeUnit.MILLISECONDS);
            Stopwatch indexStopwatch = Stopwatch.createStarted();
            this.internalIndexingService.conditionalUpdateWithVersion(deserialized.getIssue(), deserialized.getComments(), deserialized.getChanges(), deserialized.getWorklogs());
            log.trace("[DBR] [RECEIVER] Indexed a dbr update with related message: {}", (Object)dbrMessage);
            long indexDuration = indexStopwatch.stop().elapsed(TimeUnit.MILLISECONDS);
            this.stats.onProcessDBRMessageUpdateWithRelated(deserialized, dbrMessageUpdateData.getData().length, deserializeDuration, indexDuration);
        }
        catch (CannotGetIndexLockException t) {
            log.trace("[DBR] [RECEIVER] Skipped a dbr message because indexing isn't available at this moment. DBR message: {}. Exception: {}", (Object)dbrMessage, (Object)t);
            this.stats.onSkipDBRMessageWhenIndexNotAvailable(dbrMessage);
        }
        catch (Exception t) {
            log.error("[DBR] [RECEIVER] Error when processing a dbr update with related message: {}", (Object)dbrMessage, (Object)t);
            this.stats.onProcessDBRMessageUpdateWithRelatedError(t);
        }
    }

    DBRReceiverStats.MutableDBRReceiverStats.Result total() {
        return this.stats.total();
    }

    public Map totalStats() {
        Gson gson = new Gson();
        return (Map)gson.fromJson(this.stats.total().toString(), Map.class);
    }

    DBRReceiverStats.MutableDBRReceiverStats.Result snapshot(boolean reset) {
        return this.stats.snapshot(reset);
    }
}

