/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.index.ha;

import com.atlassian.event.api.EventPublisher;
import com.atlassian.jira.cluster.ClusterNodes;
import com.atlassian.jira.cluster.Node;
import com.atlassian.jira.cluster.disasterrecovery.JiraHomeChangeEvent;
import com.atlassian.jira.config.properties.ApplicationProperties;
import com.atlassian.jira.config.properties.PropertiesUtil;
import com.atlassian.jira.index.PeriodicIndexWriterCommitScheduler;
import com.atlassian.jira.index.ha.TemporaryFilesProvider;
import com.atlassian.jira.index.ha.backup.CloseableBackupBuilder;
import com.atlassian.jira.index.ha.backup.IndexBackupContributorsManager;
import com.atlassian.jira.index.ha.events.IndexArchiverSelectedEvent;
import com.atlassian.jira.index.ha.events.IndexSnapshotEvent;
import com.atlassian.jira.issue.index.IndexException;
import com.atlassian.jira.util.TempDirectoryUtil;
import com.atlassian.jira.util.compression.ArchiveUtils;
import com.atlassian.jira.util.index.IndexLifecycleManager;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import com.google.common.io.PatternFilenameFilter;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.comparator.LastModifiedFileComparator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexUtils {
    private static final Logger log = LoggerFactory.getLogger(IndexUtils.class);
    private static final String DEFAULT_INDEX_SNAPSHOT_COPY_ARCHIVER = "snappy";
    public static final String INDEX_SNAPSHOT_PREFIX = "IndexSnapshot_";
    private static final String INDEX_SNAPSHOT_EXT = "[" + String.join((CharSequence)"|", Arrays.stream(ArchiveUtils.Type.values()).map(ArchiveUtils.Type::getExtension).map(Pattern::quote).collect(Collectors.toList())) + "]";
    private static final Pattern INDEX_SNAPSHOT_PATTERN = Pattern.compile(Pattern.quote("IndexSnapshot_") + ".*" + INDEX_SNAPSHOT_EXT);
    public static final PatternFilenameFilter INDEX_SNAPSHOT_FILTER = new PatternFilenameFilter(INDEX_SNAPSHOT_PATTERN);
    public static final int MAX_SNAPSHOTS = 3;
    private final EventPublisher eventPublisher;
    private final ApplicationProperties applicationProperties;
    private final IndexBackupContributorsManager indexBackupContributorsManager;
    private final ClusterNodes clusterNodes;
    private final IndexLifecycleManager indexLifecycleManager;
    private final PeriodicIndexWriterCommitScheduler periodicIndexWriterCommitScheduler;

    public IndexUtils(EventPublisher eventPublisher, ApplicationProperties applicationProperties, IndexBackupContributorsManager indexBackupContributorsManager, ClusterNodes clusterNodes, IndexLifecycleManager indexLifecycleManager, PeriodicIndexWriterCommitScheduler periodicIndexWriterCommitScheduler) {
        this.eventPublisher = eventPublisher;
        this.applicationProperties = applicationProperties;
        this.indexBackupContributorsManager = indexBackupContributorsManager;
        this.clusterNodes = clusterNodes;
        this.indexLifecycleManager = indexLifecycleManager;
        this.periodicIndexWriterCommitScheduler = periodicIndexWriterCommitScheduler;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String performBackupOperations(@Nonnull String destinationPath, @Nonnull String snapshotId, @Nullable TemporaryFilesProvider metadataProvider, @Nullable String requestingNodeId) throws IndexException, InterruptedException, ExecutionException, TimeoutException {
        log.info("Preparing to create an index snapshot");
        if (!this.indexLifecycleManager.isIndexConsistent()) {
            throw new IndexException("Index backup can be done only on consistent index.");
        }
        ArchiveUtils.Type archiveType = this.getArchiverBasedOnAppProperties();
        Stopwatch totalTimeStopwatch = Stopwatch.createStarted();
        this.periodicIndexWriterCommitScheduler.forceImmediateCommit();
        log.info("Committing writers to disk before taking a snapshot took {}", (Object)totalTimeStopwatch.elapsed());
        String filename = this.deriveFilename(snapshotId, archiveType);
        File destination = new File(destinationPath);
        if (!destination.exists()) {
            destination.mkdir();
        }
        File tmpSnapshotDir = TempDirectoryUtil.createTempDirectory("snapshot");
        File tmpSnapshotFile = new File(tmpSnapshotDir, filename);
        File sharedSnapshotFie = new File(destination, filename);
        Stopwatch compressionTimeStopwatch = Stopwatch.createStarted();
        log.info("Creating index snapshot of type: {} in local temporary directory: {}", (Object)archiveType, (Object)tmpSnapshotFile);
        Long uncompressedFilesProcessedSize = this.createIndexSnapshot(tmpSnapshotFile, archiveType, metadataProvider);
        Duration compressionTime = compressionTimeStopwatch.elapsed();
        long compressedSize = this.calculateFileSize(tmpSnapshotFile);
        try {
            this.copyFiles(tmpSnapshotFile, sharedSnapshotFie);
            this.eventPublisher.publish((Object)new JiraHomeChangeEvent(JiraHomeChangeEvent.Action.FILE_ADD, JiraHomeChangeEvent.FileType.INDEX_SNAPSHOT, sharedSnapshotFie));
            int maxSnapshots = PropertiesUtil.getIntProperty((ApplicationProperties)this.applicationProperties, (String)"jira.index.snapshot.count", (int)3);
            this.deleteOldSnapshots(destination, maxSnapshots);
            Duration totalTime = totalTimeStopwatch.elapsed();
            int nodeIdentifier = this.getNodeIdentifier(requestingNodeId);
            log.info("Finished taking a snapshot: {}. File size uncompressed: {}, compressed: {}. Compression time {} ms, total time: {} ms", new Object[]{sharedSnapshotFie.getPath(), FileUtils.byteCountToDisplaySize((long)uncompressedFilesProcessedSize), FileUtils.byteCountToDisplaySize((long)compressedSize), compressionTime.toMillis(), totalTime.toMillis()});
            this.eventPublisher.publish((Object)new IndexSnapshotEvent(nodeIdentifier, uncompressedFilesProcessedSize, compressedSize, compressionTime, totalTime, archiveType.getExtension()));
        }
        catch (IOException filesCopyIOException) {
            log.error("Error when copying index snapshot from: {} to remote shared directory: {}! WARNING - Without up-to-date index snapshots regularly saved to shared home time to start/restart nodes in the cluster might significantly increase!  Other nodes will not be able to use recent index snapshots and might need to perform a full reindex on start. Make sure shared home is accessible for this node and has sufficient amount of space to accept new index snapshots! Current shapshot size: {}", new Object[]{tmpSnapshotFile.toPath(), sharedSnapshotFie.toPath(), FileUtils.byteCountToDisplaySize((long)compressedSize), filesCopyIOException});
            if (sharedSnapshotFie.delete()) {
                log.info("Deleted incomplete index snapshot from shared directory: {} ", (Object)sharedSnapshotFie);
            }
        }
        finally {
            try {
                if (tmpSnapshotDir.exists()) {
                    log.debug("Deleting temporary directory with index snapshot: {}", (Object)tmpSnapshotDir);
                    FileUtils.deleteDirectory((File)tmpSnapshotDir);
                }
            }
            catch (IOException e) {
                log.warn("Could not delete temporary snapshot directory: {}", (Object)tmpSnapshotDir, (Object)e);
            }
        }
        return filename;
    }

    @VisibleForTesting
    void copyFiles(File tmpSnapshotFile, File sharedSnapshotFie) throws IOException {
        log.info("Copying index snapshot from: {} to remote shared directory: {}", (Object)tmpSnapshotFile.toPath(), (Object)sharedSnapshotFie.toPath());
        Files.copy(tmpSnapshotFile.toPath(), sharedSnapshotFie.toPath(), new CopyOption[0]);
        log.info("Finished copying index snapshot from: {} to remote shared directory: {}", (Object)tmpSnapshotFile.toPath(), (Object)sharedSnapshotFie.toPath());
    }

    public String deriveFilename(String snapshotId, ArchiveUtils.Type archiveType) {
        if (archiveType == null) {
            archiveType = this.getArchiverBasedOnAppProperties();
        }
        return INDEX_SNAPSHOT_PREFIX + snapshotId + archiveType.getExtension();
    }

    private int getNodeIdentifier(String requestingNodeId) {
        return Optional.ofNullable(requestingNodeId).map(this.clusterNodes::node).map(IndexUtils::getNodeIdentifier).orElse(0);
    }

    static int getNodeIdentifier(Node node) {
        return Objects.hash(node.getNodeId(), node.getIp());
    }

    protected Long createIndexSnapshot(@Nonnull File snapshot, @Nonnull ArchiveUtils.Type archiveType, @Nullable TemporaryFilesProvider metadataProvider) {
        FileSizeCounter uncompressedSizeCounter = new FileSizeCounter();
        ProcessedFileLogger processedFileLogger = new ProcessedFileLogger(10, snapshot.getName());
        try (CloseableBackupBuilder builder = CloseableBackupBuilder.open(snapshot, archiveType, processedFileLogger.andThen(uncompressedSizeCounter));){
            this.indexBackupContributorsManager.pollContributors(builder);
            this.addMetadata(metadataProvider, builder);
        }
        processedFileLogger.done();
        return uncompressedSizeCounter.getSize();
    }

    protected long calculateFileSize(File snapshot) {
        return FileUtils.sizeOf((File)snapshot);
    }

    private void addMetadata(TemporaryFilesProvider metadataProvider, CloseableBackupBuilder backupBuilder) {
        if (metadataProvider != null) {
            File workDir = TempDirectoryUtil.createTempDirectory("JIRAIndexBackup");
            try {
                Collection<File> contributionFiles = metadataProvider.writeContribution(workDir);
                backupBuilder.addToBackup(contributionFiles, workDir);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            finally {
                FileUtils.deleteQuietly((File)workDir);
            }
        }
    }

    private ArchiveUtils.Type getArchiverBasedOnAppProperties() {
        String archiveName = this.applicationProperties.getDefaultBackedString("jira.index.snapshot.copy.archiver");
        if (archiveName == null) {
            archiveName = DEFAULT_INDEX_SNAPSHOT_COPY_ARCHIVER;
        }
        ArchiveUtils.Type archiveType = ArchiveUtils.Type.fromName(archiveName);
        this.eventPublisher.publish((Object)new IndexArchiverSelectedEvent(archiveType.getExtension()));
        return archiveType;
    }

    @VisibleForTesting
    protected int deleteOldSnapshots(File directory, int numToKeep) {
        File[] snapshots = directory.listFiles((FilenameFilter)INDEX_SNAPSHOT_FILTER);
        log.debug("Deleting old index snapshot files. Number of index snapshots on shared: {}, number of index snapshots to keep: {}", (Object)snapshots.length, (Object)numToKeep);
        Arrays.sort(snapshots, LastModifiedFileComparator.LASTMODIFIED_REVERSE);
        int numKept = 0;
        int numDeleted = 0;
        for (File snapshot : snapshots) {
            if (numKept < numToKeep) {
                ++numKept;
                continue;
            }
            if (!snapshot.delete()) continue;
            this.eventPublisher.publish((Object)new JiraHomeChangeEvent(JiraHomeChangeEvent.Action.FILE_DELETED, JiraHomeChangeEvent.FileType.INDEX_SNAPSHOT, snapshot));
            ++numDeleted;
        }
        if (numDeleted > 0) {
            log.info("Deleted: {} old index snapshot files", (Object)numDeleted);
        } else {
            log.debug("Deleted: {} old index snapshot files", (Object)numDeleted);
        }
        return numDeleted;
    }

    private class ProcessedFileLogger
    implements Consumer<File> {
        private int counter = 0;
        private final Stopwatch stopwatch;
        private final int minLogFrequencyInSeconds;
        private final String snapshotName;

        ProcessedFileLogger(int minLogFrequencyInSeconds, String snapshotName) {
            this.minLogFrequencyInSeconds = minLogFrequencyInSeconds;
            this.snapshotName = snapshotName;
            this.stopwatch = Stopwatch.createStarted();
        }

        @Override
        public void accept(File file) {
            ++this.counter;
            if (this.stopwatch.elapsed(TimeUnit.SECONDS) >= (long)this.minLogFrequencyInSeconds) {
                log.info("Progress of creating index snapshot: {}, number of index files processed: {} ...", (Object)this.snapshotName, (Object)this.counter);
                this.stopwatch.reset().start();
            }
        }

        void done() {
            log.info("Done creating index snapshot: {}, number of index files processed: {}", (Object)this.snapshotName, (Object)this.counter);
            this.stopwatch.stop();
        }
    }

    private static class FileSizeCounter
    implements Consumer<File> {
        private long size = 0L;

        private FileSizeCounter() {
        }

        @Override
        public void accept(File file) {
            if (file.exists()) {
                this.size += FileUtils.sizeOf((File)file);
            }
        }

        public long getSize() {
            return this.size;
        }
    }
}

