(function () {
    'use strict';

    var isInitial = true;
    var metrics = window['browser-metrics'] || { start: function start() {}, end: function end() {} };

    /**
     * Adds a class to elements matching a selector.
     *
     * Why?
     * ====
     *
     * For single-page-app transitions the content elements *already exist* because they were present for the
     * previous page. To work around this we add a "marker" class that indicates the element is "stale",
     * and use a :not(stale) pattern in our ready selector to ensure we don't match those.
     *
     * When the page is actually updated, the stale elements will be replaced with fresh elements that don't
     * have our marker class.
     *
     * @param selector
     * @param className
     */
    function markAsStale(selector, className) {
        var elements = document.querySelectorAll(selector);
        for (var i = 0; i < elements.length; i++) {
            elements[i].classList.add(className);
        }
    }

    var theMode = GH.RapidBoard.ViewController.displayedMode;

    /**
     * Checks whether the issue details panel is shown (or being shown).
     *
     * Why?
     * ===
     *
     * Browser-metrics only allows for one transition to occur at a time.
     * Some of our dynamic behaviours, such as showing the issue details panel
     * (aka jira.agile.issuedetail.show) can happen whilst we are loading
     * any particular transition. As such, it clobbers the original transition.
     *
     * We don't want that to happen.
     *
     * So, we check whether those dynamic behaviours are happening and, if so,
     * manually prevent the issue detail panel's event from clobbering existing
     * transitions.
     */
    function isShowingIssueDetailsPanel() {
        var detailPanelMightOpen = false;
        var anIssueIsSelected = false;
        switch (theMode) {
            case 'plan':
                detailPanelMightOpen = GH.RapidBoard.State.isDetailsViewPlanOpened();
                anIssueIsSelected = GH.BacklogSelectionController.selectionManager && GH.BacklogSelectionController.selectionManager.hasSelection();
                break;
            case 'work':
                detailPanelMightOpen = GH.RapidBoard.State.isDetailsViewOpened();
                anIssueIsSelected = GH.WorkSelectionController.selectionManager && GH.WorkSelectionController.selectionManager.hasSelection();
                break;
        }
        return detailPanelMightOpen && anIssueIsSelected;
    }

    GH.RapidBoard.ViewController.on('before:navigate',
    /**
     * @param {GH.Mode} event.mode The mode of the about-to-load rapid board.
     */
    function (event) {
        var planModeSelector = ".js-marker-backlog-header:not(.browser-metrics-stale)";
        var reportModeSelector = "#ghx-chart-header-primary:not(.browser-metrics-stale)";
        var workModeSelector = "#ghx-work .ghx-issue:not(.browser-metrics-stale), #ghx-work .ghx-sad-columns:not(.browser-metrics-stale)";
        var issueDetailsPanelSelector = "#gh.js-ghx-detail-loaded:not(.browser-metrics-stale)";

        var selectors = [];
        var key;

        theMode = event.mode;

        switch (event.mode) {
            case 'plan':
                key = 'jira.agile.plan' + (isShowingIssueDetailsPanel() ? '-detail' : '');
                selectors.push(planModeSelector);
                if (isShowingIssueDetailsPanel()) {
                    selectors.push(issueDetailsPanelSelector);
                }
                break;

            case 'report':
                key = 'jira.agile.report';
                selectors.push(reportModeSelector);
                break;

            case 'work':
                key = 'jira.agile.work' + (isShowingIssueDetailsPanel() ? '-detail' : '');
                selectors.push(workModeSelector);
                if (isShowingIssueDetailsPanel()) {
                    selectors.push(issueDetailsPanelSelector);
                }
                break;
        }

        if (!isInitial) {
            selectors.forEach(function (selector) {
                markAsStale(selector, 'browser-metrics-stale');
            });
        }

        metrics.start({
            key: key,
            isInitial: isInitial,
            ready: selectors
        });

        // An initial page load can only happen once.
        isInitial = false;
    });

    GH.DetailsView.on('after:render', function () {
        // Manually removing the browser-metrics-stale class from the #gh element,
        // because the element doesn't get destroyed and re-created like others.
        // this relates to the 'issueDetailsPanelSelector' variable above.
        document.querySelector('#gh').classList.remove('browser-metrics-stale');
    });

    /**
     * Sets up browser metrics to time the interval between a start and end event.
     * @param beforeEventEmitter the object which emits the event signalling the timer should be started. Usually an
     * event triggered before the timed operation occurs.
     * @param beforeEventName  the name of the 'before' event the object emits.
     * @param afterEventEmitter the object which emits the event signalling the timer should be stopped. Usually an
     * event triggered after the timed operation occurs.
     * @param afterEventName the name of the 'after' event the object emits.
     * @param metricKey the name to give to the timed event, stored in the browser.metrics.navigation metadata
     */
    function registerMetricListener(beforeEventEmitter, beforeEventName, afterEventEmitter, afterEventName, metricKey) {
        beforeEventEmitter.on(beforeEventName, function () {
            metrics.start({
                key: metricKey,
                isInitial: false
            });
            afterEventEmitter.on(afterEventName, function () {
                metrics.end({ key: metricKey });
            });
        });
    }

    /*
        Drag and drop issue transition.
     */
    registerMetricListener(GH.WorkDragAndDrop, 'before:issueDropRender', GH.WorkController, 'after:render', 'jira.board.work.transition.issue');

    /*
        Toggle quickfilter
     */
    registerMetricListener(GH.Components.QuickFilters.events, 'before:toggle', GH.WorkController, 'after:render', 'jira.agile.quickfilter.toggle');

    /*
        Inline edit on issue details pane.
     */
    registerMetricListener(GH.DetailsFieldEditControl.events, 'before:inlineedit', GH.DetailsFieldEditControl.events, 'after:inlineedit', 'jira.agile.issuedetail.inlineedit');
})();