GADGET = {GH: {}};

GADGET.preInit = function(baseUrl) {
    // gadgets have a slightly different lifecycle/context because they're loaded in an iframe.
    // given that, we need to explicitly trigger the initialisation event when they load.
    var GlobalEvents = require('jira-agile/rapid/global-events');
    GlobalEvents.trigger('pre-initialization');

    // redefine the context path
    GH.Ajax.CONTEXT_PATH = baseUrl;
};

GADGET.isWallboardMode = function() {
    return GH.gadget.utils.getViewName() == 'wallboard';
};

/**
 * Ajax requests to be executed for the descriptor
 */
GADGET.descriptorArgs = function (baseUrl) {
    return [
        {
            key: "rapidViews",
            ajaxOptions: baseUrl + "/rest/greenhopper/1.0/rapidview"
        }
    ];
};

/**
 * Gadget descriptor
 */
GADGET.descriptor = function (gadget, args, baseUrl) {
    //[GHS-2677]
    //Root cause is that the existing content is not cleared when an error is added as a result of a gadget refresh
    //To fix this we hide existing content when this class is appended into the body
    //Calling descriptor should remove class indicating invalidness
    AJS.$("body").removeClass('config-unavailable');

    return  {
        action: baseUrl + "/rest/greenhopper/1.0/gadgets/rapidview/pool/validate.json",
        theme: function () {
            if (gadgets.window.getViewportDimensions().width < 450) {
                return "gdt top-label";
            }
            else {
                return "gdt";
            }
        }(),
        fields: [
            GH.gadget.rapid.fields.rapidViewPicker(gadget, "rapidViewId", _.where(args.rapidViews.views, {sprintSupportEnabled: true})),
            GH.gadget.fields.checkbox(gadget, "showRapidViewName", 'gh.gadget.userpref.showrvname.name', gadget.getPref('showRapidViewName') == 'true'),
            GH.gadget.rapid.fields.sprintPicker(gadget, 'sprintId', 'rapidViewId', true, GH.gadget.rapid.server.loadOpenSprintsByBoardId),
            GH.gadget.fields.checkbox(gadget, "showSprintName", 'gh.gadget.userpref.showspname.name', gadget.getPref('showSprintName') == 'true'),
            GH.gadget.fields.checkbox(gadget, "showAssignees", 'gh.gadget.userpref.showassignees', gadget.getPref('showAssignees') == 'true'),
            AJS.gadget.fields.nowConfigured()
        ]
    };

};

/**
 * Ajax requests for the template
 */
GADGET.templateArgs = function(baseUrl) {
    //Assign wallboard class here so errors also get this class in the body
    if (GADGET.isWallboardMode()) {
        AJS.$('body').addClass('wallboard');
    }
    return [];
};

GADGET.template = function (gadget, args, baseUrl) {
    GADGET.GH.loadAndRender(gadget);
};

(function() {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GADGET.GH.Analytics = new AnalyticsTracker("gh.gadget.sprinthealth");
})();

GADGET.GH.loadAndRender = function(gadget) {

    // load rapid view data (name, etc)
    var loadConfigPromise = GADGET.GH.loadConfig(gadget)
        .fail(function() {
            GH.gadget.utils.showInlineWarning(gadget, gadget.getMsg('gh.gadget.common.error.can.not.find.board.or.nopermission'));
        });

    // all sprints (includes closed sprints!)
    var loadSprintsPromise = GADGET.GH.loadOpenSprints(gadget)
        .fail(function() {
            GH.gadget.utils.showInlineWarning(gadget, gadget.getMsg('gh.gadget.sprint.error.can.not.load.sprints'));
        });

    // continue when we get both results back
    AJS.$.when(loadConfigPromise, loadSprintsPromise).done(function(rapidViewData, sprintsData) {
        var promise = GADGET.GH.loadData(gadget, rapidViewData, sprintsData);
        if (promise) {
            promise.done(function(sprintHealthData) {
                GADGET.GH.render(gadget, rapidViewData, sprintHealthData);
            })
            .fail(function() {
                GH.gadget.utils.showInlineWarning(gadget, gadget.getMsg('gh.gadget.sprint.error.can.not.load.sprint.data'));
            });
        }
    });
};

/**
 * Loads all sprints for the selected board (rapid view).
 *
 * @return {AJS.$.Promise} an AJAX promise of array. Sprint data is available at the first position in the array
 */
GADGET.GH.loadOpenSprints = function (gadget) {
    var rapidViewId = parseInt(gadget.getPref("rapidViewId"), 10);
    return GH.gadget.rapid.server.loadOpenSprintsByBoardId(rapidViewId);
};

/**
 * Loads the rapid view config for the selected board (rapid view).
 *
 * @return {AJS.$.Promise} a promise of rapid view data
 */
GADGET.GH.loadConfig = function(gadget) {
    var rapidViewId = parseInt(gadget.getPref("rapidViewId"), 10);
    return GH.gadget.rapid.server.loadConfigByBoardId(rapidViewId);
};

GADGET.GH.loadData = function(gadget, rapidViewData, openSprints) {
    var rapidViewId = rapidViewData.id;
    var sprintId = GADGET.GH.getSelectedSprint(gadget, openSprints);
    if (!sprintId) {
        GH.gadget.utils.showInlineWarning(gadget, gadget.getMsg('gh.gadget.sprint.error.selected.sprint.not.active'));
    } else {
        return GADGET.GH.loadSprintHealth(rapidViewId, sprintId);
    }
};

GADGET.GH.getSelectedSprint = function(gadget, openSprints) {
    // find the correct sprint
    var sprintId = parseInt(gadget.getPref("sprintId"), 10);
    if (_.isNaN(sprintId)) {
        sprintId  = 'auto';
    }

    var sprint;
    if (sprintId == 'auto') {
        sprint = _.first(openSprints);
    } else {
        sprint = _.findWhere(openSprints, {id: sprintId });
    }

    if (sprint) {
        return sprint.id;
    }
};

GADGET.GH.render = function(gadget, rapidViewData, sprintHealthData) {
    var issueMetrics = {};
    _.each(sprintHealthData.issueMetrics, function(item) {
        issueMetrics[item.key] = item.value;
    });

    var progress = {};
    progress.total = _.reduce(sprintHealthData.progress.columns, function(start, item) {
            return start + item.value;
        }, 0);
    progress.todoWidth = (100 * sprintHealthData.progress.columns[0].value) / progress.total;
    progress.todo = sprintHealthData.progress.columns[0].value;
    progress.inprogressWidth = (100 * sprintHealthData.progress.columns[1].value) / progress.total;
    progress.inprogress = sprintHealthData.progress.columns[1].value;
    progress.doneWidth = (100 * sprintHealthData.progress.columns[2].value) / progress.total;
    progress.done = sprintHealthData.progress.columns[2].value;

    if(sprintHealthData.timeBasedEstimation){
        progress.todo = GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(progress.todo);
        progress.inprogress = GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(progress.inprogress);
        progress.done = GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(progress.done);
    }

    var durationComplete = _.findWhere(sprintHealthData.sprintMetrics, { key: "duration_completion_percentage"}).value;
    var workComplete = _.findWhere(sprintHealthData.sprintMetrics, { key: "work_completion_percentage"}).value;
    var scopeChange = _.findWhere(sprintHealthData.sprintMetrics, { key: "scope_change_percentage"}).value;
    scopeChange = parseInt(scopeChange, 10);

    function createStatusList(statusArray) {
        var statuses = _.map(statusArray, function(status) {
            return "%22" + status + "%22";
        });
        return statuses.join(",");
    }

    var settings = {
        showBoardName: gadget.getPref("showRapidViewName"),
        boardName: rapidViewData.name,
        boardId: rapidViewData.id,
        boardStatistic: rapidViewData.estimationStatistic.name,
        showSprintName: gadget.getPref("showSprintName"),
        issueMetrics: issueMetrics,
        sprintName: sprintHealthData.name,
        sprintId: sprintHealthData.id,
        highestPriority: sprintHealthData.highestPriority,
        assignees: sprintHealthData.assignees,
        showAssignees: gadget.getPref("showAssignees"),
        progress: progress,
        timeremaining: sprintHealthData.timeRemaining,
//        status: sprintHealthData.status.rating,
        durationComplete: durationComplete,
        workComplete: workComplete,
        scopeChange: scopeChange,
        timeTrackingEnabled: sprintHealthData.timeTrackingEnabled,
        noEstimateValues: sprintHealthData.noEstimateValues,
        flaggedField: sprintHealthData.flaggedField,
        todoStatuses: createStatusList(sprintHealthData.statusMap.notStarted),
        inProgressStatuses: createStatusList(sprintHealthData.statusMap.inProgress),
        doneStatuses: createStatusList(sprintHealthData.statusMap.done)
    };

    gadget.getView().html(GH.tpl.gadget.sprintHealth.render(settings));
    gadget.resize();

    GADGET.GH.Analytics.trigger("render", "time elapsed/work complete", durationComplete + "/" + workComplete);
};

GADGET.GH.loadSprintHealth = function(rapidViewId, sprintId) {
    return GH.Ajax.get({
        url: "/gadgets/sprints/health?rapidViewId=" + rapidViewId + "&sprintId=" + sprintId,
        deferErrorHandling: true
    });
};

