/* global AJS, GH, _, gadgets */
var GADGET = {GH: {}};

GADGET.boardInitialized = false;

GADGET.preInit = function (baseUrl) {
    // gadgets have a slightly different lifecycle/context because they're loaded in an iframe.
    // given that, we need to explicitly trigger the initialisation event when they load.
    var GlobalEvents = require('jira-agile/rapid/global-events');
    GlobalEvents.trigger('pre-initialization');

    // redefine the context path
    GH.Ajax.CONTEXT_PATH = baseUrl;
};

GADGET.isWallboardMode = function () {
    return GH.gadget.utils.getViewName() === 'wallboard';
};

/**
 * Ajax requests for the template
 */
GADGET.templateArgs = function (baseUrl) {
    //Assign wallboard class here so errors also get this class in the body
    if (GADGET.isWallboardMode()) {
        AJS.$('body').addClass('wallboard');
    }
    return [];
};

/**
 * Gadget template function
 */
GADGET.template = function (gadget, args, baseUrl) {
    if (!GADGET.boardInitialized) {
        // initialize the actual rendering code
        GADGET.GH.init(gadget);
        GADGET.boardInitialized = true;
    }

    GADGET.GH.loadAndRender(gadget);
};

(function() {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GADGET.GH.Analytics = new AnalyticsTracker("gh.report.burndownChart.gadget");
})();

/**
 * Ajax requests to be executed for the descriptor
 */
GADGET.descriptorArgs = function (baseUrl) {
    return [
        {
            key: "rapidViews",
            ajaxOptions: baseUrl + "/rest/greenhopper/1.0/rapidview"
        }
    ];
};

/**
 * Gadget descriptor
 */
GADGET.descriptor = function (gadget, args, baseUrl) {
    //[GHS-2677]
    //Root cause is that the existing content is not cleared when an error is added as a result of a gadget refresh
    //To fix this we hide existing content when this class is appended into the body
    //Calling descriptor should remove class indicating invalidness
    AJS.$("body").removeClass('config-unavailable');

    return  {
        action: baseUrl + "/rest/greenhopper/1.0/gadgets/rapidview/pool/validate.json",
        theme: (function () {
            if (gadgets.window.getViewportDimensions().width < 450) {
                return "gdt top-label";
            }
            else {
                return "gdt";
            }
        })(),
        fields: [
            GH.gadget.rapid.fields.rapidViewPicker(gadget, "rapidViewId", _.where(args.rapidViews.views, {sprintSupportEnabled: true})),
            GH.gadget.fields.checkbox(gadget, "showRapidViewName", 'gh.gadget.userpref.showrvname.name', gadget.getPref('showRapidViewName') == 'true'),
            GH.gadget.rapid.fields.sprintPicker(gadget, 'sprintId', 'rapidViewId', true, GH.gadget.rapid.server.loadOpenSprintsByBoardId),
            GH.gadget.fields.checkbox(gadget, "showSprintName", 'gh.gadget.userpref.showspname.name', gadget.getPref('showSprintName') == 'true'),
            AJS.gadget.fields.nowConfigured()
        ]
    };

};

GADGET.GH.init = function (gadget) {
    // this is the only way to prevent default error handling (non closable messages) inherited from common gadget library
    gadget.ajaxErrorHandler.handle404 = AJS.$.noop;
};

/**
 * Loads the rapid view config for the selected board (rapid view).
 *
 * @return {AJS.$.Promise} a promise of rapid view data
 */
GADGET.GH.loadConfig = function (gadget) {
    var rapidViewId = parseInt(gadget.getPref("rapidViewId"), 10);
    return GH.gadget.rapid.server.loadConfigByBoardId(rapidViewId);
};

/**
 * Loads all the open sprints for the selected board (rapid view).
 *
 * @return {AJS.$.Promise} sprints array promise.
 */
GADGET.GH.loadOpenSprints = function (gadget) {
    var rapidViewId = parseInt(gadget.getPref("rapidViewId"), 10);
    return GH.gadget.rapid.server.loadOpenSprintsByBoardId(rapidViewId);
};

GADGET.GH.loadAndRender = function (gadget) {

    // clear error message
    GH.Notification.clear();

    gadget.getView().html('<div id="gh"></div>');
    AJS.$('#gh').html(GH.tpl.board.x.renderRapidBoard());
    AJS.$('#ghx-report').html(GH.tpl.reportcontroller.renderReportView({
        shouldShowNavigation: true
    }));

    // load rapid view data (name, etc)
    var loadConfigPromise = GADGET.GH.loadConfig(gadget)
        .done(function (rapidViewData) {
            // all sprints (includes closed sprints!)
            GADGET.GH.loadOpenSprints(gadget).
                done(function (openSprints) {
                    var sprint = GADGET.GH.selectSprint(gadget, rapidViewData, openSprints);

                    // check whether we got sprint
                    if (sprint) {
                        GADGET.GH.showChart(gadget, rapidViewData, sprint);
                    } else {
                        GH.gadget.utils.showInlineWarning(gadget, gadget.getMsg('gh.gadget.sprint.error.can.not.find.sprint'));
                    }
                }).fail(function () {
                    GH.gadget.utils.showInlineWarning(gadget, gadget.getMsg('gh.gadget.sprint.error.can.not.load.sprints'));
                });
        }).fail(function () {
            GH.gadget.utils.showInlineWarning(gadget, gadget.getMsg('gh.gadget.common.error.can.not.find.board.or.nopermission'));
        });
};

/**
 * Selects the correct sprint to visualize, either the defined id or the next sprint in case of auto
 *
 * @return {?*} a sprint or null if no sprint can match the requirements
 */
GADGET.GH.selectSprint = function (gadget, rapidViewData, openSprints) {

    // find the correct sprint
    var sprintId = parseInt(gadget.getPref("sprintId"), 10);
    if (_.isNaN(sprintId)) {
        sprintId = 'auto';
    }

    var sprint;
    if (sprintId === 'auto') {
        // the sprints property is a collection of sprints ordered by natural order of start date then end date
        // it means that the first sprint in that collection is the one that starts the first and ends the first (if needed)
        // for the auto mode we pick the first
        sprint = _.first(openSprints);
    } else {
        sprint = _.findWhere(openSprints, {id: sprintId });
    }

    return sprint;
};

GADGET.GH.showChart = function (gadget, rapidViewData, sprint) {

    AJS.$('#ghx-report').show();

    var BurndownChartView = require('jira-agile/rapid/ui/chart/burndown-chart-view');
    var BurndownReportChartController = require('jira-agile/rapid/ui/chart/burndown-report-chart-controller');

    // configure chart and kick off loading
    BurndownChartView.renderChartLoading();

    var showRapidViewName = gadget.getPref("showRapidViewName") === 'true';
    var showSprintName = gadget.getPref("showSprintName") === 'true';

    // configure the view for the gadget mode
    BurndownChartView.setGadgetMode(true);
    BurndownChartView.setWallboardMode(GADGET.isWallboardMode());

    BurndownReportChartController.setRapidView(rapidViewData.id);
    BurndownReportChartController.setSprintId(sprint.id);
    BurndownReportChartController.setNonWorkingDaysShown(true);
    BurndownReportChartController.show();

    // analytics for chart type
    GADGET.GH.Analytics.trigger('show');

    GADGET.GH.injectTitle(gadget, rapidViewData, showRapidViewName, showSprintName, sprint);

    var ratio = 16 / 9;
    var $content = AJS.$('#ghx-chart-content');
    var $header = AJS.$('#ghx-chart-header');

    var height = $content.width() / ratio;
    // in wallboard mode, we empirically found that it's better to remove the header height
    if (GADGET.isWallboardMode()) {
        height -= $header.height();
    }
    AJS.$('#ghx-chart-view').height(height);

    BurndownReportChartController.loadChartData().done(function() {
        gadget.resize();
    });
};

/**
 * Set the gadget title to include the name of the selected Rapid View and sprint.
 *
 * @param {{name: string}} rapidViewData
 * @param {boolean} showRapidViewName
 * @param {boolean} showSprintName
 * @param {{name: string}} sprint
 */
GADGET.GH.injectTitle = function (gadget, rapidViewData, showRapidViewName, showSprintName, sprint) {
    if (showRapidViewName || showSprintName) {
        GADGET.GH.constructAndInsertGadgetHeader(gadget, rapidViewData, showRapidViewName, showSprintName, sprint);
    }
};

/**
 * Injects a title on top of the chart. We don't use the widget header here because in wallboard mode it will be hidden.
 *
 * @param {{name: string}} rapidViewData
 * @param {boolean} showRapidViewName
 * @param {boolean} showSprintName
 * @param {{name: string}} sprint
 */
GADGET.GH.constructAndInsertGadgetHeader = function (gadget, rapidViewData, showRapidViewName, showSprintName, sprint) {
    AJS.$('#ghx-chart-header')
        .html(GH.tpl.gadgets.renderBasicHeader({
            showTitle: showRapidViewName,
            showSubtitle: showSprintName,
            title: rapidViewData.name,
            subtitle: sprint.name,
            subtitleLabel: gadget.getMsg('gh.gadget.sprint.burndown.sprint')
        }));
};
