/**
 * A brush
 */
c3.brush = function() {

    var dragStart = 0;
    var isDragging = false;

    var brushComponent = c3.singular();

    /**
     * Start dragging.
     *
     *  - Moves the brush to startX and shows it.
     *  - Resets the brush width to 1px.
     *  - Triggers the onDrag callback.
     *
     * @param startX The x coordinate of the mouse click relative to the selection the brush is applied to.
     */
    function startDrag(startX) {
        dragStart = startX;
        isDragging = true;

        show();
        brushComponent.elements()
            .attr('x', startX)
            .attr('y', 0)
            .attr('width', '1');

        brushComponent.onDrag(startX);
    }

    /**
     * Update the brush on (on mousemove).
     *
     *  - Adjusts the width and x position of the brush rect appropriately.
     *  - Recalculates the range that the brush is spanning.
     *  - Triggers the onUpdate callback.
     *
     * @param currentX The current x position of the mouse relative to the selection the brush is applied to.
     */
    function updateBrush(currentX) {
        if (isDragging) {
            var rect = brushComponent.elements();
            var width = parseInt(currentX - dragStart, 10);

            if (width < 0) {
                rect.attr('x', currentX);
            } else {
                rect.attr('x', dragStart);
            }
            rect.attr('width', Math.abs(width));

            var range = _.sortBy([dragStart, currentX], _.identity);

            brushComponent.onUpdate(range);
            brushComponent.currentRange(range);
        }
    }

    /**
     * Finish dragging.
     *
     *  - Hides the brush.
     *  - Triggers the onEndDrag callback.
     *
     * @param currentX The current x position of the mouse relative to the selection the brush is applied to.
     */
    function endDrag(currentX) {
        if (isDragging) {
            isDragging = false;
            hide();
            brushComponent.onEndDrag(_.sortBy([dragStart, currentX], _.identity));
        }
    }

    /**
     * Show the brush by resetting the CSS display value.
     */
    function show() {
        brushComponent.elements()
            .style('display', '');
    }

    /**
     * Hide the brush.
     */
    function hide() {
        brushComponent.elements()
            .style('display', 'none');
    }

    return brushComponent
        .elementTag('rect')
        .extend(c3.withDimensions())
        .enter(function(event) {
            hide();
            event.selection
                .attr('height', this.height())
                .style('pointer-events', 'none');

            var interactionLayer = this.interactiveLayer();
            interactionLayer
                .mousedown('brush', function() {
                    startDrag(interactionLayer.mouseCoordinates()[0]);
                })
                .mousemove('brush', function() {
                    updateBrush(interactionLayer.mouseCoordinates()[0]);
                })
                .mouseup('brush', function() {
                    endDrag(interactionLayer.mouseCoordinates()[0]);
                });
        })
        .extend({
            interactiveLayer: c3.prop(),
            onDrag: c3.event(),
            onUpdate: c3.event(),
            onEndDrag: c3.event(),
            currentRange: c3.prop(),
            cancelDrag: function() {
                isDragging = false;
                hide();
            },
            isDragging: function() {
                return isDragging;
            }
        });
};