/* global c3:true, d3:true */
/**
 * A component mixin that allows a g svg element to be interacted with by the mouse.
 */

c3.interactive = function() {

    var component = c3.component();

    var cachedOffset,
        isCurrentlyWithinBounds = false;

    /**
     * Update the cached offset at most once every 200ms.
     */
    var updateCachedOffset = _.throttle(function() {
        cachedOffset = AJS.$(component.region().selection().node()).offset();
    }, 200);

    /**
     * Test whether or not a mouse event occurred within the bounds
     *
     * @param e
     * @returns {boolean}
     */
    function isWithinBounds(e) {
        var coordinates = getRelativeMouseCoordinates(e);
        var x = coordinates[0];
        var y = coordinates[1];

        component.mouseCoordinates([x, y]);

        return (x > 0 && y > 0) && (x <= component.width() && y <= component.height());
    }

    /**
     * Calculate the coordinates at which a mouse event occured, relative to the component's selection.
     *
     * @param {jQuery.event} e the mouse event
     * @returns {[number, number]}
     */
    function getRelativeMouseCoordinates(e) {

        updateCachedOffset();

        var x = (e.pageX - cachedOffset.left);
        var y = (e.pageY - cachedOffset.top);

        return [x, y];
    }

    /**
     * Prevents default on the dragstart event to fix Firefox glitches.
     */
    function overrideDragStart() {
        AJS.$(component.selection().node()).on('dragstart', function(e) {
            e.preventDefault();
        });
    }

    /**
     * Initialise the component:
     *
     *  - Binds to the mousemove event of the domContext to determine when the mouseout/mouseover events occur
     *  - Prevent native behaviour of dragstart event to fix Firefox glitches
     */
    function initialise() {
        component.domContext()
            .on('mousemove', function(e) {
                var isMouseWithinBounds = isWithinBounds(e);
                if (!isMouseWithinBounds && isCurrentlyWithinBounds) {
                    isCurrentlyWithinBounds = false;
                    component.mouseout(e);
                } else if (isMouseWithinBounds && !isCurrentlyWithinBounds) {
                    isCurrentlyWithinBounds = true;
                    component.mouseover(e);
                }
            })
            .on('mouseout', function(e) {
                if (isCurrentlyWithinBounds) {
                    isCurrentlyWithinBounds = false;
                    component.mouseout(e);
                }
            });

        overrideDragStart();
    }

    /**
     * Creates a function wrapped with a check to see if the event occurs within the bounds of this selection.
     *
     * @param eventName the event to bind to
     * @returns {Function}
     */
    function wrapBoundedEvent(eventName) {
        return function(id, handler) {
            component.domContext().on(eventName + '.c3.' + id, function(e) {
                if (isWithinBounds(e)) {
                    d3.event = e;
                    handler.call(this, e);
                }
            });
            return this;
        };
    }

    return component
        .extend(c3.withDimensions())
        .extend(_.once(initialise))
        .extend({
            mouseCoordinates: c3.prop(),
            domContext: c3.prop(),
            region: c3.prop(),
            mousedown: wrapBoundedEvent('mousedown'),
            mouseup: wrapBoundedEvent('mouseup'),
            mousemove: wrapBoundedEvent('mousemove'),
            mouseover: c3.event(),
            mouseout: c3.event()
        });
};