AJS.test.require(["com.pyxis.greenhopper.jira:gh-rapid-common", "com.pyxis.greenhopper.jira:gh-test-common"], function () {
    var Ajax = require('jira-agile/rapid/ajax');

    module('GH.Ajax', {
        setup: function setup() {
            this.server = sinon.fakeServer.create();
            this.clock = sinon.useFakeTimers(new Date().getTime()); // use now
            Ajax.xhrRegistery = {};
            this.oldStallTimeout = Ajax.stallTimeout;
            Ajax.stallTimeout = -1;
            sinon.stub(Ajax, 'handleContextualErrors');
            sinon.stub(Ajax, 'handleGlobalErrors');
            sinon.stub(Ajax, 'matchesCurrentUser').returns(true);
        },
        teardown: function teardown() {
            this.server.restore();
            this.clock.restore();
            Ajax.stallTimeout = this.oldStallTimeout;
            Ajax.handleContextualErrors.restore();
            Ajax.handleGlobalErrors.restore();
            Ajax.matchesCurrentUser.restore();
        }
    });

    test('get with done promise', function () {
        var onSuccess, onError, promise;
        onSuccess = sinon.spy();
        onError = sinon.spy();
        promise = Ajax.get({}, 'testResource').done(onSuccess);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 1, 'only one request should have been sent');
        this.server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify({ id: 1 }));
        strictEqual(onError.callCount, 0, 'Error callback should never be called');
        strictEqual(onSuccess.callCount, 1, 'Success callback should be called once');
        deepEqual(onSuccess.getCall(0).args[0], { id: 1 }, 'data should be as expected');
    });

    test('restful get with done promise', function () {
        var onSuccess, onError, promise;
        onSuccess = sinon.spy();
        onError = sinon.spy();
        promise = Ajax.get({}, 'testResource').done(onSuccess);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 1, 'only one request should have been sent');
        this.server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify({ id: 1, error: {} }));
        strictEqual(onError.callCount, 0, 'Error callback should never be called');
        strictEqual(onSuccess.callCount, 1, 'Success callback should be called once');
        deepEqual(onSuccess.getCall(0).args[0], { id: 1, error: {} }, 'data should be as expected');
    });

    test('get with fail promise', function () {
        var onSuccess, onError, promise;
        onSuccess = sinon.spy();
        onError = sinon.spy();
        promise = Ajax.get({ success: onSuccess }, 'testResource').fail(onError);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 1, 'only one request should have been sent');
        this.server.requests[0].respond(500, { "Content-Type": "application/json" }, JSON.stringify({}));
        strictEqual(onSuccess.callCount, 0, 'Success callback should never be called');
        strictEqual(onError.callCount, 1, 'Error callback should be called once');
        ok(!onError.getCall(0).args[0].isSuccess(), 'data should be as expected');
    });

    test('restful get with fail promise', function () {
        var onSuccess, onError, promise;
        onSuccess = sinon.spy();
        onError = sinon.spy();
        promise = Ajax.get({}, 'testResource').fail(onError);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 1, 'only one request should have been sent');
        this.server.requests[0].respond(404, { "Content-Type": "application/json" }, JSON.stringify({}));
        strictEqual(onSuccess.callCount, 0, 'Success callback should never be called');
        strictEqual(onError.callCount, 1, 'Error callback should be called once');
        ok(!onError.getCall(0).args[0].isSuccess(), 'data should be as expected');
    });

    test('stalling get', function () {
        Ajax.stallTimeout = 50;
        var onSuccess, onError, promise;
        onSuccess = sinon.spy();
        onError = sinon.spy();
        promise = Ajax.get({}, 'testResource');
        ok(promise, 'promise should be returned');

        promise.done(onSuccess).fail(onError);

        strictEqual(this.server.requests.length, 1, 'only one request should have been sent');

        promise = Ajax.get({}, 'testResource');
        ok(promise, 'promise should be returned');

        strictEqual(this.server.requests.length, 1, 'No new request should have been sent');

        this.clock.tick(49);
        strictEqual(this.server.requests.length, 1, 'No new request should have been sent');

        promise = Ajax.get({}, 'testResource');
        ok(promise, 'promise should be returned');

        strictEqual(this.server.requests.length, 1, 'No new request should have been sent');

        this.clock.tick(49);
        strictEqual(this.server.requests.length, 1, 'No new request should have been sent');

        this.clock.tick(1);
        strictEqual(this.server.requests.length, 2, 'No new request should have been sent');

        strictEqual(onSuccess.callCount, 0, 'Success callback should never be called');
        strictEqual(onError.callCount, 0, 'Error callback should never be called');
    });

    test("getSettings should add the REST prefix if it's not there already", function () {

        var settings = Ajax._getSettings({ url: "/welcome" });

        equal(settings.url, Ajax.buildRestUrl("/welcome"));
    });

    test("getSettings should not add the REST prefix if we supply a bare url", function () {
        var ghRestUrl = Ajax.buildRestUrl("/welcome");
        var bareRestUrl = Ajax.buildBareRestUrl("/welcome");

        var settings = Ajax._getSettings({ bareUrl: ghRestUrl });
        equal(settings.url, ghRestUrl);

        settings = Ajax._getSettings({ bareUrl: bareRestUrl });
        equal(settings.url, bareRestUrl);
    });

    asyncTest("timed out makeRequest should reject result promise", function () {
        expect(2);

        var deferred = Ajax.makeRequest({ url: "/abortXhr", timeout: 60000 });
        deferred.fail(function () {
            start();
            ok(true, "Deferred rejected");
        });

        ok(this.server.requests[0], "There is some request in queue");

        this.clock.tick(60000);
    });
});