define("jira-agile/rapid/durations-timer", ['jira-agile/rapid/events'], function (Events) {
    "use strict";

    var windowActivityNotifier = GH.windowActivityNotifier;

    /**
     * A timer that triggers a 'tick' event at the specified durations after the DurationsTimer is constructed.
     *
     * For example, the following DurationsTimer will tick 3 times, at 1, 2 and 3 seconds:
     *    var timer = new DurationsTimer([1000, 2000, 3000]);
     *    timer.on('tick', doSomething);
     *
     * The timer is paused when the window loses focus, and unpaused when it gains focus again.
     *
     * @param {number[]} durations in milliseconds. Must be in ascending order.
     * @constructor
     */
    function DurationsTimer(durations) {
        Events(this);

        this._durations = durations.slice(); // Clone the array since we mutate it
        this._accumulatedDuration = 0;

        if (durations.length) {
            windowActivityNotifier.on('focus', this.start, this).on('blur', this.pause, this).on('unload', this.pause, this);

            this.start();
        }
    }

    DurationsTimer.prototype = {
        start: function start() {
            if (this._timeout) {
                // Already started
                return this;
            }

            var nextInterval = this._durations[0] - this._accumulatedDuration;
            if (nextInterval < 0) {
                this._tick();
                return this;
            }

            var timer = this;
            this._timeoutStart = new Date();
            this._timeout = setTimeout(function () {
                timer._clearTimeout();
                timer._accumulatedDuration += new Date() - timer._timeoutStart;
                timer._tick();
            }, nextInterval);
            return this;
        },
        pause: function pause() {
            if (this._timeout) {
                this._clearTimeout();
                this._accumulatedDuration += new Date() - this._timeoutStart;
            }
            return this;
        },
        destroy: function destroy() {
            this._clearTimeout();
            windowActivityNotifier.off(this);
            return this;
        },
        _tick: function _tick() {
            this.trigger('tick', this._durations[0]);
            this._durations.shift();
            if (this._durations.length) {
                this.start();
            } else {
                this.destroy();
            }
        },
        _clearTimeout: function _clearTimeout() {
            if (this._timeout) {
                clearTimeout(this._timeout);
                this._timeout = null;
            }
        }
    };

    return DurationsTimer;
});

AJS.namespace('GH.DurationsTimer', null, require("jira-agile/rapid/durations-timer"));