define('jira-agile/rapid/events', ['require'], function (require) {
    var _ = require('underscore');

    /**
     * A generic event aggregator.
     *
     * Constructing:
     *   new Events(); // a new event aggregator
     *   Events(obj);  // add Events functions onto an existing object
     *
     * Triggering events:
     *   var events = new Events();
     *   events.trigger('change');
     *   events.trigger('change', before, after); // Any number of arguments can be passed with the event
     *
     * Simple way of binding to events:
     *   events.on('change', function(before, after) {
     *       // respond to change event
     *   });
     *
     * Recommended way of binding to events, by setting the object that is listening as the context:
     *   events.on('change', this.respond, this);
     *
     * Subsequent bindings to the same event and context will *replace* the previous binding:
     *   events.on('change', this.respondV2, this); // this.respond will no longer be called
     *
     * By setting the context, cleanup is also easy if the listener needs to be destroyed:
     *   events.off(this);
     *
     * @param {*} [obj]
     * @returns {*}
     * @constructor
     */
    function Events(obj) {
        if (obj) {
            _.extend(obj, Events.prototype);
        } else {
            obj = this;
        }
        obj._bindings = [];
        return obj;
    }

    Events.prototype = {
        /**
         * Binds an event handler for the specified event type.
         * If a context is included, any previous handler of the same type and context will be unbound.
         *
         * @param {string} type
         * @param {function} handler
         * @param {*} [context] used as the context when the handler is called
         * @returns {Events}
         */
        on: function on(type, handler, context) {
            if (context) {
                this._off({
                    type: type,
                    context: context
                });
            }
            this._bindings.push({
                type: type,
                handler: handler,
                context: context
            });
            return this;
        },

        /**
         * Unbinds matching event bindings.
         * If the type is a string, all bindings for that event type will be unbound.
         * If the type is a function, all bindings for that handler will be unbound.
         * If the type is an Object, all bindings for that context will be unbound.
         *
         * @param {string|function|Object} type
         * @returns {Events}
         */
        off: function off(type) {
            if (_.isString(type)) {
                this._off({
                    type: type
                });
            } else if (_.isFunction(type)) {
                this._off({
                    handler: type
                });
            } else if (_.isObject(type)) {
                this._off({
                    context: type
                });
            }
            return this;
        },

        _off: function _off(descriptor) {
            this._bindings = _.reject(this._bindings, function (binding) {
                var eventTypeMatches = !descriptor.type || descriptor.type === binding.type;
                var namespaceMatches = !descriptor.context || descriptor.context === binding.context;
                var handlerMatches = !descriptor.handler || descriptor.handler === binding.handler;
                return eventTypeMatches && namespaceMatches && handlerMatches;
            });
        },

        /**
         * Trigger an event of the specified type.
         *
         * @param {string} type
         * @param {...*} eventArgs
         * @returns {Events}
         */
        trigger: function trigger(type) {
            var eventArgs = Array.prototype.slice.call(arguments, 1);
            _.each(this._bindings, function (binding) {
                if (binding.type === type) {
                    binding.handler.apply(binding.context, eventArgs);
                }
            });
            return this;
        }
    };

    return Events;
});

AJS.namespace('GH.Events', null, require('jira-agile/rapid/events'));