define("jira-agile/rapid/time-format", ["jira/util/formatter", "underscore", "jira-agile/rapid/time-format-duration-units", "jira-agile/rapid/time-format-displays"], function (Formatter, _, TimeFormatDurationUnits, TimeFormatDisplays) {
    "use strict";

    var TimeFormat = {};

    /**
     * Default options for formatting durations which are suited to JIRA's Time Tracking Administration configuration values.
     * This is overriden by GH.RapidViewConfig.handleData to get the actual configuration.
     */
    TimeFormat.timeTrackingConfigOpts = {
        hoursPerDay: 8,
        daysPerWeek: 5,
        timeFormat: TimeFormatDisplays.PRETTY,
        symbols: {
            'week': 'w',
            'day': 'd',
            'hour': 'h',
            'minute': 'm'
        }
    };

    /**
     * Default options for formatting durations which are used in "normal display" situations e.g. charts that do not specifically
     * related to Time Tracking fields.
     */
    TimeFormat.normalDisplayOpts = {
        hoursPerDay: 24,
        daysPerWeek: 7,
        timeFormat: TimeFormatDisplays.PRETTY,
        symbols: {
            'week': 'w',
            'day': Formatter.I18n.getText('core.durationutils.unit.day'),
            'hour': Formatter.I18n.getText('core.durationutils.unit.hour'),
            'minute': Formatter.I18n.getText('core.durationutils.unit.minute')
        }
    };

    TimeFormat.formatMinuteDuration = function (duration) {
        return TimeFormat.formatDuration(duration ? duration * 60000 : 0);
    };

    TimeFormat.formatShortMinuteDuration = function (duration) {
        return TimeFormat.formatShortDuration(duration ? duration * 60000 : 0);
    };

    /**
     * Formats an duration in days, hours, minutes with 1 day = 24 hours.
     * @param duration specified in milliseconds
     */
    TimeFormat.formatDuration = function (duration) {
        // ensure we don't format undefined
        if (!duration) {
            duration = 0;
        }

        var days = Math.floor(duration / 86400000); // (1000*60*60*24)
        var remaining = duration - days * 86400000;
        var hours = Math.floor(remaining / 3600000); // (1000*60*60)
        remaining = remaining - hours * 3600000;
        var minutes = Math.floor(remaining / 60000); // (1000*60)

        var text = '';
        if (days > 0) {
            text += days + ' ';
            if (days > 1) {
                text += AJS.format(Formatter.I18n.getText('core.dateutils.days'), days);
            } else {
                text += AJS.format(Formatter.I18n.getText('core.dateutils.day'), days);
            }
        }

        if (hours > 0) {
            if (text.length > 0) {
                text += ' ';
            }
            text += hours + ' ';
            if (hours > 1) {
                text += AJS.format(Formatter.I18n.getText('core.dateutils.hours'), hours);
            } else {
                text += AJS.format(Formatter.I18n.getText('core.dateutils.hour'), hours);
            }
        }

        if (duration > 0 || text.length === 0) {
            // show 0
            if (text.length > 0) {
                text += ' ';
            }
            text += minutes + ' ';
            if (minutes === 0 || minutes > 1) {
                text += AJS.format(Formatter.I18n.getText('core.dateutils.minutes'), minutes);
            } else {
                text += AJS.format(Formatter.I18n.getText('core.dateutils.minute'), minutes);
            }
        }
        return text;
    };

    /**
     * Formats an duration which is being used by Time Tracking fields in short 'pretty' format.
     * @param duration duration specified in <b>seconds</b>
     */
    TimeFormat.formatShortDurationForTimeTrackingConfiguration = function (duration, options) {
        if (_.isUndefined(options)) {
            options = {
                durationUnit: TimeFormatDurationUnits.SECONDS
            };
        }
        return TimeFormat._formatShortDuration(duration, _.extend({}, TimeFormat.timeTrackingConfigOpts, options));
    };

    /**
     * Formats an duration which is being used in "normal display" situations in short 'pretty' format.
     * @param duration duration specified in <b>milliseconds</b>
     */
    TimeFormat.formatShortDuration = function (duration) {
        return TimeFormat._formatShortDuration(duration, _.extend({}, TimeFormat.normalDisplayOpts, {
            durationUnit: TimeFormatDurationUnits.MILLISECONDS
        }));
    };

    /**
     * Formats an duration which is being used in "normal display" situations in short 'pretty' format.
     * Minutes are not shown if the duration is longer than a day
     * @param duration duration specified in <b>milliseconds</b>
     */
    TimeFormat.formatSensibleShortDuration = function (duration) {
        var formattedShortDuration = TimeFormat._formatShortDuration(duration, _.extend({}, TimeFormat.normalDisplayOpts, {
            durationUnit: TimeFormatDurationUnits.MILLISECONDS,
            dropMinutesIfLongerThanADay: true
        }));
        if (formattedShortDuration.length === 0) {
            formattedShortDuration = TimeFormat.getZeroMinutesShortFormat();
        }
        return formattedShortDuration;
    };

    /**
     * Returns a time format in short format for 0 minutes
     */
    TimeFormat.getZeroMinutesShortFormat = function () {
        return Formatter.I18n.getText('core.durationutils.unit.minute', 0);
    };

    /**
     * Does the hard work of formatting a short duration, based on the options provided.
     */
    TimeFormat._formatShortDuration = function (duration, opts) {
        // ensure we don't format undefined
        if (!duration) {
            duration = 0;
        }

        if (duration === 0) {
            return "0";
        }

        // make sure we have all the options
        var defaultOpts = TimeFormat.normalDisplayOpts;
        opts = !_.isUndefined(opts) ? opts : {};
        opts = _.extend({}, defaultOpts, opts);

        var weeks = void 0;
        var days = void 0;
        var hours = void 0;
        var minutes = void 0;
        var remaining = void 0;
        var MILLIS_PER_MINUTE = 60000; // (1000*60)
        var MILLIS_PER_HOUR = 60 * MILLIS_PER_MINUTE; // (1000*60*60)
        var MILLIS_PER_DAY = opts.hoursPerDay * MILLIS_PER_HOUR;
        var MILLIS_PER_WEEK = opts.daysPerWeek * MILLIS_PER_DAY;

        // if duration is specified in SECONDS, bump it up to MILLISECONDS to make calculations consistent
        if (opts.durationUnit === TimeFormatDurationUnits.SECONDS) {
            duration *= 1000;
        }

        // break down the duration into units
        weeks = Math.floor(duration / MILLIS_PER_WEEK);
        remaining = duration % MILLIS_PER_WEEK;
        days = Math.floor(remaining / MILLIS_PER_DAY);
        remaining = remaining % MILLIS_PER_DAY;
        hours = Math.floor(remaining / MILLIS_PER_HOUR);
        remaining = remaining % MILLIS_PER_HOUR;
        minutes = Math.floor(remaining / MILLIS_PER_MINUTE);
        if (opts.dropMinutesIfLongerThanADay && days > 0) {
            minutes = 0;
        }

        var amounts = [];

        if (opts.timeFormat === TimeFormatDisplays.PRETTY) {
            // display all units
            amounts = [{ val: weeks, fmt: opts.symbols.week }, { val: days, fmt: opts.symbols.day }, { val: hours, fmt: opts.symbols.hour }, { val: minutes, fmt: opts.symbols.minute }];
        } else if (opts.timeFormat === TimeFormatDisplays.DAYS) {
            // redefine days in terms of weeks and days
            days = days + weeks * opts.daysPerWeek;

            // redefine hours in terms of hours and minutes
            hours = hours + minutes / 60;

            // only display days and hours units
            amounts = [{ val: days, fmt: opts.symbols.day }, { val: hours, fmt: opts.symbols.hour }];
        } else if (opts.timeFormat === TimeFormatDisplays.HOURS) {
            // redefine hours in terms of weeks, days, hours and minutes
            days = days + weeks * opts.daysPerWeek;
            hours = days * opts.hoursPerDay + hours + minutes / 60;

            // only display hours unit
            amounts = [{ val: hours, fmt: opts.symbols.hour }];
        } else if (opts.timeFormat === TimeFormatDisplays.MOST_SIGNIFICANT) {
            if (weeks > 0) {
                amounts = [{ val: GH.Util.roundToDecimalPlaces(duration / MILLIS_PER_WEEK, 1), fmt: opts.symbols.week }];
            } else if (days > 0) {
                amounts = [{ val: GH.Util.roundToDecimalPlaces(duration / MILLIS_PER_DAY, 1), fmt: opts.symbols.day }];
            } else if (hours > 0) {
                amounts = [{ val: GH.Util.roundToDecimalPlaces(duration / MILLIS_PER_HOUR, 1), fmt: opts.symbols.hour }];
            } else {
                amounts = [{
                    val: minutes < 1 ? '' : GH.Util.roundToDecimalPlaces(duration / MILLIS_PER_MINUTE, 1),
                    fmt: opts.symbols.minute
                }];
            }
        }

        // define a function for iteratively building a string of amounts in different units
        var buildStringFromAmounts = function buildStringFromAmounts(amounts) {
            var text = '';
            _.each(amounts, function (amount) {
                if (amount.val > 0) {
                    if (text.length > 0) {
                        text += ' ';
                    }
                    var fmt = amount.fmt;
                    if (fmt.indexOf('{0}') < 0) {
                        fmt = '{0}' + fmt;
                    }
                    var formattedValue = GH.NumberFormat.format(amount.val);
                    text += AJS.format(fmt, formattedValue);
                }
            });

            // Durations less than the lowest time unit displayed will have an empty text at this point.
            // Find the lowest unit and display "< 1" of that unit.
            if (!text && amounts.length) {
                text = AJS.format(_.last(amounts).fmt, '< 1');
            }

            return text;
        };

        return buildStringFromAmounts(amounts);
    };

    return TimeFormat;
});

AJS.namespace("GH.TimeFormat", null, require("jira-agile/rapid/time-format"));