/**
 * Quick Filters configuration tab
 */
define('jira-agile/rapid/configuration/quick-filter-config', [
    "jira-agile/rapid/ajax",
    "jira/util/formatter",
    "aui/restful-table",
    "underscore",
    "jquery",
    "jira/util/strings",
    "jira-agile/rapid/analytics-tracker"
], function(Ajax, formatter, RestfulTable, _, $, Strings, AnalyticsTracker) {
    "use strict";

    const TableColumns = [
        {
            id        : 'name',
            header    : formatter.I18n.getText('gh.rapid.config.quickfilters.label.name'),
            fieldName : 'name'
        },
        {
            id        : 'query',
            header    : formatter.I18n.getText('gh.rapid.config.quickfilters.label.jql'),
            fieldName : 'query',
            editView  : GH.RapidBoard.Config.Util.JQLEditView.extend({ namespace : 'ghx-quick-filter'})
        },
        {
            id        : 'description',
            header    : formatter.I18n.getText('gh.rapid.config.quickfilters.label.description'),
            fieldName : 'description',
            editView  : GH.RapidBoard.Config.Util.TextAreaEditView
        }
    ];

    const QuickFilter = {};

    QuickFilter.Analytics = {
        base: new AnalyticsTracker("gh.quickfilter"),
        add: new AnalyticsTracker("gh.quickfilter", "create"),
        edit: new AnalyticsTracker("gh.quickfilter", "edit"),
        remove: new AnalyticsTracker("gh.quickfilter", "delete")
    };

    /**
     * Initializes the Quick Filters config.
     */
    QuickFilter.init = function(data) {
        QuickFilter.model = data.quickFilterConfig;

        QuickFilter.initializeTable();
    };

    /**
     * Renders the edit form using the locally stored model
     */
    QuickFilter.initializeTable = function() {
        var model = QuickFilter.model;
        var canEdit = model.canEdit;
        var rapidViewId = model.rapidViewId;

        $('#ghx-config-quickFilters').html(GH.tpl.quickfilterconfig.renderQuickFilterConfig());

        // setup the restful table
        var restfulTable = new RestfulTable({
            el: $("#ghx-quick-filter-table"), // table to add entries to. Entries are appended to the tables <tbody> element
            resources: {
                all: function(next) {
                    next(QuickFilter.model.quickFilters);
                },
                self: Ajax.buildRestUrl('/quickfilters/' + rapidViewId + '/')
            },
            columns: TableColumns,
            deleteConfirmation: function(model) {
                return GH.tpl.rapid.view.deleteForm({
                    message: formatter.I18n.getText('gh.rapid.quickfilter.delete.confirm', Strings.escapeHtml(String(model.name)))
                });
            },
            views: {
                editRow: RestfulTable.EditRow.extend({
                    submit: function () {
                        var values = this.$el.serializeObject();
                        var oldJql = this.model.attributes.query;
                        var newJql = values && values.query ? values.query : "";
                        if(this.model.id)
                        {
                            var analytics = "rapidViewId: " + rapidViewId +
                                ", oldJql: '" + oldJql + "'" +
                                ", newJql: '" + newJql + "'";
                            QuickFilter.Analytics.edit.trigger(analytics);
                        } else {
                            // else we have a create event
                            QuickFilter.Analytics.add.trigger("rapidViewId: " + GH.RapidBoard.Config.editedRapidViewId + ", jql: '" + newJql + "'");
                        }

                        return RestfulTable.EditRow.prototype.submit.apply(this, arguments);
                    }
                })
            },
            allowEdit: canEdit,
            allowDelete: canEdit,
            allowReorder: canEdit,
            allowCreate: canEdit
        });

        if (model.canEdit) {
            // register jql auto complete
            // Delay the initialization until after the RestfulTable has rendered. This allows the heights to be calculated
            // correctly and the autocomplete suggestion box to be aligned correctly
            _.delay(GH.JQLAutoComplete.initialize, 0, 'ghx-quick-filter-jql-create', 'ghx-quick-filter-jql-errormsg-create');

            // initializing the autocomplete when the field is hidden results in incorrect dropdown positioning
            $(document).bind("tabSelect", function() {
                GH.JQLAutoComplete.fixSuggestionPositioning('ghx-quick-filter-jql-create');
            });

            $(restfulTable).bind(RestfulTable.Events.ROW_REMOVED, function(evt, row) {
                var analytics = "rapidViewId: " + rapidViewId +
                    ", jql: '" + row.model.attributes.query + "'";
                QuickFilter.Analytics.remove.trigger(analytics);
            });
        }
    };

    return QuickFilter;
});

AJS.namespace('GH.RapidBoard.Config.QuickFilter', null, require('jira-agile/rapid/configuration/quick-filter-config'));
