/**
 * Note: this is a copy of
 * https://bitbucket.org/shghs/aui/src/612a10e756ce/src/storage.js.
 *
 * Slightly adapted to allow using either local or session storage, defaulting to local storage if not provided
 * Also added a json parse exception handler to gracefully handle invalid data stored in the storage
 */

// require AUI
// provide AUI.storage

define("jira-agile/rapid/state/storage", [], function () {
    "use strict";

    var storage = {};

    /**
     * @param {string} key
     * @returns {Object|string|number|boolean}
     */
    storage.get = function (key, useSessionStorage) {
        var value = void 0;
        if (useSessionStorage) {
            value = sessionStorage.getItem(key);
        } else {
            value = localStorage.getItem(key);
        }
        if (typeof value === "string") {
            // Ensure we catch json parse exceptions as value could be a string containing invalid json
            try {
                return JSON.parse(value);
            } catch (err) {
                GH.log('GH.storage: Caught error: ' + err);
                GH.log('GH.storage: Cannot parse json string retrieved from ' + (useSessionStorage ? 'session' : 'local') + ' storage. Ignoring value ' + value);
                return undefined;
            }
        } else {
            return value;
        }
    };

    /**
     * @param {string} key
     * @param {Object|string|number|boolean} value
     * @returns {boolean}
     */
    storage.put = function (key, value, useSessionStorage) {
        if (value === null || value === undefined) {
            if (useSessionStorage) {
                sessionStorage.removeItem(key);
            } else {
                localStorage.removeItem(key);
            }
        } else {
            try {
                if (useSessionStorage) {
                    sessionStorage.setItem(key, JSON.stringify(value));
                } else {
                    localStorage.setItem(key, JSON.stringify(value));
                }
            } catch (QUOTA_EXCEEDED_ERR) {
                return false;
            }
        }
        return true;
    };

    if (!window.localStorage || !window.sessionStorage) {
        // When localStorage is unavailable, get and put methods always fail.
        storage.get = function () {
            return null;
        };
        storage.put = function () {
            return false;
        };
    }

    return storage;
});

AJS.namespace("GH.storage", null, require("jira-agile/rapid/state/storage"));