/**
 * Burndown Rate
 * @module jira-agile/rapid/ui/chart/burndown-rate
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/chart/burndown-rate', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');

    /**
     * Provides the burndown rates for a given date range. These rates are used to draw the guideline taking off time correctly into account
     */
    var BurndownRate = {};

    BurndownRate.WORKING_DAY = 1;
    BurndownRate.NON_WORKING_DAY = 0;

    BurndownRate.setRateData = function (rateData) {
        BurndownRate.rateData = rateData;
    };

    /**
     * Calculates the statistics table data displayed below the chart
     */
    BurndownRate.getRateDefinitions = function () {
        return BurndownRate.rateData.rates;
    };

    BurndownRate.getNonWorkingDays = function () {
        var rates = BurndownRate.getRateDefinitions();
        return BurndownRate.limitToNonWorkingDays(rates);
    };

    BurndownRate.getWorkingDays = function () {
        var rates = BurndownRate.getRateDefinitions();
        return BurndownRate.limitToWorkingDays(rates);
    };

    BurndownRate.limitToWorkingDays = function (rates) {
        return _.filter(rates, function (block) {
            return block.rate === BurndownRate.WORKING_DAY;
        });
    };

    BurndownRate.limitToNonWorkingDays = function (rates) {
        return _.filter(rates, function (block) {
            return block.rate === BurndownRate.NON_WORKING_DAY;
        });
    };

    BurndownRate.limitToTimeRange = function (rates, startTime, endTime) {
        // remove out of range rate elements
        var onlyInRange = _.reject(rates, function (rate) {
            return endTime < rate.start || rate.end < startTime;
        });

        // ensure start and end are inside the range
        return _.map(onlyInRange, function (rate) {
            return {
                start: Math.max(startTime, rate.start),
                rate: rate.rate,
                end: Math.min(endTime, rate.end)
            };
        });
    };

    BurndownRate.isDateInNonWorkingBlock = function (date) {
        var isNonWorkingDate = false;
        var nonWorkingDays = BurndownRate.getNonWorkingDays();
        for (var i = 0; i < nonWorkingDays.length; i++) {
            var block = nonWorkingDays[i];
            if (block.start <= date && block.end > date) {
                isNonWorkingDate = true;
                break;
            }
        }
        return isNonWorkingDate;
    };

    BurndownRate.getAdjustedNonWorkingDays = function (startDate) {
        var nonWorkingDays = BurndownRate.getNonWorkingDays();

        var relevantNonWorkingDays = _.filter(nonWorkingDays, function (block) {
            return block.end > startDate;
        });

        return _.map(relevantNonWorkingDays, function (block) {
            return {
                moveTo: block.start === startDate ? block.end : block.start,
                start: block.start,
                end: block.end
            };
        });
    };

    return BurndownRate;
});