/* global GH */
/**
 * Burndown Table Producer
 * @module jira-agile/rapid/ui/chart/burndown-table-producer
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/chart/burndown-chart-model
 * @requires module:jira-agile/rapid/ui/chart/burndown-timeline-producer
 */
define('jira-agile/rapid/ui/chart/burndown-table-producer', ['require'], function(require) {
    'use strict';

    var _ = require('underscore');
    var BurndownChartModel = require('jira-agile/rapid/ui/chart/burndown-chart-model');
    var BurndownTimelineProducer = require('jira-agile/rapid/ui/chart/burndown-timeline-producer');

    // GLOBALS... FIX ME
    var ChartUtils = GH.ChartUtils;
    var tpl = GH.tpl;

    /**
     * Provides the data table displayed alongside the burndown.
     *
     * Note, each statistic has a different TableHandler implementation that knows how to generate change events for the given
     * timeline
     */
    var BurndownTableProducer = {};

    /**
     * Calculates the statistics table data displayed below the chart
     */
    BurndownTableProducer.getDataTable = function(timelineData, tableHandler, statisticConsumer, lastUserWhoClosedHtml) {
        // put together a data carrier object
        var data = {
            // holds the rows of the statistics table
            rows: [],

            // holds past rows that contain estimate changes. key: '{issueKey}-{changeDate}'.
            // We have to add notes about later recording only where we show a backdated message
            pastChangeRows: {},

            // holds the current values as we go along the timeline (updated with the deltas)
            currentValues: []
        };


        // Go through the timeline, add one row per entry/issue, plus one for start and complete
        _.each(timelineData.timeline, function(timelineEntry) {

            // start sprint entry
            if (timelineEntry.initial) {
                // the first row is the sprint start values - we show all values in a single line
                var firstRow = {
                    initial: true,
                    time: timelineEntry.time,
                    event: BurndownTableProducer.getSprintStartChangeEvent(),
                    values: timelineEntry.values,
                    issues: timelineEntry.issues
                };
                data.rows.push(firstRow);

                // initial values we adapt on each change
                data.currentValues = _.clone(timelineEntry.values);
            }

            // complete sprint entry
            else if (timelineEntry.complete) {
                var lastRow = {
                    complete: true,
                    time: timelineEntry.time,
                    event: BurndownTableProducer.getSprintCompleteChangeEvent(lastUserWhoClosedHtml),
                    values: timelineEntry.values,
                    issues: timelineEntry.issues
                };
                data.rows.push(lastRow);
            }

            // entry to describe a user closing/reopening the sprint
            else if (timelineEntry.openCloseInstances) {
                _.each(timelineEntry.openCloseInstances, function(instance) {
                    data.rows.push({
                        time: timelineEntry.time,
                        event: BurndownTableProducer.getSprintStateChangeEvent(instance),
                        openClose: true,
                        values: {}
                    });
                });
            }

            // change entry
            else {
                // add row for each change in the point
                _.each(timelineEntry.issues, function (issue) {
                    // update the current values with the deltas provided by the issue
                    BurndownTimelineProducer.addStatisticValueDeltas(issue.deltas, data.currentValues, 1, statisticConsumer);

                    // create a row for this change
                    var row = {
                        time:timelineEntry.time,
                        key:issue.key,
                        deltas:issue.deltas,
                        values:_.clone(data.currentValues)
                    };

                    // generate the event
                    row.event = tableHandler.getChangeEvent(issue);

                    // add additional information
                    tableHandler.addAdditionalRowInformation(data, timelineEntry, issue, row);

                    // add the row
                    data.rows.push(row);
                });
            }
        });

        // return the data table description
        var desc = {
            statistics: tableHandler.getStatisticDefinitions(),
            rows: data.rows
        };
        return desc;
    };

    /**
     * Change event  for sprint start
     */
    BurndownTableProducer.getSprintStartChangeEvent = function() {
        return {
            type: AJS.I18n.getText('gh.rapid.chart.burndown.event.sprint.start.type'),
            detail: ''
        };
    };

    /**
     * Change event for sprint complete
     */
    BurndownTableProducer.getSprintCompleteChangeEvent = function(closingUser) {
        return {
            type: tpl.burndownchart.renderSprintCompleteEventType({closingUserHtml: closingUser}),
            typeAsHtml: true,
            detail: ''
        };
    };

    /**
     * Change event for sprint reopen
     */
    BurndownTableProducer.getSprintReopenChangeEvent = function(reopenUser) {
        return {
            type: GH.tpl.burndownchart.renderSprintReopenEventType({reopenUserHtml: reopenUser}),
            typeAsHtml: true,
            detail: ''
        };
    };

    BurndownTableProducer.getSprintStateChangeEvent = function(instance) {
        if (instance.operation === 'CLOSE') {
            // in this case it can be either instance.user if it is the last close change that was added by
            // GH.BurndownTimelineProducer.calculateTimeline at the end or it is a regular openClose change
            // and therefore instance.userDisplayNameHtml
            return BurndownTableProducer.getSprintCompleteChangeEvent(instance.user || instance.userDisplayNameHtml);
        } else {
            return BurndownTableProducer.getSprintReopenChangeEvent(instance.user || instance.userDisplayNameHtml);
        }
    };

    /**
     * Returns an event for either the sprint or column change
     */
    BurndownTableProducer.getSprintOrColumnChangeEvent = function(issue, burndownEventTypeName) {
        // handle issue added/removed from sprint
        if (issue.scopeChange) {
            if (issue.inScope) {
                return {
                    type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                    detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.issue.added')
                };
            } else {
                return {
                    type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                    detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.issue.removed')
                };
            }
        }

        return BurndownTableProducer.getColumnChangeEvent(issue, burndownEventTypeName);
    };

    BurndownTableProducer.getColumnChangeEvent = function(issue, burndownEventTypeName) {
        // handle column change
        if (issue.columnChange) {
            if (issue.column === 'notDone') {
                if (issue.oldColumn) { // moved from done column
                    return {
                        type: burndownEventTypeName, //AJS.I18n.getText('gh.rapid.chart.burndown.event.issue.state.change'),
                        detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.burndown.detail.issue.not.done')
                    };
                } else { // moved from unmapped column
                    return {
                        type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                        detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.not.done.issue.moved.in')
                    };
                }
            }
            if (issue.column === 'done') {
                if (issue.oldColumn) { // moved from notDone column
                    return {
                        type: burndownEventTypeName, //AJS.I18n.getText('gh.rapid.chart.burndown.event.issue.state.change'),
                        detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.burndown.detail.issue.done')
                    };
                } else { // moved from unmapped column
                    return {
                        type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                        detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.done.issue.moved.in')
                    };
                }
            }
            else { // moved into unmapped column
                if (issue.oldColumn === 'notDone') { // moved from not done column
                    return {
                        type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                        detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.not.done.issue.moved.out')
                    };
                } else { // moved from done column
                    return {
                        type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                        detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.done.issue.moved.out')
                    };
                }
            }
        }

        // none of the above
        return false;
    };

    /**
     * Time tracking timeline handler
     */
    BurndownTableProducer.TimeTrackingTableHandler = {

        /**
         * Get the statistics contained in the timeline
         */
        getStatisticDefinitions: function() {
            var statistics = [
                // fake time spent for now
                {
                    id: 'timeSpent',
                    name: AJS.I18n.getText('gh.issue.time'),
                    isSum: true // this is a sum, so its not about remaining
                },
                BurndownChartModel.getStatistic()
            ];
            statistics[0].valueId = 'timeSpent';
            statistics[1].valueId = 'estimate';

            return statistics;
        },

        /** Adds time tracking related information to the table row */
        addAdditionalRowInformation: function(data, timelineEntry, issue, row) {
            // add recorded change date
            if (issue.change && issue.change.timeC && issue.change.timeC.changeDate) {
                // only add if not in the same minute as timeline entry
                var changeDate = issue.change.timeC.changeDate;
                if ((timelineEntry.time - (timelineEntry.time % 60000)) !== (changeDate - (changeDate % 60000))) {
                    row.recordedTime = changeDate;
                }

                // could be back dated, lets record for back referencing
                data.pastChangeRows[issue.key + '-' + changeDate] = row;
            }

            // mark previous rows as backdated if the current issue contains a reference to them
            var affectingDeltas = BurndownTimelineProducer.TimeTrackingStatisticConsumer.getAffectingEstimateChanges(issue, issue.change);
            _.each(affectingDeltas, function (estimateDelta) {
                var backdatedRow = data.pastChangeRows[estimateDelta.issueKey + '-' + estimateDelta.date];
                if (backdatedRow) {
                    // ensure we add the note once only
                    if (backdatedRow.event.loggedNoteAdded) { return; }
                    backdatedRow.event.loggedNoteAdded = true;

                    // add note
                    if (!backdatedRow.event.notes) { backdatedRow.event.notes = []; }
                    backdatedRow.event.notes.push(AJS.I18n.getText('gh.rapid.chart.burndown.event.backdated.recorded.at', ChartUtils.renderUTCMillisAsDate(estimateDelta.date)));
                }
            });
        },

        /** Calculates a change event for a timetracking change entry */
        getChangeEvent: function(issue) {
            // handle sprint in/out or column change
            var burndownEventTypeName = AJS.I18n.getText('gh.rapid.chart.burndown.event.issue.state.change');
            var sprintOrColumnEvent = BurndownTableProducer.getSprintOrColumnChangeEvent(issue, burndownEventTypeName);
            if (sprintOrColumnEvent) { return sprintOrColumnEvent; }

            // handle estimate / time spent change
            if (issue.estimateChange || issue.timeChange) {
                // data to display
                var timeSpentDelta = 0;
                var hasTimeSpent = false;
                var oldEstimate = 0;
                var hasOldEstimate = false;
                var newEstimate = 0;
                var hasNewEstimate = false;

                if (issue.timeChange) {
                    // work was logged
                    timeSpentDelta = issue.deltas.timeSpent;
                    hasTimeSpent = true;
                }
                if (issue.estimateChange) {
                    newEstimate = issue.values.estimate;
                    oldEstimate = issue.values.estimate - issue.deltas.estimate||0;
                    hasOldEstimate = !_.isUndefined(oldEstimate);
                    hasNewEstimate = !_.isUndefined(newEstimate);
                }

                var data = {};

                // Choose event type. Logged work includes estimate changes
                if (hasTimeSpent) {
                    data.type = AJS.I18n.getText('gh.rapid.chart.burndown.event.timetracking.work.logged');
                } else {
                    var remainingLong = AJS.I18n.getText('gh.rapid.config.tracking.statistic.timetracking.remaining');
                    var remainingAbbrev = AJS.I18n.getText('gh.rapid.config.tracking.statistic.timetracking.remaining.acronym');
                    var abbrevHtml = '<acronym title="' + remainingLong + '">' + remainingAbbrev + '</acronym>';
                    data.type = AJS.I18n.getText('gh.rapid.chart.burndown.event.timetracking.estimate.change', abbrevHtml);
                    data.typeAsHtml = true;
                }

                // put together event details
                var detail = '';
                var notes = [];
                if (hasTimeSpent) {
                    detail += AJS.I18n.getText('gh.rapid.chart.burndown.event.timetracking.work.logged.desc', BurndownChartModel.renderStatisticText(timeSpentDelta));
                }
                if (hasOldEstimate || hasNewEstimate) {
                    var oldEstimateText = hasOldEstimate ? BurndownChartModel.renderStatisticText(oldEstimate) : AJS.I18n.getText('gh.rapid.chart.burndown.event.timetracking.estimate.change.novalue');
                    var newEstimateText = hasNewEstimate ? BurndownChartModel.renderStatisticText(newEstimate) : AJS.I18n.getText('gh.rapid.chart.burndown.event.timetracking.estimate.change.novalue');
                    detail += detail.length > 0 ? ', ' : '';
                    detail += AJS.I18n.getText('gh.rapid.chart.burndown.event.timetracking.estimate.change.desc', BurndownChartModel.getStatistic().name, oldEstimateText, newEstimateText);

                    // add backdated changes
                    var affectingDeltas = BurndownTimelineProducer.TimeTrackingStatisticConsumer.getAffectingEstimateChanges(issue, issue.change);
                    _.each(affectingDeltas, function(estimateDelta) {
                        notes.push(
                            AJS.I18n.getText('gh.rapid.chart.burndown.event.timetracking.estimate.change.backdated',
                                ChartUtils.renderStatisticText(estimateDelta.delta),
                                ChartUtils.renderUTCMillisAsDate(estimateDelta.worklogDate),
                                ChartUtils.renderUTCMillisAsDate(estimateDelta.date)
                            )
                        );
                    });
                }
                data.detail = detail;
                data.notes = notes;

                return data;
            }

            // should never happen
            return {
                type: 'unknown',
                detail: ['unknown change']
            };
        }
    };

    /**
     * Statistics timeline handler
     */
    BurndownTableProducer.EstimateStatisticTableHandler = {

        /** Get the statistics defined by this handler */
        getStatisticDefinitions: function() {
            var statistics = [
                BurndownChartModel.getStatistic()
            ];
            statistics[0].valueId = 'estimate';
            return statistics;
        },

        /** Add additional data to the row */
        addAdditionalRowInformation: function(data, timelineEntry, issue, row) {
            // nothing to do
        },

        /** Calculates a change event for a timetracking change entry */
        getChangeEvent: function(issue) {
            // handle sprint in/out or column change
            var burndownEventTypeName = AJS.I18n.getText('gh.rapid.chart.burndown.event.burndown.type');
            var sprintOrColumnEvent = BurndownTableProducer.getSprintOrColumnChangeEvent(issue, burndownEventTypeName);
            if (sprintOrColumnEvent) { return sprintOrColumnEvent; }

            // handle statistic change
            if (issue.estimateChange) {
                // data to display
                var oldEstimate = issue.values.oldEstimateFieldValue;
                var newEstimate = issue.values.estimateFieldValue;
                var hasOldEstimate = !_.isUndefined(oldEstimate);
                var hasNewEstimate = !_.isUndefined(newEstimate);

                if (hasOldEstimate && hasNewEstimate) {
                    return {
                        type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                        detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.statistic.value.changed', BurndownChartModel.renderStatisticText(oldEstimate), BurndownChartModel.renderStatisticText(newEstimate))
                    };
                } else if (hasOldEstimate) {
                    return {
                        type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                        detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.statistic.value.removed', BurndownChartModel.renderStatisticText(oldEstimate))
                    };
                } else {
                    return {
                        type: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.type'),
                        detail: AJS.I18n.getText('gh.rapid.chart.burndown.event.scope.change.detail.statistic.value.added', BurndownChartModel.renderStatisticText(newEstimate))
                    };
                }
            }

            // should never happen
            return {
                type: 'unknown',
                detail: 'unknown change'
            };
        }
    };

    return BurndownTableProducer;
});
