/* global AJS */
/**
 * CFD Data
 * @module jira-agile/rapid/ui/chart/cfd-data
 * @requires module:jquery
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/chart/cfd-data', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var $ = require('jquery');
    var _ = require('underscore');

    // GLOBALS... FIX ME
    var escapeHtml = AJS.escapeHtml;

    var CFDData = {};

    CFDData.raw = {};

    /**
     * @returns the raw data we're currently working on, or an empty object if not yet been set
     */
    CFDData.getRawData = function () {
        return CFDData.raw;
    };

    /**
     * keep the raw data around for future series recalculations
     */
    CFDData.setRawData = function (data) {
        CFDData.raw = data;
    };

    /**
     * Turn the server-side model into chartable series. What we want in the end is the number of issues in a certain column
     * at a certain time. So the target structure looks like this:
     *
     * series: [
     *      todo: [ [ day1, issueCount ], [ day2, issueCount ], ... ],
     *      in progress: [ [ day1, issueCount ], [ day2, issueCount ], ... ],
     *      ...
     * ]
     *
     * @param model : see CumulativeFlowChartModel for details
     * @param max : the right boundary of the chart
     */
    CFDData.computeSeries = function (model, max) {
        var series = [];
        var changes = model.columnChanges;
        var columns = model.columns;

        // flot requires to have aligned data series. So what we'll do is keep an issue counter for each column,
        // and flush that into the column dataset every time _any_ change is encountered.
        _.each(columns, function (column) {
            column.issueCount = 0;
            column.data = [];
        });

        // now, iterate over all changes
        _.each(changes, function (change, time) {
            // it's an object key, so comes in as string
            time = parseInt(time, 10);

            // purely theoretical - it's possible that two changes happen at the same millisecond.
            // so each timestamp has an array of change records.
            _.each(change, function (issueChange) {

                // now process that change for the columns
                _.each(columns, function (column, colIndex) {

                    // issue moved out from the column, decrease counter
                    if (issueChange.columnFrom == colIndex) {
                        column.issueCount--;
                    }

                    // issue moved into the column, increase counter
                    if (issueChange.columnTo == colIndex) {
                        column.issueCount++;
                    }

                    // regardless of the current column being affected, flush the current counter
                    // into the dataset (flot needs this)
                    column.data.push([time, column.issueCount]);
                });
            });
        });

        // now, move the collected datasets into the series. We do this backwards, since
        // the stack flot plugin shows the last on top (which is, in the CFD case, the first
        // column).
        for (var i = columns.length - 1; i >= 0; i--) {
            var column = columns[i];
            var data = column.data;

            // the end of the viewport ('now') is having a good chance of being outside the series.
            // So we carry the current counters on to a synthetic end datapoint,
            // to avoid the chart suddenly dropping off to the floor.
            if (data.length > 0 && data[data.length - 1][0] < max) {
                data.push([max, data[data.length - 1][1]]);
            }

            // finally! shove it into the series
            series.push({
                data: data,
                label: escapeHtml(String(column.name)),
                columnIndex: i
            });
        }

        return series;
    };

    /**
     * Get the change information for a given timestamp
     */
    CFDData.getChangeDetails = function (time) {
        var data = CFDData.raw;

        var changes = [];
        _.each(data.columnChanges[time], function (change) {
            change = $.extend(true, {}, change);
            if (change.columnFrom != undefined) {
                change.columnFrom = data.columns[change.columnFrom].name;
            }

            if (change.columnTo != undefined) {
                change.columnTo = data.columns[change.columnTo].name;
            }

            changes.push(change);
        });

        return changes;
    };

    /**
     * Check if the datapoint at the given timestamp is a 'real' datapoint for the column
     * specified by columnIndex
     */
    CFDData.isRealDatapoint = function (time, columnIndex) {
        var data = CFDData.raw;
        var changes = data.columnChanges[time];

        // if there's a change actually relating to that column, then it's a real datapoint.
        // otherwise, it's a synthetic one we've created to please flot's stacking.
        return _.any(changes, function (change) {
            return columnIndex == change.columnTo;
        });
    };

    return CFDData;
});