/* global GH */
/**
 * CFD View
 * @module jira-agile/rapid/ui/chart/cfd-view
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/ui/chart/cfd-controller
 * @requires module:jira-agile/rapid/ui/chart/cfd-data
 * @requires module:jira-agile/rapid/ui/chart/chart
 * @requires module:jira-agile/rapid/ui/chart/chart-view
 */
define('jira-agile/rapid/ui/chart/cfd-view', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var $ = require('jquery');
    var _ = require('underscore');
    var CFDController;
    var CFDData;
    var Chart = require('jira-agile/rapid/ui/chart/chart');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var GlobalEvents = require('jira-agile/rapid/global-events');

    // Resolve circular dependency
    GlobalEvents.on('pre-initialization', function () {
        CFDController = require('jira-agile/rapid/ui/chart/cfd-controller');
        CFDData = require('jira-agile/rapid/ui/chart/cfd-data');
    });

    // GLOBALS... FIX THIS
    var ChartFilters = GH.ChartFilters;
    var ChartTimeFrames = GH.ChartTimeFrames;
    var FlotChartUtils = GH.FlotChartUtils;
    var Logger = GH.Logger;
    var tpl = GH.tpl;

    var CFDView = {};

    /** the colours for the columns. index in the array relates to column position (fist colour, first column). */
    CFDView.COLUMN_COLORS = ['#FF800D', '#23819C', '#9669FE', '#59955C', '#B9264F', '#B05F3C', '#2966B8', '#9A03FE', '#FF2626', '#FF5353', '#C8B400', '#79FC4E', '#8C8CFF', '#C48484', '#4FBDDD', '#FFBE28', '#99C7FF', '#AE70ED', '#FFA8A8', '#DAAF85'];

    /** the context holding the data necessary to draw the current chart */
    CFDView.context = {};

    /** Holds the last hovered point. */
    CFDView.previousPlotHoverPoint = null;

    /**
     * Showing the datapoints on a stacked linechart is not working very well in flot. There are two main problems
     * here:
     *  - flot draws the chart bottom to top, so the top layers overlap.
     *  - we need to synthesize datapoints for the 'other' lines (see CFDData.computeSeries), but we want the detail hover
     *    only on 'real' datapoints.
     *
     * Since flot is all matryoshka-style closures, we achieve this by using plugin hooks it provides to collect the draw
     * grid information and then paint it ourselves on the overlay canvas.
     */
    CFDView.init = function () {
        //    // reset some data we collected on the way when the chart is being redrawn on window resize. Flot doesn't know
        //    // about resizing and we collect the data on every draw operation.
        //    $(GH).bind(Chart.EVENT_RESIZE_REDRAW,
        //        function() {
        //            CFDView.resetRealPoints();
        //
        //            // damn thing keeps turning around :)
        //            CFDView._reverseLegendTable();
        //        }
        //    );
    };

    CFDView.show = function () {
        // reset some data we collected on the way when the chart is being redrawn on window resize. Flot doesn't know
        // about resizing and we collect the data on every draw operation.
        $(GH).unbind(Chart.EVENT_RESIZE_REDRAW + "." + CFDController.id);
        $(GH).bind(Chart.EVENT_RESIZE_REDRAW + "." + CFDController.id, function (event, data) {
            if (data.id !== CFDController.id) {
                return;
            }

            Logger.log("Updating CFD after window resize event", Logger.Contexts.ui);
            CFDView.resetRealPoints();

            // damn thing keeps turning around :)
            CFDView._reverseLegendTable();
        });
    };

    CFDView.hide = function () {
        // reset some data we collected on the way when the chart is being redrawn on window resize. Flot doesn't know
        // about resizing and we collect the data on every draw operation.
        $(GH).unbind(Chart.EVENT_RESIZE_REDRAW + "." + CFDController.id);
        Chart.destroy(CFDController.id + 'Overview');
        Chart.destroy(CFDController.id);
    };

    /**
     * Render the chart for the given raw data. Takes boundaries from the timeframe filter into account.
     */
    CFDView.renderChart = function (data) {
        // GHS-4621: temporary fix to stop exceptions from blowing up the page
        Chart.destroy(CFDController.id + 'Overview');
        Chart.destroy(CFDController.id);

        // fetch the current limits for the chart
        var xMin = ChartTimeFrames.getChartStartDate();
        var xMax = ChartTimeFrames.getChartEndDate();

        //ChartTimeFrames.setBoundaries(xMin, xMax);

        // build the series
        // use 'data.now' as max point, not xMax (otherwise we adapt the endpoint of the overview chart to xMax)
        var series = CFDData.computeSeries(data, data.now);
        var colors = CFDView.getColors(ChartFilters.getColumns(), ChartFilters.getActiveColumns());

        var options = {
            colors: colors,
            xaxis: {
                mode: 'time',
                min: xMin,
                max: xMax
            },
            yaxis: {
                tickDecimals: 0,
                minTickSize: 1, // no half issues
                min: 0,
                tickLength: 'full' // show horizontal grid lines
            },
            series: {
                stack: true,
                lines: {
                    show: true,
                    fill: 1,
                    lineWidth: 1 // the default of 2px causes artifacts when the top area goes to zero, setting it to 0px causes
                    // artifacts between fill areas (probably floating point inaccuracy). 1px seems to be the best
                    // compromise without touching the lib.
                }
            },
            grid: {
                hoverable: true,
                clickable: true,
                autoHighlight: false // we've got to handle highlighting ourselves since it doesn't work properly with stacking
            },
            mouseEnterExitEvents: true,
            selection: { mode: 'x' },
            legend: {
                container: null,
                backgroundOpacity: 0.5,
                position: 'nw'
            }
        };

        var overviewOptions = {
            colors: colors,
            xaxis: { mode: 'time' },
            yaxis: { show: false },
            series: {
                stack: true,
                lines: {
                    show: true,
                    fill: 1,
                    lineWidth: 1
                }
            },
            selection: { mode: 'x' },
            legend: { show: false }
        };

        ChartView.hideChartStatus();
        ChartView.showChartGroup();

        // set the chart labels
        FlotChartUtils.setAndAlignAxisLabels(AJS.I18n.getText('gh.rapid.chart.cfd.xaxis.label'), AJS.I18n.getText('gh.chart.issuecount'));

        // we have two views, the main (big) and overview (small, for selection)
        var chartContainer = ChartView.getChartView(true);
        var overviewGroup = $('#ghx-chart-overview-group');
        var overviewContainer = $('#ghx-chart-overview');
        var overviewHeader = $(overviewGroup).find('h4');
        var overviewDesc = $(overviewGroup).find('.ghx-description');

        // set the header and desc text
        $(overviewHeader).text(AJS.I18n.getText('gh.rapid.charts.cfd.overview.name'));
        $(overviewDesc).text(AJS.I18n.getText('gh.rapid.charts.cfd.overview.description'));

        // Show the required elements
        overviewHeader.show();
        overviewDesc.show();
        chartContainer.show();
        overviewGroup.show();

        // draw the main chart
        var ctx = CFDView.context = {
            chartContainer: chartContainer,
            series: series,
            options: options,
            realPoints: [],
            showPoints: false
        };
        chartContainer.unbind('plothover plotselected');
        chartContainer.bind('plotselected', CFDView.handleSelection);
        chartContainer.bind('plothover', CFDView.handlePlotHover);

        ctx.mainChart = Chart.draw(CFDController.id, chartContainer, series, options);

        // mouse enter/leave events
        chartContainer.unbind('flotMouseEnter flotMouseLeave').bind('flotMouseEnter', CFDView.handleMouseEnter).bind('flotMouseLeave', CFDView.handleMouseLeave);

        // register hooks to handle dot painting
        ctx.mainChart.hooks.drawSeries.push(CFDView.gatherPoints);
        ctx.mainChart.hooks.draw.push(CFDView.deferredRenderPoints);

        // We reversed the data, so the legend now looks reversed too. Re-Reverse it again once rendered
        CFDView._reverseLegendTable();

        overviewContainer.unbind('plotselected dblclick');
        overviewContainer.bind('plotselected', CFDView.handleSelection);
        overviewContainer.bind('dblclick', CFDView.resetSelection);

        // draw the overview chart
        ctx.overviewChart = Chart.draw(CFDController.id + 'Overview', overviewContainer, series, overviewOptions);

        // select the overview chart
        ctx.overviewChart.setSelection({
            xaxis: {
                from: xMin,
                to: xMax
            }
        }, true);
    };

    /**
     * Determine the colours to be shown. We've got a preset array that is to be used, from left to right by column order.
     * This gets slightly complicated by the fact that certain columns can be hidden, but the colours still have to map
     * correctly (as if all columns were present).
     */
    CFDView.getColors = function (allColumns, activeColumns) {
        var colors = [];

        // collect the correct colours for the active columns
        _.each(allColumns, function (column, index) {
            var active = _.any(activeColumns, function (activeColumn) {
                return activeColumn.id == column.id;
            });

            if (active) {
                colors.push(CFDView.COLUMN_COLORS.slice(index));
            }
        });

        // the series are drawn last to first, since that's required for flot-stack to work. So we need to fetch the column colours
        // for the series and turn them around as well.
        return colors.reverse();
    };

    CFDView._reverseLegendTable = function () {
        var legendTable = ChartView.getChartView().find('.legend').find('tbody');
        legendTable.children().each(function () {
            legendTable.prepend($(this));
        });
    };

    /**
     * Updates the charts with the new timeframe information.
     * Sets the preview label on the small chart and the absolute sc
     */
    CFDView.updateTimeFrameSelection = function () {
        var xMin = ChartTimeFrames.getChartStartDate();
        var xMax = ChartTimeFrames.getChartEndDate();

        var ctx = CFDView.context;
        ctx.options.xaxis.min = xMin;
        ctx.options.xaxis.max = xMax;

        // select the overview chart, but only if it is not all time
        var selectedTimeFrame = ChartTimeFrames.getSelectedTimeFrame();
        if (selectedTimeFrame.days != 0) {
            ctx.overviewChart.setSelection({
                'xaxis': {
                    'from': xMin,
                    'to': xMax
                }
            }, true);
        }

        // draw the main chart again
        ctx.mainChart = Chart.draw(CFDController.id, ctx.chartContainer, ctx.series, ctx.options);

        // reverse the legend table
        CFDView._reverseLegendTable();
    };

    /**
     * When the selection on either the main or the overview chart changes, redraw the main chart with the new boundaries.
     */
    CFDView.handleSelection = function (event, ranges) {
        var from = ranges.xaxis.from;
        var to = ranges.xaxis.to;

        // this will trigger an event which will cause the controller to redraw the chart.
        ChartTimeFrames.setNewChartLimits(-1, from, to);
    };

    CFDView.resetSelection = function (event) {
        var xaxis = CFDView.context.overviewChart.getXAxes()[0];

        ChartTimeFrames.setNewChartLimits(-1, xaxis.min, xaxis.max); // set to All Time
    };

    // ==================== custom datapoint hover pluing ====================

    /**
     * Basically, this is a combined mouseMove / mouseLeave handler on the chart canvas.
     *
     * @param event : flot munges both events into one custom 'plothover' event
     * @param pos : the cursor position
     * @param item : the item flot believes we're currently hovering.
     */
    CFDView.handlePlotHover = function (event, pos, item) {
        if (item) {

            if (item.dataIndex != CFDView.previousPlotHoverPoint && CFDView.isRealPoint(item)) {
                $('#ghx-tooltip').remove();
                CFDView.previousPlotHoverPoint = item.dataIndex;
                CFDView.renderTooltip(item.datapoint[0], item.pageX, item.pageY);
            }
        } else {
            $('#ghx-tooltip').remove();
            CFDView.previousPlotHoverPoint = null;
        }
    };

    /**
     * The item flot gives us here is not necessarily the right one. The problem is that flot
     * needs synthetic datapoints on all lines for a change, not only on the line where it happened,
     * to make the stacking work.
     * So we need to double-check that ourselves, if we're looking at a real highlighted datapoint
     * or just a synthetic one which we don't want to show.
     *
     * Since we've figured out earlier on which datapoints we actually want to highlight (stored in
     * context.realPoints), we emulate flot's "nearest" function to see if we're actually close to a
     * real point.
     */
    CFDView.isRealPoint = function (item) {
        var xaxis = item.series.xaxis;
        var yaxis = item.series.yaxis;
        // normalise data into chart coords
        var ix = xaxis.p2c(item.datapoint[0]);
        var iy = yaxis.p2c(item.datapoint[1]);

        return _.any(CFDView.context.realPoints, function (point) {
            // again, normalise into chart coords for comparison
            var px = xaxis.p2c(point.time);
            var py = yaxis.p2c(point.issueCount);

            // datapoints can be pretty close to each other, so we're having
            // a fairly low distance threshold (1px). More leads to too many false positives,
            // although the user should zoom in on tight series anyway.
            return Math.abs(ix - px) < 1 && Math.abs(iy - py) < 1;
        });
    };

    /**
     * We want to show the datapoint highlights only when the user has the curser inside the chart.
     * Unfortunately, flot munges mouseMove and mouseLeave into one custom event, so there's no way
     * of distinguishing through the API. Instead, we're using our plugin to register custom
     * listeners.
     */
    CFDView.handleMouseEnter = function (event) {
        var ctx = CFDView.context;
        if (!ctx.showPoints) {
            ctx.showPoints = true;
            // reset the gathered real datapoints, since the chart will be redrawn which will gather
            // them again.
            ctx.realPoints = [];
            ctx.mainChart.draw();
        }
    };

    /**
     * Hide the datapoints once the cursor leaves the chart
     */
    CFDView.handleMouseLeave = function (event) {
        var ctx = CFDView.context;
        if (ctx.showPoints) {
            ctx.showPoints = false;
            ctx.mainChart.draw();
        }
    };

    /**
     * Draw the tooltip information for the given time, at the given coords.
     *
     * @param time : timestamp of the changes we're interested in
     * @param x, y : where to draw
     */
    CFDView.renderTooltip = function (time, x, y) {
        // for now, we're using UTC to display the date
        var dateUserTz = new Date(time);
        var dateUtc = new Date(dateUserTz.getUTCFullYear(), dateUserTz.getUTCMonth(), dateUserTz.getUTCDate(), dateUserTz.getUTCHours(), dateUserTz.getUTCMinutes(), dateUserTz.getUTCSeconds());

        var changes = CFDData.getChangeDetails(time);
        var tooltip = $(tpl.chartview.renderCfdTooltip({
            changes: changes,
            time: dateUtc.print(ChartTimeFrames.timeFormat)
        })).css({
            position: 'absolute',
            display: 'none',
            top: y + 5
        });

        // append to the body (which will give us the correct with)
        tooltip.appendTo('body');

        // check whether we overshoot the right border, in which case we want to display the tooltip to the left
        var tooltipWidth = tooltip.width();
        var canvas = $('#ghx-chart-view canvas.base');

        // now set the correct x position for the tooltip
        if (x + tooltipWidth > canvas.offset().left + canvas.width()) {
            tooltip.css({ left: x - (30 + tooltipWidth) });
        } else {
            tooltip.css({ left: x + 15 });
        }
        tooltip.fadeIn(200);
    };

    /**
     * 'real' points gathering happens during chart draw operations, so we frequently need to reset this
     */
    CFDView.resetRealPoints = function (plot, options) {
        CFDView.context.realPoints = [];
    };

    /**
     * The sequence flot draws things doesn't work too well for us with stacking enabled. At this point, we can't
     * draw the points yet, since the series are drawn bottom up and the next one would cover what we've just drawn.
     *
     * Instead, we gather the 'real' datapoints that we want to draw into a cache and defer drawing to a later stage.
     *
     * @param plot : the chart object
     * @param ctx : the canvas 2d context
     * @param series : series currently being drawn
     */
    CFDView.gatherPoints = function (plot, ctx, series) {
        // don't do anything if we're either not a CFD (this is a global flot plugin) or if the
        // points aren't displayed currently because the user doesn't have the cursor on the chart.
        if (!CFDController.displayed || !CFDView.context.showPoints) {
            return;
        }

        var datapoints = series.datapoints;
        var points = datapoints.points;
        var ps = datapoints.pointsize;
        var axisx = series.xaxis;
        var axisy = series.yaxis;

        for (var i = 0; i < points.length; i += ps) {
            var time = points[i];
            var issueCount = points[i + 1];

            // again, avoid synthetic datapoints on lines that actually don't have the change
            if (!CFDData.isRealDatapoint(time, series.columnIndex)) {
                continue;
            }

            if (time == null || time < axisx.min || time > axisx.max || issueCount < axisy.min || issueCount > axisy.max) {
                continue;
            }

            var x = axisx.p2c(time);
            var y = axisy.p2c(issueCount);

            CFDView.context.realPoints.push({
                x: x,
                y: y,
                time: time,
                issueCount: issueCount
            });
        }
    };

    /**
     * Now flot is done drawing, and we can go through our gathered points from earlier and draw them on top of the stacked
     * chart.
     *
     * @param plot
     * @param ctx
     */
    CFDView.deferredRenderPoints = function (plot, ctx) {
        // don't do anything if we're either not a CFD (this is a global flot plugin) or if the
        // points aren't displayed currently because the user doesn't have the cursor on the chart.
        if (!CFDController.displayed || !CFDView.context.showPoints) {
            return;
        }

        ctx.save();
        ctx.translate(plot.getPlotOffset().left, plot.getPlotOffset().top);

        // style the datapoint highlight circles
        ctx.strokeStyle = 'rgb(100, 100, 100)';
        ctx.fillStyle = 'rgba(255, 255, 255, 0.5)';

        _.each(CFDView.context.realPoints, function (point) {
            FlotChartUtils.drawPoint(ctx, point);
        });

        ctx.restore();
    };

    // init once
    $(CFDView.init);

    return CFDView;
});