/* global GH */
/**
 * Chart
 * @module jira-agile/rapid/ui/chart/chart
 * @requires module:jquery
 */
define('jira-agile/rapid/ui/chart/chart', ['require'], function (require) {
    'use strict';

    // REQUIRES
    var $ = require('jquery');

    // GLOBALS... FIX ME
    var Layout = GH.Layout;
    var log = GH.log;
    var Logger = GH.Logger;

    /**
     * This library is a wrapper for flot (see http://people.iola.dk/olau/flot/API.txt) for API doc.
     * It allows us to centralise utility functions and add default options to all charts.
     */
    var Chart = {};

    /** fired before a chart is redrawn on window resize, giving others the opportunity to do stuff */
    Chart.EVENT_RESIZE_REDRAW = 'gh.chart.resize_redraw';

    /**
     * Default chart options. Specific options will override these.
     */
    Chart.defaultOptions = {
        colors: ["#9eb6d4", "#afd8f8", "#cb4b4b", "#4da74d", "#9440ed"],
        grid: {
            backgroundColor: "#fff",
            borderColor: "#bbb",
            borderWidth: 1,
            internalLineWidth: 0,
            labelMargin: 2,
            minBorderMargin: 1
        },
        legend: {
            container: "#ghx-chart-legend",
            noColumns: 1 // number of columns in legend table
        },
        shadowSize: 0,
        xaxis: {
            tickLength: 4 // sets the little marks next to the units on the x axis, a value here also stops horizontal lines across chart
        },
        yaxis: {
            tickLength: 4 // sets the little marks next to the units on the y axis, a value here also stops horizontal lines across chart
        }

    };

    /**
     * Draw a chart using flot
     *
     * @param container : jQuery object or selector of the element where the chart will be drawn
     * @param series : flot data series
     * @param options : chart options (optional). Override default options.
     *
     * @returns the created flot chart object, for future reference
     */
    Chart.draw = function (id, container, series, options) {
        container = $(container);

        // clone the global defaults into a local variable (we might have more than one chart in a page)
        var effectiveOptions = $.extend(true, {}, Chart.defaultOptions);
        if (options) {
            // override the defaults with the given options
            $.extend(true, effectiveOptions, options);
        }

        var chart = $.plot(container, series, effectiveOptions);
        Chart.bindToWindowSize(id, chart);

        return chart;
    };

    Chart.bindToWindowSize = function (id, chart) {
        $(GH).bind(Layout.EVENT_DELAYED_RESIZE + '.' + id, function () {
            log('adjusting chart [' + id + '] to window size');
            $(GH).trigger(Chart.EVENT_RESIZE_REDRAW, {
                id: id,
                chart: chart
            });

            // GHS-4621: we are temporarily catching potential errors due to canvas resizing until we refactor this code.
            try {
                chart.resize();
                chart.setupGrid();
                chart.draw();
            } catch (error) {
                log("Got an error while trying to resize chart: " + error, Logger.Contexts.ui);
            }
        });
    };

    Chart.destroy = function (id) {
        $(GH).unbind(Layout.EVENT_DELAYED_RESIZE + '.' + id);
    };

    return Chart;
});
