/* global GH, AJS, _, moment */
/**
 * Chart view component.
 *
 * Loads chart data, then passes contorl to chart implementation for actual rendering
 */
GH.ChartUtils = {};

GH.ChartUtils.oneMinuteInMillis = 60 * 1000;

GH.ChartUtils.oneHourInMillis = GH.ChartUtils.oneMinuteInMillis * 60;

GH.ChartUtils.oneDayInMillis = 24 * GH.ChartUtils.oneHourInMillis;

/**
 * Fixes Button Groups Which Wrap to Multiple Lines
 */
GH.ChartUtils.fixButtons = function (el) {

    var bars = AJS.$(el).find('.aui-buttons');
    var buts;
    var barPos;
    var barX;
    var butPos;
    var butX;

    bars.each(function () {

        barPos = AJS.$(this).position();
        barX = barPos.left;

        buts = AJS.$(this).find('.aui-button');
        buts.each(function () {

            butPos = AJS.$(this).position();
            butX = butPos.left;

            if (butX === barX) {
                AJS.$(this).addClass('first');
                AJS.$(this).prev('.aui-button').addClass('last');
            }
        });
    });
};

/**
 * Registers
 */
GH.ChartUtils.registerInlineDialogHacks = function (dialogId) {
    // hacks to handle closing of the layer. We need to attach a close evaluation function that does NOT close if we either click on the dialog
    // or on one of the calendars.
    AJS.$(document).bind("showLayer", function (e, type, inlineLayer) {
        GH.log(e.type + " handled", "anonymous function");

        if (type === "inlineDialog" && AJS && AJS.InlineDialog && AJS.InlineDialog.current && AJS.InlineDialog.current.id === dialogId) {
            // handle InlineLayer out of zone click handler
            inlineLayer._validateClickToClose = function (e) {
                if (AJS.$(e.target).parents('.calendar').length > 0) {
                    return false;
                }
                if (AJS.$(e.target).parents('#inline-dialog-' + AJS.InlineDialog.current.id).length > 0) {
                    return false;
                }
                return true;
            };
            // handle InlineDialog out of zone click handler - inline layer is good enough plus this guy doesn't allow us any overwriting.
            // --> Kill
            AJS.$('body').unbind('click.' + AJS.InlineDialog.current.id + '.inline-dialog-check');
            // Fix Wrapped Button Group
            setTimeout(function () {
                // when switching between charts the dialog may be destroyed before this is run
                // so we only need to fix the buttons if it's still there
                if (AJS.InlineDialog.current && AJS.InlineDialog.current.id) {
                    GH.ChartUtils.fixButtons('#inline-dialog-' + AJS.InlineDialog.current.id);
                }
            }, 0);
        }
    });
};

GH.ChartUtils.createIssueJql = function (issues) {
    var keys = _.pluck(issues, "key");
    return 'issueKey in (' + keys.join(',') + ')';
};

GH.ChartUtils.createEpicJql = function (epic) {
    return '"' + GH.EpicConfig.getEpicLinkFieldName() + '" = "' + epic.key + '"';
};

GH.ChartUtils.createVersionJql = function (version) {
    return 'fixVersion=' + encodeURIComponent(version.jqlEncodedName) + ' AND project=' + version.projectKey;
};

GH.ChartUtils.createStatusJql = function (statuses) {
    var statusNames = _.pluck(statuses, "name");
    return 'Status IN ("' + statusNames.join('","') + '")';
};

GH.ChartUtils.getStatisticConsumer = function (isTimeTracking) {
    var BurndownTimelineProducer = require('jira-agile/rapid/ui/chart/burndown-timeline-producer');
    // choose the correct statisticConsumer
    if (isTimeTracking) {
        return BurndownTimelineProducer.TimeTrackingStatisticConsumer;
    } else {
        return BurndownTimelineProducer.EstimateStatisticConsumer;
    }
};

/**
 * Renders a statistic value for use on tooltip
 */
GH.ChartUtils.renderStatisticText = function (statisticValue, abs, renderer) {
    abs = _.isUndefined(abs) ? false : abs;

    var value = abs ? Math.abs(statisticValue) : statisticValue;

    if (renderer === 'duration') {
        var prefix = !abs && statisticValue < 0 ? '-' : '';
        // time formatting does not handle negative values, so we always take the absolute
        value = Math.abs(value);
        return prefix + GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(value);
    } else {
        return GH.NumberFormat.format(value);
    }
};

/**
 * Renders UTC millis as a date string with the time of day as well
 */
GH.ChartUtils.renderUTCMillisAsDate = function (utcMillis) {
    return GH.ChartUtils.renderUTCMillisInFormat(utcMillis, GH.ChartTimeFrames.momentTimeFormat);
};

/**
 * Renders UTC millis as a date string
 */
GH.ChartUtils.renderUTCMillisAsDateWithoutTime = function (utcMillis) {
    return GH.ChartUtils.renderUTCMillisInFormat(utcMillis, GH.ChartTimeFrames.momentDateFormat);
};

/**
 * Renders UTC millis as a date string using the specified format.
 *
 * @param utcMillis - the date to be formatted, represented in UTC milliseconds
 * @param format - used by moment.js to format the date in a specific way that is customised for jira
 * and which respects the users date/time look-and-feel preferences, e.g. "LL" results in day/month/year format
 * to be produced, but the actual string representation will differ depending on the user's preferences (the date
 * May 14 2010 may result in "05/14/2010" or "14/05/2010" of "May 14 2010", for example).
 */
GH.ChartUtils.renderUTCMillisInFormat = function (utcMillis, format) {
    // input is local (but in utc millis)
    var dateString = moment.utc(utcMillis).format(format);

    return dateString;
};

/**
 * Provide params to create the "normalized" object for CFD and control chart
 */
GH.ChartUtils.getTimeFrameReportUrlParams = function (chartId) {
    var params = {
        days: GH.RapidBoard.State.getBoardSetting(null, 'chart.timeframe.days', 7),
        swimlane: GH.RapidBoard.State.getBoardSetting(null, 'chart.filters.swimlanes', []),
        column: GH.RapidBoard.State.getBoardSetting(null, 'chart.filters.columns.' + chartId, []), // columns are chart type specific
        quickFilter: GH.RapidBoard.State.getBoardSetting(null, 'chart.filters.quickFilters', [])
    };

    if (params.days === -1) {
        params.from = GH.RapidBoard.State.getBoardSetting(null, 'chart.timeframe.from');
        params.to = GH.RapidBoard.State.getBoardSetting(null, 'chart.timeframe.to');
    } else {
        params.from = 0;
        params.to = 0;
    }

    return params;
};

/**
 * Interpret the normalized params and identify the relevant ones to be converted into URL params
 * Applicable for CFD report and control chart
 */
GH.ChartUtils.timeFrameReportParamsToUrl = function (params) {
    var urlParams = {};
    if (params.swimlane.length > 0) {
        urlParams.swimlane = params.swimlane;
    }
    if (params.column.length > 0) {
        urlParams.column = params.column;
    }
    if (params.quickFilter.length > 0) {
        urlParams.quickFilter = params.quickFilter;
    }

    if (params.days === -1) {
        urlParams.from = new Date(params.from).print(GH.RapidBoard.UrlState.dateFormat);
        urlParams.to = new Date(params.to).print(GH.RapidBoard.UrlState.dateFormat);
    } else if (params.days > 0) {
        urlParams.days = params.days;
    }

    return urlParams;
};

GH.ChartUtils.getNormalizedFromUrlForTimeFrameReport = function (params) {
    var normalized = {
        // filters - empty by default
        swimlane: GH.RapidBoard.UrlState.normalizeNumberArrayParam(params.swimlane),
        column: GH.RapidBoard.UrlState.normalizeNumberArrayParam(params.column),
        quickFilter: GH.RapidBoard.UrlState.normalizeNumberArrayParam(params.quickFilter)
    };
    // time frames
    var days = GH.RapidBoard.UrlState.normalizeNumberParam(params.days, 0);
    var from = GH.RapidBoard.UrlState.normalizeDateParam(params.from, null);
    var to = GH.RapidBoard.UrlState.normalizeDateParam(params.to, null);
    if (from !== null && to !== null) {
        normalized.days = -1;
        normalized.from = from.getTime();
        normalized.to = to.getTime();
    } else {
        normalized.days = days;
        normalized.from = 0;
        normalized.to = 0;
    }

    return normalized;
};

GH.ChartUtils.getChartOptions = function (model) {
    var ChartController = require('jira-agile/rapid/ui/chart/chart-controller');

    return {
        xaxis: {
            mode: "time",
            min: model.getXAxisStart(),
            max: model.getXAxisEnd()
        },
        yaxes: model.getYAxes(),
        legend: {
            container: AJS.$("#ghx-chart-legend"),
            backgroundOpacity: 0.5,
            position: "se"
        },
        grid: {
            markings: ChartController.getNonWorkingBlocks,
            hoverable: true,
            clickable: false,
            autoHighlight: false
        },
        mouseEnterExitEvents: true
    };
};

GH.ChartUtils.updateFromNormalizedStateDateTimeParams = function (normalized) {
    GH.RapidBoard.State.setBoardSetting('chart.timeframe.days', normalized['days']);
    GH.RapidBoard.State.setBoardSetting('chart.timeframe.from', normalized['from']);
    GH.RapidBoard.State.setBoardSetting('chart.timeframe.to', normalized['to']);

    GH.RapidBoard.State.setBoardSetting('chart.filters.swimlanes', normalized['swimlane']);
    GH.RapidBoard.State.setBoardSetting('chart.filters.columns.' + normalized['chart'], normalized['column']);
    GH.RapidBoard.State.setBoardSetting('chart.filters.quickFilters', normalized['quickFilter']);
};

// Calculate the y projection of line between pt0 and pt2 over the line pt0 to pt1
GH.ChartUtils.calculateYProjection = function (pt0, pt1, pt2) {
    function convert(pt) {
        if (_.isArray(pt)) {
            return { x: pt[0], y: pt[1] };
        }
        return pt;
    }

    pt0 = convert(pt0);
    pt1 = convert(pt1);
    pt2 = convert(pt2);

    var dx1 = pt1.x - pt0.x;
    var dx2 = pt2.x - pt0.x;
    var dy1 = pt1.y - pt0.y;

    var dy2 = dy1 * dx2 / dx1;
    return dy2 + pt0.y;
};