GH.EpicReport = {};

GH.EpicReport.createModel = function () {
    GH.EpicReport.model = new GH.EpicReportModel();
    GH.EpicReport.model.setParams({
        statistic: GH.EpicReportController.rapidViewConfig.estimationStatistic,
        issueAddedText: AJS.I18n.getText('gh.rapid.charts.epic.event.scope.change.detail.issue.added'),
        issueRemovedText: AJS.I18n.getText('gh.rapid.charts.epic.event.scope.change.detail.issue.removed'),
        startText: AJS.I18n.getText('gh.report.epic.event.epic.start')
    });
};

/**
 * @class GH.EpicReportModel
 * @constructor
 */
GH.EpicReportModel = function () {};

(function () {
    var ChangesReportModel = require('jira-agile/rapid/ui/chart/changes-report-model');
    GH.EpicReportModel.prototype = new ChangesReportModel();
})();

GH.EpicReportModel.prototype.setRawEpicChartData = function (data) {
    // ensure we can read start time from epic creation time
    data.startTime = data.epicCreationTime;
    this.setRawChartData(data);
};

GH.EpicReportModel.prototype.setRawData = function (data) {
    this.data = data;
    this.processData();
    this.generateJql();
    this.generateSummary();
};

GH.EpicReportModel.prototype.generateSummary = function () {
    var summary = {},
        stats = {},
        content = this.getReportContent();

    function getTotalEstimate(issues) {
        return _.reduce(issues, function (sum, issue) {
            var estimate = issue.estimateStatistic.statFieldValue.value;
            if (_.isUndefined(estimate)) {
                estimate = 0;
            }
            return sum + estimate;
        }, 0);
    }

    stats.done = content.completedIssues.length;
    stats.notEstimated = content.incompleteUnestimatedIssues.length;
    stats.totalIssueCount = stats.done + stats.notEstimated + content.incompleteEstimatedIssues.length;
    summary.epicStats = stats;

    summary.estimateDone = getTotalEstimate(content.completedIssues);
    summary.estimateTotal = summary.estimateDone + getTotalEstimate(content.incompleteEstimatedIssues);
    summary.estimateStatisticName = this.getStatistic().name;
    summary.isIssueCountStatistic = this.isIssueCountStatistic();

    summary.estimateDone = GH.BacklogStatistics.formatStatisticForRendering(GH.EpicReportController.rapidViewConfig.estimationStatistic, summary.estimateDone);
    summary.estimateTotal = GH.BacklogStatistics.formatStatisticForRendering(GH.EpicReportController.rapidViewConfig.estimationStatistic, summary.estimateTotal);

    var epic = this.getEpic();
    summary.epicName = GH.EpicConfig.getEpicNameString(epic.label, epic.key);

    this.data.epicSummary = summary;
};

GH.EpicReportModel.prototype.generateJql = function () {
    var statistic = this.getStatistic();

    this.jql = {};
    var epicJql = GH.ChartUtils.createEpicJql(this.data.epic);
    var notDoneJql = epicJql + " AND " + GH.ChartUtils.createStatusJql(this.getNotDoneStatuses());

    if (this.isIssueCountStatistic()) {
        this.jql.incompleteEstimatedIssues = notDoneJql;
    } else {
        var statisticFieldId = statistic.fieldId;
        var match = /^customfield_(\d+)/.exec(statisticFieldId),
            isCustomField = match && match.length > 0;

        if (isCustomField) {
            statisticFieldId = 'cf[' + match[1] + ']';
        }

        // "incomplete unestimated" issues will only ever include issues that CAN have an estimate, but do not.
        this.jql.incompleteUnestimatedIssues = notDoneJql + ' AND ' + statisticFieldId + ' IS EMPTY';

        // GHS-7191: "incomplete estimated" issues MAY include issues that CANNOT have an estimate. Therefore, we can't
        // always use JQL that contains the statistic field, because if the field is a custom field which only applies
        // to certain issue types, the correct results will not be returned by using "field IS NOT EMPTY".
        // in this case, fall back to a JQL string with issue keys.
        if (isCustomField) {
            if (this.data.contents.incompleteEstimatedIssues.length > 0) {
                this.jql.incompleteEstimatedIssues = GH.ChartUtils.createIssueJql(this.data.contents.incompleteEstimatedIssues);
            }
        } else {
            this.jql.incompleteEstimatedIssues = notDoneJql + ' AND ' + statisticFieldId + ' IS NOT EMPTY';
        }
    }

    this.jql.completedIssues = epicJql + " AND " + GH.ChartUtils.createStatusJql(this.getDoneStatuses());
    this.jql.allIssues = epicJql;
};

GH.EpicReportModel.prototype.getEpic = function () {
    return this.data.epic;
};

GH.EpicReportModel.prototype.getEpicSummary = function () {
    return this.data.epicSummary;
};

GH.EpicReportModel.prototype.getStatistic = function () {
    return GH.EpicReportController.rapidViewConfig.estimationStatistic;
};

GH.EpicReportModel.prototype.canUserEditEpics = function () {
    return !!this.data.canEditEpics;
};

GH.EpicReportModel.prototype.supportsPages = function () {
    return !!this.data.supportsPages;
};

/**
 * Check the report data against the chart data for each issues still in the chart. The statistic values should correspond, if they
 * don't we have a mismatch!
 */
GH.EpicReportModel.prototype.getIssuesWithStatisticFieldMismatch = function () {
    // this can only happen for non-issuecount statistics
    if (this.isIssueCountStatistic()) {
        return [];
    }

    // fetch the issue states calculated by the burndown model
    var chartIssues = this.timelineData.finalIssueStates;
    // fetch all issues from the report
    var contents = this.data.contents;
    var reportIssues = _.flatten([contents.completedIssues, contents.incompleteEstimatedIssues, contents.incompleteUnestimatedIssues]);

    // go through all report issues, because they are the ones we care about. some issues in the chart might be wrong, but
    // as long as they are moved out of scope we don't care as they won't influence the final numbers
    var problematicIssues = [];
    _.each(reportIssues, function (reportIssue) {
        var chartIssue = chartIssues[reportIssue.key];
        if (chartIssue) {
            // compare the statistic values
            if (reportIssue.estimateStatistic && reportIssue.estimateStatistic.statFieldValue) {
                if (reportIssue.estimateStatistic.statFieldValue.value != chartIssue.values.estimateFieldValue) {
                    problematicIssues.push({
                        key: chartIssue.key,
                        reportValue: reportIssue.estimateStatistic.statFieldValue.value,
                        chartValue: chartIssue.values.estimateFieldValue,
                        reportIssue: reportIssue,
                        chartIssue: chartIssue
                    });
                }
            }
        }
    });
    return problematicIssues;
};