/*
Contains functions to assist with drawing and interacting with flot charts as used in Burndown, Epic Report, CFD and Control Chart
 */
GH.FlotChartUtils = {};
/**
 * Handler for plot click events
 */
GH.FlotChartUtils.handlePlotClick = function (event, pos, item) {};

/**
 * Draws points along the line - deferred because it runs once flot is finished drawing
 *
 * @param plot
 * @param ctx
 */
GH.FlotChartUtils.deferredRenderPoints = function (plot, ctx) {
    ctx.save();
    ctx.translate(plot.getPlotOffset().left, plot.getPlotOffset().top);

    // style the datapoint highlight circles
    ctx.strokeStyle = 'rgb(100, 100, 100)';
    ctx.fillStyle = 'rgba(255, 255, 255, 0.5)';

    _.each(plot.realPointsPerSeries, function (realPoints) {
        _.each(realPoints, function (point) {
            GH.FlotChartUtils.drawPoint(ctx, point);
        });
    });

    ctx.restore();
};

GH.FlotChartUtils.drawPoint = function (ctx, point) {
    ctx.beginPath();
    // draw circles with 3px diameter
    ctx.arc(point.x, point.y, 3, 0, Math.PI * 2, false);
    ctx.fill();
    ctx.closePath();
    ctx.stroke();
};

GH.FlotChartUtils.handleMouseOverStateChange = function (event, plot) {
    GH.Logger.log('Handling mouse over state change in ChartView', GH.Logger.Contexts.ui);
    plot.realPointsPerSeries = {}; // clear the real points
    plot.draw();
};

/**
 * Set the chart axis labels and properly align the y axis
 */
GH.FlotChartUtils.setAndAlignAxisLabels = function (xAxisLabel, yAxisLabel, yRightAxisLabel) {
    // set the proper x and y axis labels
    AJS.$('#ghx-axis-x').text(xAxisLabel);
    AJS.$('#ghx-axis-y').find('.ghx-inner').text(yAxisLabel);
    if (yRightAxisLabel) {
        AJS.$('#ghx-axis-y-right').find('.ghx-inner').text(yRightAxisLabel);
    }
};

/**
 * Return a y-axis object (suitable for flot) to best represent this chart
 */
GH.FlotChartUtils.calculateYAxis = function (maxStatisticValue, statistic) {
    var yAxisData = {};
    if (statistic.renderer === "duration") {
        return GH.FlotChartUtils.createYAxisForDuration(maxStatisticValue);
    } else if (statistic.typeId === 'issueCount') {
        return GH.FlotChartUtils.createYAxisForIssueCount(maxStatisticValue);
    } else {
        // Just set a tick formatter
        return {
            tickFormatter: GH.FlotChartUtils.numberFormatter
        };
    }
    return yAxisData;
};

GH.FlotChartUtils.issueCountFormatter = function (v, axis) {
    // obliterate fractional part
    return v.toFixed(0);
};

GH.FlotChartUtils.durationFormatter = function (v, axis) {
    return GH.TimeFormat.formatShortDurationForTimeTrackingConfiguration(v, {
        timeFormat: GH.TimeFormatDisplays.HOURS, // don't use pretty format here - hours will do for now
        durationUnit: GH.TimeFormatDurationUnits.SECONDS
    });
};

GH.FlotChartUtils.numberFormatter = function (v, axis) {
    return GH.NumberFormat.format(v);
};

GH.FlotChartUtils.createYAxisForIssueCount = function (maxStatisticValue) {

    var yAxis = {
        tickFormatter: GH.FlotChartUtils.issueCountFormatter
    };

    // issue count ticks should never include decimal points
    if (maxStatisticValue < 15) {
        // before 15 issues, flot behaves okay
        yAxis.minTickSize = 1;
    } else {
        // 15 or more and there's a chance it can make the tick size things like '2.5'
        var numTicks = 8;
        var minTickSize = Math.ceil(maxStatisticValue / numTicks);
        yAxis.tickSize = minTickSize;
    }
    return yAxis;
};

GH.FlotChartUtils.createYAxisForDuration = function (maxStatisticValue) {
    // calculate y-axis tick size (hard code number of ticks to 8)
    var numTicks = 8,
        minTickSize = parseInt(maxStatisticValue / numTicks, 10);

    if (maxStatisticValue === 0) {
        minTickSize = 3600;
    }

    // round the minimum tick size to the nearest hour (values are in seconds)
    var r = minTickSize % 3600;
    if (r !== 0) {
        minTickSize += 3600 - r;
    }

    return {
        tickSize: minTickSize,
        tickFormatter: GH.FlotChartUtils.durationFormatter
    };
};

GH.FlotChartUtils.gatherPoints = function (series, seriesData) {
    var datapoints = series.datapoints;
    var points = datapoints.points,
        pointsize = datapoints.pointsize;
    var axisx = series.xaxis;
    var axisy = series.yaxis;

    var realPoints = [];

    // go through the points. Each point is flattened into the array, so points are apart by pointsize
    for (var i = 0; i < points.length; i += pointsize) {
        // calculate the index of the current data point
        var dataIndex = Math.ceil(i / pointsize); // ps = 2 in 2d
        // fetch data for current point, skip if we got none
        var data = seriesData[dataIndex];
        if (_.isEmpty(data)) {
            continue;
        }

        var time = points[i],
            statValue = points[i + 1];

        if (time == null || time < axisx.min || time > axisx.max || statValue < axisy.min || statValue > axisy.max) {
            continue;
        }

        var x = axisx.p2c(time);
        var y = axisy.p2c(statValue);

        realPoints.push({
            x: x,
            y: y,
            time: time,
            statValue: statValue
        });
    }
    return realPoints;
};