/* global GH */
/**
 * Version Report Model
 * @module jira-agile/rapid/ui/chart/version-report-model
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/ui/chart/changes-report-model
 * @requires module:jira-agile/rapid/ui/chart/chart-colors
 * @requires module:jira-agile/rapid/ui/chart/chart-controller
 * @requires module:jira-agile/rapid/ui/chart/release-date-predictor
 */
define('jira-agile/rapid/ui/chart/version-report-model', ['require'], function (require) {
    'use strict';

    // REQUIRES
    var _ = require('underscore');
    var ChangesReportModel = require('jira-agile/rapid/ui/chart/changes-report-model');
    var ChartColors = require('jira-agile/rapid/ui/chart/chart-colors');
    var ChartController = require('jira-agile/rapid/ui/chart/chart-controller');
    var ReleaseDatePredictor = require('jira-agile/rapid/ui/chart/release-date-predictor');

    // GLOBALS... FIX ME
    var ChartUtils = GH.ChartUtils;

    /**
     * @class EpicReportModel
     * @constructor
     */
    var VersionReportModel = function (estimationStatistic) {
        this.estimationStatistic = estimationStatistic;
    };

    VersionReportModel.prototype = new ChangesReportModel();

    VersionReportModel.prototype.getStatistic = function () {
        return this.estimationStatistic;
    };

    VersionReportModel.prototype.getProjectKey = function () {
        return this.chartData.projectKey;
    };

    VersionReportModel.prototype.getVersionName = function () {
        return this.chartData.version.name;
    };

    VersionReportModel.prototype.getNow = function () {
        return this.chartData.now;
    };

    VersionReportModel.prototype.getVersion = function () {
        if (this.data && this.data.version) {
            return this.data.version;
        } else if (this.chartData && this.chartData.version) {
            return this.chartData.version;
        }
        return {};
    };

    VersionReportModel.prototype.setRawVersionChartData = function (data) {
        if (data.version.released) {
            data.endTime = data.version.releaseDate;
            data.completeTime = data.endTime;
        }
        this.setRawChartData(data);
    };

    VersionReportModel.prototype.getXAxisEnd = function () {
        if (!_.isUndefined(this.xAxisEndDate)) {
            return this.xAxisEndDate;
        }
        return this.chartData.endTime;
    };

    VersionReportModel.prototype.getYAxes = function () {
        var yAxes = [];
        yAxes.push(this.yAxisData);
        if (!this.isIssueCountStatistic()) {
            yAxes.push({
                position: "right",
                min: 0,
                max: 100,
                tickFormatter: function (v, axis) {
                    // obliterate fractional part
                    return v.toFixed(0);
                },
                tickSize: 10
            });
        }
        return yAxes;
    };

    VersionReportModel.prototype.setRawData = function (data) {
        this.data = data;
        this.processData();
        this.generateJql();
    };

    VersionReportModel.prototype.generateJql = function () {
        var statistic = this.getStatistic();

        this.jql = {};
        var versionJql = ChartUtils.createVersionJql(this.getVersion());
        var notDoneJql = versionJql + " AND " + ChartUtils.createStatusJql(this.getNotDoneStatuses());

        if (this.isIssueCountStatistic()) {
            this.jql.incompleteEstimatedIssues = notDoneJql;
        } else {
            var statisticFieldId = statistic.fieldId;
            var match = /^customfield_(\d+)/.exec(statisticFieldId);
            var isCustomField = match && match.length > 0;

            if (isCustomField) {
                statisticFieldId = 'cf[' + match[1] + ']';
            }

            // "incomplete unestimated" issues will only ever include issues that CAN have an estimate, but do not.
            this.jql.incompleteUnestimatedIssues = notDoneJql + ' AND ' + statisticFieldId + ' IS EMPTY';

            // GHS-7191: "incomplete estimated" issues MAY include issues that CANNOT have an estimate. Therefore, we can't
            // always use JQL that contains the statistic field, because if the field is a custom field which only applies
            // to certain issue types, the correct results will not be returned by using "field IS NOT EMPTY".
            // in this case, fall back to a JQL string with issue keys.
            if (isCustomField) {
                if (this.data.contents.incompleteEstimatedIssues.length > 0) {
                    this.jql.incompleteEstimatedIssues = ChartUtils.createIssueJql(this.data.contents.incompleteEstimatedIssues);
                }
            } else {
                this.jql.incompleteEstimatedIssues = notDoneJql + ' AND ' + statisticFieldId + ' IS NOT EMPTY';
            }
        }

        this.jql.completedIssues = versionJql + " AND " + ChartUtils.createStatusJql(this.getDoneStatuses());
        this.jql.allIssues = versionJql;
    };

    /**
     * Overrides the method provided by ChangesReportModel to also add a lookup for data points for prediction series
     *
     * @param seriesId
     * @return {*}
     */
    VersionReportModel.prototype.getDataBySeries = function (seriesId) {
        var seriesData = ChangesReportModel.prototype.getDataBySeries.call(this, seriesId);
        if (seriesData) {
            return seriesData;
        }
        // try looking up id in predictions data
        if (this.isReleaseDatePredictionSeriesId(seriesId)) {
            seriesData = _.findWhere(this.series, {id: seriesId});
            if (seriesData && seriesData.data.length > 0) {
                // We only want a data point for the expected release date, so we create an empty array of series' size and only set the last element
                var predictedReleaseDateData = new Array(seriesData.data.length - 1);
                predictedReleaseDateData.push(_.last(seriesData.data));
                return predictedReleaseDateData;
            }
        }

        if (seriesId === "trendline") {
            seriesData = _.findWhere(this.series, {id: seriesId});
            if (seriesData) {
                return [_.first(seriesData.data)];
            }
        }

        if (seriesId === "unestimatedPercent") {
            seriesData = _.findWhere(this.series, {id: seriesId});
            if (seriesData && seriesData.data.length > 0) {
                return _.map(seriesData.data, function (val, idx, arr) {
                    var next = arr[idx + 1];
                    if (!_.isUndefined(next) && val[0] === next[0]) {
                        return;
                    }
                    return val;
                });
            }
        }

        return;
    };

    VersionReportModel.prototype.getDataBySeriesAtIndex = function (seriesId, index) {
        var series = this.getDataBySeries(seriesId);
        if (!series) {
            return;
        }

        if (seriesId !== "unestimatedPercent") {
            return series[index];
        }

        var pointData = series[index];
        if (!pointData) {
            return;
        }

        var totalIssuesData = ChangesReportModel.prototype.getDataBySeries.call(this, "issueCount");
        var totalData = _.findWhere(totalIssuesData, {time: pointData[0]});
        if (!_.isUndefined(totalData)) {
            return totalData;
        }

        var unestimatedIssuesData = ChangesReportModel.prototype.getDataBySeries.call(this, "unestimatedIssueCount");
        var unestimatedData = _.findWhere(unestimatedIssuesData, {time: pointData[0]});
        return unestimatedData;
    };

    VersionReportModel.prototype.getVersionSeries = function (excludeNonWorkingDays) {
        var version = this.getVersion();

        // if we did not add the trend, add it
        if (!this.attemptedTrendlinePrediction && _.isUndefined(_.findWhere(this.series, {id: "trendline"}))) {
            this.series = this.series.concat(this.getTrendlineSeries());
            this.attemptedTrendlinePrediction = true;
        }

        if (!this.isIssueCountStatistic() && _.isUndefined(_.findWhere(this.series, {id: "unestimatedPercent"}))) {
            this.series = this.series.concat(this.getUnestimatedSeries());
            this.series = _.reject(this.series, function (s) {
                return s.id === "issueCount" || s.id === "unestimatedIssueCount";
            });
        }

        // if it's already released, do not project
        if (version.released) {
            return this.series;
        }

        // if we already added the predictions, just return the series
        if (_.isUndefined(_.findWhere(this.series, {id: "prediction"}))) {
            // add predictions
            this.series = this.series.concat(this.getPredictedSeries());
            var trendlineSeries = _.findWhere(this.series, {id: "trendline"});
            if (!_.isUndefined(trendlineSeries)) {
                trendlineSeries.label = AJS.I18n.getText("gh.rapid.charts.version.predicted.release.date");
            }
        }

        return this.series;
    };

    VersionReportModel.prototype.getPredictedSeries = function () {
        var predictionData = this.getPredictionData();
        if (predictionData.xAxisEndDate) {
            this.xAxisEndDate = predictionData.xAxisEndDate;
        }

        var lastOptimisticPoint = _.last(predictionData.optimistic);
        var lastStandardPoint = _.last(predictionData.standard);
        var lastPessimisticPoint = _.last(predictionData.pessimistic);
        if (_.isUndefined(lastStandardPoint)) {
            return [];
        }

        var lineWidth0 = {lineWidth: 0};
        var lineWidth0Fill = {lineWidth: 0, fill: 0.3};
        var lineWidth2 = {lineWidth: 3};

        var prediction = {
            color: ChartColors.trend,
            data: predictionData.standard,
            id: "prediction",
            lines: lineWidth2
        };

        var optimistic = {
            color: ChartColors.trend,
            data: predictionData.optimistic,
            id: "optimistic",
            lines: lineWidth0Fill,
            fillBetween: "prediction"
        };

        var optimisticFiller = {
            color: ChartColors.trend,
            data: [lastOptimisticPoint, lastStandardPoint],
            id: "optimisticFiller",
            lines: lineWidth0Fill,
            fillBetween: "prediction",
            hoverable: false
        };

        var pessimistic = {
            color: ChartColors.trend,
            data: predictionData.pessimistic,
            id: "pessimistic",
            lines: lineWidth0Fill,
            fillBetween: "prediction",
            hoverable: false
        };

        var pessimisticFillerTop = {
            color: ChartColors.trend,
            data: [lastStandardPoint, lastPessimisticPoint],
            id: "pessimisticFillerTop",
            lines: lineWidth0,
            hoverable: false
        };

        var firstPessimisticPoint = _.first(predictionData.pessimistic);
        var standardProjectionOverPessimistic = ChartUtils.calculateYProjection(firstPessimisticPoint, lastPessimisticPoint, lastStandardPoint);
        var pessimisticFillerBottom = {
            color: ChartColors.trend,
            data: [[lastStandardPoint[0], standardProjectionOverPessimistic], lastPessimisticPoint],
            id: "pessimisticFillerBottom",
            lines: lineWidth0Fill,
            fillBetween: "pessimisticFillerTop"
        };

        var scope = {
            color: ChartColors.scope,
            data: predictionData.scope,
            id: "scope",
            lines: {
                lineWidth: 1,
                fill: 0.3
            }
        };

        return [scope, prediction, optimistic, optimisticFiller, pessimistic, pessimisticFillerBottom, pessimisticFillerTop];
    };

    VersionReportModel.prototype.getPredictionData = function () {
        var lastTotalEstimate = _.last(_.findWhere(this.series, {id: "totalEstimate"}).data);
        var lastCompleted = _.last(_.findWhere(this.series, {id: "completedEstimate"}).data);
        var estimatePerDay = this.getPredictedCompletedEstimatePerDay();

        return ReleaseDatePredictor.calculateSeries({
            startDate: this.getNow(),
            totalScope: lastTotalEstimate[1],
            totalCompleted: lastCompleted[1],
            estimatePerDay: estimatePerDay
        });
    };

    VersionReportModel.prototype.getPredictedCompletedEstimatePerDay = function () {

        var startTime = this.getVersionStartDate();
        var endTime = this.chartData.now + ChartUtils.oneDayInMillis;
        var seriesData = _.findWhere(this.series, {id: "completedEstimate"}).data;

        return ReleaseDatePredictor.getPredictedCompletedEstimatePerDay(seriesData, startTime, endTime, this.seriesData["completedEstimate"]);
    };

    VersionReportModel.prototype.getFurthestDateForNonWorkingDaysData = function () {

        if (_.isUndefined(this.series) || this.series.length == 0) {
            return Infinity;
        }

        var lastTotalEstimate = _.last(_.findWhere(this.series, {id: "totalEstimate"}).data);
        var lastCompleted = _.last(_.findWhere(this.series, {id: "completedEstimate"}).data);
        var remainingScope = lastTotalEstimate[1] - lastCompleted[1];

        if (remainingScope === 0) {
            return this.getXAxisEnd();
        }

        var estimatePerDay = this.getPredictedCompletedEstimatePerDay();

        // calculate the date - divide estimate by 5 to get a date 5 times as far out
        // TODO we need to reevaluate this approach - the non-working days data should be queries by the prediction calculation
        // on as needed basis, starting with big enough period (like 1m or more)
        if (estimatePerDay.pessimistic > 0) {
            return ReleaseDatePredictor.calculatePredictionDate(this.getVersionStartDate(), estimatePerDay.pessimistic / 5, remainingScope);
        }

        if (estimatePerDay.standard > 0) {
            return ReleaseDatePredictor.calculatePredictionDate(this.getVersionStartDate(), estimatePerDay.standard / 5, remainingScope);
        }

        return ReleaseDatePredictor.calculatePredictionDate(this.getVersionStartDate(), estimatePerDay.optimistic / 5, remainingScope);
    };

    VersionReportModel.prototype.getTrendlineSeries = function () {
        var issueHistoryData;
        if (this.isIssueCountStatistic()) {
            issueHistoryData = _.findWhere(this.series, {id: "totalEstimate"}).data;
        } else {
            issueHistoryData = _.findWhere(this.series, {id: "issueCount"}).data;
        }
        var firstIssueAdded = _.first(issueHistoryData);

        var completed = _.findWhere(this.series, {id: "completedEstimate"}).data;
        var lastCompleted = _.last(completed);

        var chartBeginDate = this.getVersionStartDate();
        var trendlineBegin = [chartBeginDate, 0];
        var trendlineEnd = lastCompleted;

        if (chartBeginDate > firstIssueAdded[0]) {
            // get the last value for the completed stories before the start date and start there
            var trendlinePos = _.chain(completed)
                .filter(function (data) {
                    return data[0] < chartBeginDate;
                })
                .last()
                .value();

            if (_.isUndefined(trendlinePos)) {
                trendlinePos = _.first(completed);
            }

            trendlineBegin = [chartBeginDate, trendlinePos[1]];
        }

        if (trendlineEnd[1] === 0 || trendlineEnd[1] <= trendlineBegin[1]) {
            return [];
        }

        return [{
            id: "trendline",
            color: ChartColors.trend,
            data: [trendlineBegin, trendlineEnd],
            lines: {lineWidth: 3},
            label: AJS.I18n.getText('gh.rapid.charts.progress.trend')
        }];
    };

    VersionReportModel.prototype.getUnestimatedSeries = function () {
        var totalIssuesData = _.findWhere(this.series, {id: "issueCount"}).data;
        var unestimatedIssuesData = _.findWhere(this.series, {id: "unestimatedIssueCount"}).data;

        var totalIssuesInitalValue = _.first(totalIssuesData)[1];
        var unestimatedIssuesInitialValue = _.first(unestimatedIssuesData)[1];

        function reduceSeries(data, stat) {
            return _.reduce(data, function (acc, el) {
                var nel = {
                    time: el[0]
                };
                nel[stat] = el[1];
                var last = _.last(acc);

                if (!_.isUndefined(last) && last.time === nel.time) {
                    last[stat] = nel[stat];
                } else {
                    acc.push(nel);
                }

                return acc;
            }, []);
        }

        var totalIssuesPoints = reduceSeries(totalIssuesData, "total");
        var unestimatedIssuePoints = reduceSeries(unestimatedIssuesData, "unestimated");

        var allPoints = _.sortBy(_.flatten([totalIssuesPoints, unestimatedIssuePoints], true), "time");
        var allPointsGrouped = _.reduce(allPoints, function (acc, el) {
            var last = _.last(acc);
            if (!_.isUndefined(last) && last.time == el.time) {
                _.defaults(last, el);
            } else {
                acc.push(el);
            }
            return acc;
        }, []);

        var pointsStepping = _.flatten(_.map(allPointsGrouped, function (el, idx, lst) {
            if (idx < lst.length - 1) {
                var nel = lst[idx + 1];
                return [el, _.defaults({time: nel.time}, el)];
            } else {
                return [el];
            }
        }), true);
        pointsStepping.unshift({
            time: pointsStepping[0].time,
            total: totalIssuesInitalValue,
            unestimated: unestimatedIssuesInitialValue
        });

        _.each(pointsStepping, function (el, idx, lst) {
            if (idx < 1) {
                return;
            }

            var prev = lst[idx - 1];
            if (_.isUndefined(el.total)) {
                el.total = prev.total;
            }
            if (_.isUndefined(el.unestimated)) {
                el.unestimated = prev.unestimated;
            }
        });

        var percentageSeries = _.map(pointsStepping, function (el) {
            return [el.time, el.total === 0 ? 0 : (el.unestimated / el.total) * 100];
        });

        return {
            id: "unestimatedPercent",
            label: AJS.I18n.getText('gh.rapid.charts.progress.unestimated.percent.legend'),
            color: ChartColors.unestimatedPercent,
            data: percentageSeries,
            yaxis: 2,
            lines: {
                lineWidth: 1
            }
        };
    };

    VersionReportModel.prototype.getChartOptions = function () {
        var options = ChartUtils.getChartOptions(this);
        options.grid.markings = this.getMarkings();
        return options;
    };

    VersionReportModel.prototype.getMarkings = function () {
        var model = this;
        return function (axes) {
            var markings = ChartController.getNonWorkingBlocks();
            var now = model.getNow();
            if (axes.xmax > now && (!model.getVersion().released || (model.getVersion().releaseDate <= axes.xmax))) {
                // mark "Now"
                // divide the y-axis into 50 dashes, each with a size:gap ratio 4:1, but without a gap at the top or bottom
                var tickSize = axes.ymax / 198;
                var dashSize = tickSize * 4;
                model.drawDashedMarkings(markings, now, dashSize, tickSize, 70);

                // mark release date
                var releaseDate = model.getVersion().releaseDate;
                if (!_.isUndefined(releaseDate)) {
                    dashSize = 2 * dashSize;
                    tickSize = 2 * tickSize;
                    model.drawDashedMarkings(markings, releaseDate, dashSize, tickSize, 20, ChartColors.releaseDate);
                }
            }

            return markings;
        };
    };

    VersionReportModel.prototype.drawDashedMarkings = function (markings, xAxisVal, dashSize, tickSize, max, color) {
        for (var i = 0; i < max; i++) {
            var dashStart = i * (dashSize + tickSize);
            markings.push({
                color: color || ChartColors.dashedMarking,
                lineWidth: 1,
                xaxis: {
                    from: xAxisVal,
                    to: xAxisVal

                },
                yaxis: {
                    from: dashStart,
                    to: dashStart + dashSize
                }
            });
        }
    };

    VersionReportModel.prototype.isReleaseDatePredictionSeriesId = function (seriesId) {
        switch (seriesId) {
            case "prediction" :
            case "optimistic" :
            case "pessimisticFillerBottom" :
                return true;
        }
        return false;
    };

    VersionReportModel.prototype.getStartTime = function () {
        if (this.chartData.startTime) {
            return this.chartData.startTime;
        }
        return this.getNow();
    };

    VersionReportModel.prototype.getVersionStartDate = function () {
        if (this.getVersion().startDate) {
            return this.getVersion().startDate;
        }
        return this.getStartTime();
    };

    VersionReportModel.prototype.notStarted = function () {
        return this.getVersionStartDate() > this.getNow();
    };

    VersionReportModel.prototype.getXAxisStart = function () {
        // add a percent of the width at the left side, otherwise the start of the epic kisses the left y axis
        var leftSideBuffer = 0.02;
        var startTime = this.chartData.startTime;
        if (this.getVersion().startDate) {
            startTime = this.getVersion().startDate;
        }
        return startTime - ((this.chartData.endTime - startTime) * leftSideBuffer);
    };

    return VersionReportModel;
});
