/* global module, test, deepEqual, equal, ok, strictEqual, GH, _ */

AJS.test.require('com.pyxis.greenhopper.jira:gh-rapid', function () {
    module('BurnupTransformer tests', {
        setup: function setup() {
            this.service = GH.Reports.SprintBurnupTransformer;
        },

        emptyClosedSprint: function emptyClosedSprint() {
            return {
                changes: {},
                startTime: 1000,
                endTime: 10000,
                completeTime: 10000,
                sprintCompleted: true,
                now: 20000,
                statisticField: {
                    typeId: 'field',
                    fieldId: 'customfield_10002',
                    id: 'field_customfield_10002',
                    name: 'Story Points',
                    isValid: true,
                    isEnabled: true,
                    renderer: 'number'
                },
                issueToParentKeys: {},
                workRateData: {
                    timezone: 'Australia/Sydney',
                    rates: [{
                        start: 0,
                        end: 10000,
                        rate: 1
                    }]
                },
                openCloseChanges: {
                    10000: [{
                        userDisplayNameHtml: '<a class=\'user-hover\' rel=\'admin\' id=\'_admin\' href=\'/jira/secure/ViewProfile.jspa?name=admin\'>admin</a>',
                        operation: 'CLOSE'
                    }]
                },
                lastUserWhoClosedHtml: '<a class=\'user-hover\' rel=\'admin\' id=\'_admin\' href=\'/jira/secure/ViewProfile.jspa?name=admin\'>admin</a>',
                rapidViewId: 0,
                issueToSummary: []
            };
        },

        /**
         * Various functions for creating the change entries returned from the server.
         */
        addToSprint: function addToSprint(key) {
            return {
                key: key,
                added: true
            };
        },
        removeFromSprint: function removeFromSprint(key) {
            return {
                key: key,
                added: false
            };
        },
        moveToDone: function moveToDone(key) {
            return {
                key: key,
                column: { notDone: false }
            };
        },
        moveToNotDone: function moveToNotDone(key) {
            return {
                key: key,
                column: { notDone: true }
            };
        },
        changeEstimate: function changeEstimate(key, estimate) {
            return {
                key: key,
                statC: {
                    newValue: estimate
                }
            };
        },
        /**
         * Generating a single change that:
         *     - creates an issue,
         *     - gives it an estimate,
         *     - adds it to the ScopeBurndown
         */
        addAndEstimate: function addAndEstimate(key, estimate) {
            return [_.extend(this.moveToNotDone(key), this.changeEstimate(key, estimate), this.addToSprint(key))];
        },
        /**
         * Generating a single change that:
         *     - creates an issue,
         *     - adds it to the ScopeBurndown
         */
        addWithoutEstimate: function addWithoutEstimate(key) {
            return [_.extend(this.moveToNotDone(key), this.addToSprint(key))];
        }
    });

    test('Empty sprint', function () {
        var data = this.emptyClosedSprint();
        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result, {
            now: data.now,
            start: data.startTime,
            startKeys: [],
            end: data.endTime,
            sprintCompleted: true,
            endKeys: [],
            nonWorkPeriods: [],
            // empty data gets a 0 point till start till end
            scope: [{
                timestamp: data.startTime,
                value: 0,
                change: {
                    fromValue: undefined,
                    key: undefined,
                    line: 'scope',
                    toValue: undefined,
                    type: 'start event'
                }
            }, {
                timestamp: data.endTime,
                value: 0,
                change: {
                    fromValue: undefined,
                    key: undefined,
                    line: 'scope',
                    toValue: undefined,
                    type: 'end event'
                }
            }],
            // empty data gets a 0 point till start till end
            work: [{
                timestamp: data.startTime,
                value: 0,
                change: {
                    fromValue: undefined,
                    key: undefined,
                    line: 'done',
                    toValue: undefined,
                    type: 'start event'
                }
            }, {
                timestamp: data.endTime,
                value: 0,
                change: {
                    fromValue: undefined,
                    key: undefined,
                    line: 'done',
                    toValue: undefined,
                    type: 'end event'
                }
            }],
            // copied directly from input data
            statisticField: data.statisticField,
            rapidViewId: data.rapidViewId,
            estimatedIssueEvents: 0,
            issueToSummary: []
        });
    });

    test('Changes sorted by timestamp', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            1300: this.addAndEstimate('KEY-3', 20),
            1200: this.addAndEstimate('KEY-2', 20),
            1100: this.addAndEstimate('KEY-1', 20)
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1100,
            value: 20,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'scope',
                toValue: 20,
                type: 'issue added'
            }
        }, {
            timestamp: 1200,
            value: 40,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'scope',
                toValue: 20,
                type: 'issue added'
            }
        }, {
            timestamp: 1300,
            value: 60,
            change: {
                fromValue: 0,
                key: 'KEY-3',
                line: 'scope',
                toValue: 20,
                type: 'issue added'
            }
        }, {
            timestamp: data.endTime,
            value: 60,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('Some events with estimates', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            1100: this.addAndEstimate('KEY-2', 20), // event with estimate
            1200: this.addAndEstimate('KEY-3', 20) // event with estimate
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        // 2 events with estimates, +1 because an event with non-zero estimate is created for us at the end of the sprint
        equal(result.estimatedIssueEvents, 3);
    });

    test('Some events without estimates', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            1100: this.addWithoutEstimate('KEY-2'),
            1200: this.addWithoutEstimate('KEY-3')
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        // no events had estimates
        equal(result.estimatedIssueEvents, 0);
    });

    test('End dates', function () {
        var data = this.emptyClosedSprint();

        // active sprint
        data.startTime = 0;
        data.now = 1000;
        data.completeTime = undefined;
        data.endTime = 5000;

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));
        equal(data.endTime, result.end);
        equal(data.now, result.now);

        // complete after end time
        data.completeTime = 10000;
        data.endTime = 5000;

        result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));
        equal(data.completeTime, result.end);

        // complete before end time
        data.completeTime = 500;
        data.endTime = 5000;

        result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));
        equal(data.completeTime, result.end);

        // now before endTime sprint
        data.startTime = 0;
        data.now = 1000;
        data.completeTime = undefined;
        data.endTime = 5000;

        result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));
        equal(data.now, result.now);

        // now after endTime sprint
        data.startTime = 0;
        data.now = 5000;
        data.completeTime = undefined;
        data.endTime = 5000;

        result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));
        equal(data.now, result.now);
    });

    test('issues with estimates added before sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            // timestamps before sprint start
            400: this.addWithoutEstimate('KEY-0'),
            500: this.addAndEstimate('KEY-1', 10),
            700: this.addAndEstimate('KEY-2', 20)
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 30,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 30,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue with estimate added after start', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 10),
            // after start
            1700: this.addAndEstimate('KEY-2', 20)
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 10,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 30,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'scope',
                toValue: 20,
                type: 'issue added'
            }
        }, {
            timestamp: data.endTime,
            value: 30,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue without estimate added after start', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 10),
            // after start
            1700: this.addWithoutEstimate('KEY-2')
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 10,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 10,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'scope',
                toValue: 0,
                type: 'issue added'
            }
        }, {
            timestamp: data.endTime,
            value: 10,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue estimate updated before sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addWithoutEstimate('KEY-1'),
            700: [this.changeEstimate('KEY-1', 5)],
            900: [this.changeEstimate('KEY-1', 15)]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 15,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 15,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue estimate updated in sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addWithoutEstimate('KEY-1'),
            // after start
            1700: [this.changeEstimate('KEY-1', 5)],
            3400: [this.changeEstimate('KEY-1', 15)]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 5,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'scope',
                toValue: 5,
                type: 'estimate update'
            }
        }, {
            timestamp: 3400,
            value: 15,
            change: {
                fromValue: 5,
                key: 'KEY-1',
                line: 'scope',
                toValue: 15,
                type: 'estimate update'
            }
        }, {
            timestamp: data.endTime,
            value: 15,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue estimate updated after sprint', function () {
        // should be excluded from the chart

        var data = this.emptyClosedSprint();
        data.startTime = 1000;
        data.endTime = 10000;

        data.changes = {
            500: this.addWithoutEstimate('KEY-1'),
            // after start
            1700: [this.changeEstimate('KEY-1', 5)],
            3400: [this.changeEstimate('KEY-1', 15)],
            //after end
            13400: [this.changeEstimate('KEY-1', 0)]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 5,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'scope',
                toValue: 5,
                type: 'estimate update'
            }
        }, {
            timestamp: 3400,
            value: 15,
            change: {
                fromValue: 5,
                key: 'KEY-1',
                line: 'scope',
                toValue: 15,
                type: 'estimate update'
            }
        }, {
            timestamp: data.endTime,
            value: 15,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue with estimate removed from sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            // after start
            1700: [this.removeFromSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 0,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'scope',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue without estimate removed from sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            600: this.addWithoutEstimate('KEY-2'),
            // after start
            1700: [this.removeFromSprint('KEY-2')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 9,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'scope',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: data.endTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue not in sprint removed from sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            // after start
            1700: [this.removeFromSprint('NEW-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue removed then estimate updated', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            600: this.addAndEstimate('KEY-2', 9),
            // after start
            1700: [this.removeFromSprint('KEY-1')],
            1800: [this.changeEstimate('KEY-1', 20)],
            1900: [this.addToSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 18,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 9,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'scope',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: 1900,
            value: 29,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'scope',
                toValue: 20,
                type: 'issue added'
            }
        }, {
            timestamp: data.endTime,
            value: 29,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue removed, then added then estimate updated', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            600: this.addAndEstimate('KEY-2', 9),
            // after start
            1700: [this.removeFromSprint('KEY-1')],
            1800: [this.addToSprint('KEY-1')],
            1900: [this.changeEstimate('KEY-1', 20)]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 18,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 9,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'scope',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: 1800,
            value: 18,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'scope',
                toValue: 9,
                type: 'issue added'
            }
        }, {
            timestamp: 1900,
            value: 29,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'scope',
                toValue: 20,
                type: 'estimate update'
            }
        }, {
            timestamp: data.endTime,
            value: 29,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('negative estimate', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', -1)
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('work completed during sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            600: this.addAndEstimate('KEY-2', 9),
            // after start
            1700: [this.moveToDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 9,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'done',
                toValue: 9,
                type: 'issue done'
            }
        }, {
            timestamp: data.endTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('work completed before sprint', function () {

        // any issue which is completed at the start of a sprint is excluded from the scope and work

        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            600: [this.moveToDone('KEY-1')],
            800: this.addAndEstimate('KEY-2', 9),
            //after start
            1600: [this.moveToDone('KEY-2')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1600,
            value: 9,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'done',
                toValue: 9,
                type: 'issue done'
            }
        }, {
            timestamp: data.endTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('work completed then uncompleted before sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            600: [this.moveToDone('KEY-1')],
            700: [this.moveToNotDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('repeated state updates on incomplete issue', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            // after start
            1800: [this.moveToNotDone('KEY-1')],
            1900: [this.changeEstimate('KEY-1', 9)],
            2000: [this.addToSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('repeated state updates on complete issue', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            600: [this.moveToDone('KEY-1')],
            // after start
            1900: [this.changeEstimate('KEY-1', 9)],
            1800: [this.moveToDone('KEY-1')],
            2000: [this.removeFromSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('work completed then uncompleted during sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            600: this.addAndEstimate('KEY-2', 9),
            // after start
            1700: [this.moveToDone('KEY-1')],
            1800: [this.moveToNotDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 9,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'done',
                toValue: 9,
                type: 'issue done'
            }
        }, {
            timestamp: 1800,
            value: 0,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'done',
                toValue: 0,
                type: 'issue not done'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 18,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 18,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('work completed then estimated updated during sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addAndEstimate('KEY-1', 9),
            600: this.addAndEstimate('KEY-2', 9),
            // after start
            1700: [this.moveToDone('KEY-1')],
            1800: [this.changeEstimate('KEY-1', 1)]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 9,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'done',
                toValue: 9,
                type: 'issue done'
            }
        }, {
            timestamp: 1800,
            value: 1,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'done',
                toValue: 1,
                type: 'estimate update'
            }
        }, {
            timestamp: data.endTime,
            value: 1,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 18,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1800,
            value: 10,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'scope',
                toValue: 1,
                type: 'estimate update'
            }
        }, {
            timestamp: data.endTime,
            value: 10,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('unestimated completed issue estimated during sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            500: this.addWithoutEstimate('KEY-1'),
            600: this.addAndEstimate('KEY-2', 9),
            // after start
            1700: [this.moveToDone('KEY-1')],
            1800: [this.changeEstimate('KEY-1', 1)]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 0,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'done',
                toValue: 0,
                type: 'issue done'
            }
        }, {
            timestamp: 1800,
            value: 1,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'done',
                toValue: 1,
                type: 'estimate update'
            }
        }, {
            timestamp: data.endTime,
            value: 1,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1800,
            value: 10,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'scope',
                toValue: 1,
                type: 'estimate update'
            }
        }, {
            timestamp: data.endTime,
            value: 10,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('completed issue with estimate removed from sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            600: this.addAndEstimate('KEY-1', 9),
            // after start
            1700: [this.moveToDone('KEY-1')],
            1800: [this.removeFromSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 9,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'done',
                toValue: 9,
                type: 'issue done'
            }
        }, {
            timestamp: 1800,
            value: 0,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'done',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1800,
            value: 0,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'scope',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('completed issue without estimate removed from sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            600: this.addAndEstimate('KEY-1', 9),
            700: this.addWithoutEstimate('KEY-2'),
            // after start
            1700: [this.moveToDone('KEY-2')],
            1800: [this.removeFromSprint('KEY-2')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 0,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'done',
                toValue: 0,
                type: 'issue done'
            }
        }, {
            timestamp: 1800,
            value: 0,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'done',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }

        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1800,
            value: 9,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'scope',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: data.endTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('completed issue estimated during sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            600: this.addAndEstimate('KEY-1', 9),
            700: this.addAndEstimate('KEY-2', 0),
            // after start
            1700: [this.moveToDone('KEY-2')],
            1800: [this.changeEstimate('KEY-2', 1)]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1700,
            value: 0,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'done',
                toValue: 0,
                type: 'issue done'
            }
        }, {
            timestamp: 1800,
            value: 1,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'done',
                toValue: 1,
                type: 'estimate update'
            }
        }, {
            timestamp: data.endTime,
            value: 1,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1800,
            value: 10,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'scope',
                toValue: 1,
                type: 'estimate update'
            }
        }, {
            timestamp: data.endTime,
            value: 10,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('completed issue with estimate added during sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            600: this.addAndEstimate('KEY-1', 9),
            // after start
            1600: this.addAndEstimate('KEY-2', 1).concat([this.moveToDone('KEY-2')])
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1600,
            value: 1,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'done',
                toValue: 1,
                type: 'issue done'
            }
        }, {
            timestamp: data.endTime,
            value: 1,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1600,
            value: 10,
            change: {
                fromValue: 0,
                key: 'KEY-2',
                line: 'scope',
                toValue: 1,
                type: 'issue added'
            }
        }, {
            timestamp: data.endTime,
            value: 10,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue removed from sprint, estimate updated, then done, then added to sprint', function () {
        var data = this.emptyClosedSprint();

        // anything that happens while the issue is not in the sprint should be track but not reflected on the chart.
        //  i.e. if the issue returns to the sprint it should have the updates that happened while it wasn't in the sprint.

        data.startTime = 1000;
        data.changes = {
            600: this.addAndEstimate('KEY-1', 9),
            // after start
            1600: [this.removeFromSprint('KEY-1')],
            1700: [this.changeEstimate('KEY-1', 5)],
            1800: [this.moveToDone('KEY-1')],
            1900: [this.addToSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1900,
            value: 5,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'done',
                toValue: 5,
                type: 'issue added'
            }
        }, {
            timestamp: data.endTime,
            value: 5,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1600,
            value: 0,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'scope',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: 1900,
            value: 5,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'scope',
                toValue: 5,
                type: 'issue added'
            }
        }, {
            timestamp: data.endTime,
            value: 5,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue complete, removed from sprint, then uncompleted, then added to sprint', function () {
        var data = this.emptyClosedSprint();

        // anything that happens while the issue is not in the sprint should be track but not reflected on the chart.
        //  i.e. if the issue returns to the sprint it should have the updates that happened while it wasn't in the sprint.

        data.startTime = 1000;
        data.changes = {
            600: this.addAndEstimate('KEY-1', 9),
            // after start
            1500: [this.moveToDone('KEY-1')],
            1600: [this.removeFromSprint('KEY-1')],
            1700: [this.moveToNotDone('KEY-1')],
            1900: [this.addToSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1500,
            value: 9,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'done',
                toValue: 9,
                type: 'issue done'
            }
        }, {
            timestamp: 1600,
            value: 0,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'done',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1600,
            value: 0,
            change: {
                fromValue: 9,
                key: 'KEY-1',
                line: 'scope',
                toValue: 0,
                type: 'issue removed'
            }
        }, {
            timestamp: 1900,
            value: 9,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'scope',
                toValue: 9,
                type: 'issue added'
            }
        }, {
            timestamp: data.endTime,
            value: 9,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('issue complete, estimated twice before sprint, then reopened during sprint', function () {
        var data = this.emptyClosedSprint();

        // anything that happens while the issue is not in the sprint should be track but not reflected on the chart.
        //  i.e. if the issue returns to the sprint it should have the updates that happened while it wasn't in the sprint.

        data.startTime = 1000;
        data.changes = {
            600: this.addAndEstimate('KEY-1', 1),
            700: this.addAndEstimate('KEY-1', 3),
            800: [this.moveToDone('KEY-1')],
            // after start
            1100: [this.moveToNotDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.work, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: data.endTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'done',
                toValue: undefined,
                type: 'end event'
            }
        }]);

        deepEqual(result.scope, [{
            timestamp: data.startTime,
            value: 0,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'start event'
            }
        }, {
            timestamp: 1100,
            value: 3,
            change: {
                fromValue: 0,
                key: 'KEY-1',
                line: 'scope',
                toValue: 3,
                type: 'issue not done'
            }
        }, {
            timestamp: data.endTime,
            value: 3,
            change: {
                fromValue: undefined,
                key: undefined,
                line: 'scope',
                toValue: undefined,
                type: 'end event'
            }
        }]);
    });

    test('unestimated issue reopened during sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.endTime = 10000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-1'),
            200: [this.moveToDone('KEY-1')],
            1300: [this.moveToNotDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result, {
            now: data.now,
            start: data.startTime,
            startKeys: [],
            end: data.endTime,
            endKeys: ['KEY-1'],
            sprintCompleted: true,
            issueToSummary: [],
            nonWorkPeriods: [],
            work: [{
                timestamp: data.startTime,
                value: 0,
                change: {
                    fromValue: undefined,
                    key: undefined,
                    line: 'done',
                    toValue: undefined,
                    type: 'start event'
                }
            }, {
                timestamp: data.endTime,
                value: 0,
                change: {
                    fromValue: undefined,
                    key: undefined,
                    line: 'done',
                    toValue: undefined,
                    type: 'end event'
                }
            }],
            scope: [{
                timestamp: data.startTime,
                value: 0,
                change: {
                    fromValue: undefined,
                    key: undefined,
                    line: 'scope',
                    toValue: undefined,
                    type: 'start event'
                }
            }, {
                timestamp: 1300,
                value: 0,
                change: {
                    fromValue: 0,
                    key: 'KEY-1',
                    line: 'scope',
                    toValue: 0,
                    type: 'issue not done'
                }
            }, {
                timestamp: data.endTime,
                value: 0,
                change: {
                    fromValue: undefined,
                    key: undefined,
                    line: 'scope',
                    toValue: undefined,
                    type: 'end event'
                }
            }],
            // copied directly from input data
            statisticField: data.statisticField,
            rapidViewId: data.rapidViewId,
            estimatedIssueEvents: 0
        });
    });

    /**
     * Tests for issue lists at start/end of sprint.
     */

    test('no issues at all', function () {
        var data = this.emptyClosedSprint();

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, []);
        strictEqual(result.endKeys.length, 0);
    });

    test('issue only added after sprint start', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            1100: this.addWithoutEstimate('KEY-1')
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, []);
        deepEqual(result.endKeys, ['KEY-1']);
    });

    test('removed issue before sprint start', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-1'),
            300: [this.removeFromSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, []);
        deepEqual(result.endKeys, []);
    });

    test('removed issue during sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.endTime = 10000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-1'),
            1300: [this.removeFromSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, ['KEY-1']);
        deepEqual(result.endKeys, []);
    });

    test('removed issue after sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.endTime = 10000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-1'),
            10300: [this.removeFromSprint('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, ['KEY-1']);
        deepEqual(result.endKeys, ['KEY-1']);
    });

    test('exclude issues done before sprint starts', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-1'),
            200: [this.moveToDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, []);
        deepEqual(result.endKeys, []);
    });

    test('include issues un-done before sprint starts', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.endTime = 10000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-1'),
            200: [this.moveToDone('KEY-1')],
            300: [this.moveToNotDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, ['KEY-1']);
        deepEqual(result.endKeys, ['KEY-1']);
    });

    test('include issues un-done after sprint starts', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.endTime = 10000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-1'),
            200: [this.moveToDone('KEY-1')],
            1300: [this.moveToNotDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, []);
        deepEqual(result.endKeys, ['KEY-1']);
    });

    test('include issues done during sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.endTime = 10000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-1'),
            1200: [this.moveToDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, ['KEY-1']);
        deepEqual(result.endKeys, ['KEY-1']);
    });

    test('include issues done after sprint', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.endTime = 10000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-1'),
            10200: [this.moveToDone('KEY-1')]
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, ['KEY-1']);
        deepEqual(result.endKeys, ['KEY-1']);
    });

    test('issue keys sorted for sprint start', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-7'),
            200: this.addWithoutEstimate('KEY-4'),
            300: this.addWithoutEstimate('KEY-5'),
            400: this.addWithoutEstimate('KEY-2'),
            500: this.addWithoutEstimate('KEY-3'),
            600: this.addWithoutEstimate('KEY-1'),
            800: this.addWithoutEstimate('KEY-18'),
            700: this.addWithoutEstimate('KEY-6')
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.startKeys, ['KEY-1', 'KEY-2', 'KEY-3', 'KEY-4', 'KEY-5', 'KEY-6', 'KEY-7', 'KEY-18']);
    });

    test('issue keys sorted for sprint end', function () {
        var data = this.emptyClosedSprint();

        data.startTime = 1000;
        data.changes = {
            100: this.addWithoutEstimate('KEY-7'),
            200: this.addWithoutEstimate('KEY-4'),
            300: this.addWithoutEstimate('KEY-5'),
            400: this.addWithoutEstimate('KEY-2'),
            500: this.addWithoutEstimate('KEY-3'),
            600: this.addWithoutEstimate('KEY-1'),
            800: this.addWithoutEstimate('KEY-18'),
            700: this.addWithoutEstimate('KEY-6')
        };

        var result = filter(this.service.getBurnupModelFromScopeBurndownModel(data));

        deepEqual(result.endKeys, ['KEY-1', 'KEY-2', 'KEY-3', 'KEY-4', 'KEY-5', 'KEY-6', 'KEY-7', 'KEY-18']);
    });

    test('current time after end time and sprint open', function () {
        var data = this.emptyClosedSprint();

        data.now = 5000;
        data.endTime = 1000;
        delete data.completeTime;

        // end should be the current time
        var model = this.service.getBurnupModelFromScopeBurndownModel(data).chartData;
        equal(5000, model.end);
        equal(false, model.sprintCompleted);
    });

    test('current time after end time and sprint closed', function () {
        var data = this.emptyClosedSprint();

        data.now = 5000;
        data.endTime = 10000;
        data.completeTime = 6000;

        // end should be sprint's completed time
        var model = this.service.getBurnupModelFromScopeBurndownModel(data).chartData;
        equal(6000, model.end);
        equal(true, model.sprintCompleted);
    });

    test('current time before end time and sprint open', function () {
        var data = this.emptyClosedSprint();

        data.now = 1000;
        data.endTime = 5000;
        delete data.completeTime;

        // end should be scheduled end time
        var model = this.service.getBurnupModelFromScopeBurndownModel(data).chartData;
        equal(5000, model.end);
        equal(false, model.sprintCompleted);
    });

    test('current time before end time and sprint closed', function () {

        var data = this.emptyClosedSprint();

        data.now = 1000;
        data.endTime = 5000;
        data.completeTime = 6000;

        // end should be sprint's completed time
        var model = this.service.getBurnupModelFromScopeBurndownModel(data).chartData;
        equal(6000, model.end);
        equal(true, model.sprintCompleted);
    });

    /**
     * extract the chart data that we want to test and clean out date formatters etc as we don't care about them.
     * @param raw
     * @returns {*}
     */
    var filter = function filter(raw) {
        // this mutates raw, but it's not a big deal as we're throwing it away anyway.
        var data = raw.chartData;
        delete data.formatDate;
        delete data.formatEventType;
        delete data.formatShortDate;
        delete data.formatStatistic;

        return data;
    };
});