/* global d3, c3, _, GH, moment */

GH.Reports = GH.Reports || {};
GH.Reports.controlChart = function () {
    'use strict';

    var chart = c3.component()
        .extend(function () {
            this.clusters(chart.data().getClusteredIssueSeries(20, this.xScale(), this.yScale()));
            // Hide any control chart dialogs on render
            if(chart.data().issues.length > 0) {
                scrubber.enableScrubber();
            } else {
                scrubber.disableScrubber();
            }
            GH.Reports.ControlChartDialog.hide();
        })
        .extend(c3.borderLayout())
        .extend(c3.utils.defaultViewBox)
        .extend({
            xDomain: function() {
                var domain = chart.data().bounds.x;
                _.each(domain, function(n) {
                    c3.checkIsNumber(n);
                });
                return domain;
            },
            yDomain: function() {
                var domain = chart.data().bounds.y;
                _.each(domain, function(n) {
                    c3.checkIsNumber(n);
                });
                return domain;
            },
            clusters: c3.prop()
        })
        .xScaleConstructor(d3.time.scale)
        .yScaleConstructor(function() {
            if (chart.data().bounds.y[1] < 30) {
                return d3.scale.linear();
            } else {
                return d3.scale.pow().exponent(1/3);
            }
        });

    var xAxis = c3.labelledAxis()
        .axisConstructor(function () {
            return d3.svg.axis()
                .tickFormat(function (d) {
                    var span = moment(chart.data().bounds.x[1]).diff(chart.data().bounds.x[0], 'days');
                    return moment(d).format(span < 2 ? 'LLL' : 'LL');
                });
        })
        .extend(function () {
            // Remove all ticks positioned before start of axis
            // Temporary workaround while we sort out bad Date object effects on d3
            this.selection().selectAll('g.tick').filter(function () {
                return this.getAttribute('transform').indexOf('translate(-') !== -1;
            }).remove();
        })
        .height(50)
        .text(AJS.I18n.getText('gh.rapid.charts.transitiondate'))
        .orient('bottom');

    var yAxis = c3.labelledAxis()
        .axisConstructor(function () {
            return d3.svg.axis()
                .tickFormat(function (d) {
                    return GH.NumberFormat.format(d);
                });
        })
        .width(60)
        .text(AJS.I18n.getText('gh.rapid.charts.days.elapsedtime'))
        .orient('left');

    var circlePlot = function () {
        return c3.circlePlot()
            .xAccessor(function (d) {
                return d.point[0];
            })
            .yAccessor(function(d) {
                return d.point[1];
            });
    };

    var clusterRadiusScale = d3.scale.log()
        .domain([2, 100])
        .range([6, 15])
        .clamp(true);

    var fillingRect = c3.fillWithRect()
        .elementClass('ghx-dragging');

    var scrubber = GH.Reports.ControlChartScrubber();

    var plots = c3.component()
        .extend(c3.interactive()
            .domContext(AJS.$("#ghx-chart-group"))
            .region(fillingRect)
        )
        .extend(c3.layerable())
        .extend(scrubber)
        .clipped(true)
        .addLayer('standard-deviation', c3.deviationPlot()
            .extend({
                data: function() {
                    return chart.data().series.standardDeviationArea;
                }
            })
            .elementClass('standard-deviation')
        )
        .addLayer('grid', c3.gridLines()
            .orient('left')
        )
        .addLayer('mean', c3.linePlot()
            .extend({
                data: function() {
                    return chart.data().series.meanLine;
                }
            })
            .elementClass('control-chart-mean')
        )
        .addLayer('aggregate-mean', c3.linePlot()
            .extend({
                data: function() {
                    return chart.data().series.rollingAverageLine;
                }
            })
            .elementClass('control-chart-aggregate-mean')
        )
        .addLayer('filler', fillingRect)
        .addLayer('issues', circlePlot()
            .extend({
                data: function() {
                    return chart.clusters().unclustered;
                }
            })
            .enter(function(event) {
                event.selection.on('click.dialog', function(d) {
                    GH.Reports.ControlChartDialog.show(this, d.issue);

                    // don't show the scrubber if the issue dialog is open
                    scrubber.hideScrubber();
                });
            })
            .dataKey(function(d) { return d.issue.key; })
            .elementClass('issue')
        )
        .addLayer('issue-clusters', circlePlot()
            .extend({
                data: function() {
                    return chart.clusters().clustered;
                }
            })
            .enter(function(event) {
                event.selection.on('click.dialog', function(d) {
                    GH.Reports.ControlChartDialog.show(this, d.issues);

                    // don't show the scrubber if the cluster dialog is open
                    scrubber.hideScrubber();
                });
            })
            .dataKey(function(d) { return d.point[0]; })
            .elementClass('cluster')
            .radiusAccessor(function (d) {
                return clusterRadiusScale(d.size);
            })
        );

    return chart.center(plots)
        .north(c3.withDimensions().height(10))
        .south(xAxis)
        .west(yAxis);
};