/* global clusterVectors */
define('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-data-service', ['require'], function(require) {
    "use strict";

    // REQUIRES
    var _ = require('underscore');
    var d3 = require('jira-agile/d3');
    var GlobalEvents = require('jira-agile/rapid/global-events');

    // GLOBALS... FIX ME
    var ClusterVectors;
    var Ajax;

    GlobalEvents.on('pre-initialization', function() {
        ClusterVectors = clusterVectors;
        Ajax = GH.Ajax;
    });

    /**
     * @typedef {object} ControlChartData
     * @property {ControlChartIssue[]} issues
     * @property {object} series
     * @property {object} stats
     * @property {{x: number[], y: number[]}} bounds
     * @property {number} earliestTime
     * @property {number} currentTime
     * @property {function} getClusteredIssueSeries
     */

    /**
     * @typedef {object} ControlChartIssue
     * @property {string} key
     * @property {string} summary
     * @property {number} endTime
     * @property {number} duration - total cycle time, based on active columns
     * @property {object[]} columns
     */

    var DAY = 24 * 60 * 60 * 1000;

    var durationAccessor = function (d) {
        return d.duration;
    };

    /**
     * Sends request to retrieve data
     *
     * @param {object} requestParams
     * @returns {Promise}
     */
    function fetchData(requestParams) {
        var opts = {
            url: '/rapid/charts/controlchart.json',
            data: requestParams
        };
        return Ajax.get(opts, 'rapidChartData');
    }

    /**
     * Converts an issue into a more easily consumable format.
     *
     * @param {object} issue
     * @param {object} data
     * @param {number[]} activeColumnIndices
     * @param {boolean} includeNonWorkingDays in all duration calculations
     * @returns {ControlChartIssue|null}
     */
    function processIssue(issue, data, activeColumnIndices, includeNonWorkingDays) {
        var endTime = -1;
        var totalDuration = 0;
        var columns = [];

        // sum the durations of the active columns in the issue
        // and determine the end date
        _.each(activeColumnIndices, function (activeIndex) {
            var leaveTime = issue.leaveTimes[activeIndex];
            if (leaveTime === data.currentTime) {
                return;
            }

            var duration = includeNonWorkingDays ?
                issue.totalTime[activeIndex] :
                issue.workingTime[activeIndex];

            totalDuration += duration;

            columns.push({
                name: data.columns[activeIndex].name,
                duration: duration
            });

            if (leaveTime > endTime) {
                endTime = leaveTime;
            }
        });

        if (endTime === -1) {
            return null;
        }

        return {
            key: issue.key,
            summary: issue.summary,
            endTime: endTime,
            duration: totalDuration,
            columns: columns
        };
    }

    /**
     * Upfront processing to convert issue data into a more easily consumable format.
     * Applies the non-working-days and columns filters.
     *
     * @param {object} data
     * @param {object} filterOptions
     * @returns {ControlChartIssue[]}
     */
    function processIssues(data, filterOptions) {
        var activeColumnIndices = [];

        filterOptions = filterOptions || {};

        // determine which columns to consider in the data
        _.each(data.columns, function (elem, i) {
            if (_.contains(filterOptions.refinements.selected.columnIds, elem.id)) {
                activeColumnIndices.push(i);
            }
        });

        var issues = _.compact(_.map(data.issues, function (issue) {
            return processIssue(issue, data, activeColumnIndices, filterOptions.viewingOptions.includeNonWorkingDays);
        }));

        return _.sortBy(issues, 'endTime');
    }

    /**
     * Calculates the standard deviation of an array given a mean
     *
     * @param {number} mean
     * @param {number[]} values
     * @param {Function} [accessor]
     * @returns {number}
     */
    function standardDeviation(mean, values, accessor) {
        var differenceSquareSum = 0;
        var difference = 0;
        _.each(values, function (value) {
            if (accessor) {
                value = accessor(value);
            }
            difference = value - mean;
            differenceSquareSum += (difference * difference);
        });
        var variance = differenceSquareSum / (values.length > 0 ? values.length : 1);
        return Math.sqrt(variance);
    }

    function getClusteredIssuePlotSeries(vectors, labels, clusterThreshold, xScale, yScale) {
        var clusteredVectors = new ClusterVectors(vectors, labels, clusterThreshold, xScale, yScale);

        return {
            clustered: _.filter(clusteredVectors, function (vector) {
                return vector.cluster;
            }),
            unclustered: _.reject(clusteredVectors, function (vector) {
                return vector.cluster;
            })
        };
    }

    function calculateRollingSeries(allIssues, issuesInTimeFrame, filterOptions) {
        var WINDOW_SIZE_MIN = 5;
        var WINDOW_SIZE_RATIO = 0.2;

        var result = {
            rollingAverageLine: [],
            standardDeviationArea: [],
            rollingAverageWindow: 0,
            maxStandardDeviation: 0
        };

        if (!issuesInTimeFrame.length) {
            return result;
        }

        var targetWindowSize = Math.max(WINDOW_SIZE_MIN, issuesInTimeFrame.length * WINDOW_SIZE_RATIO);
        var windowDelta = Math.floor((targetWindowSize - 1) / 2);
        result.rollingAverageWindow = windowDelta * 2 + 1;

        // Add 1 issue each to the beginning and end of the time frame, so that the rolling average and std dev
        // extends to the edge of the graph.
        var baseIndex = _.indexOf(allIssues, issuesInTimeFrame[0]);
        var fromIndex = issuesInTimeFrame[0].endTime > filterOptions.timeFrameTimes.from ?
            Math.max(0, baseIndex - 1) : baseIndex;
        var toIndex = _.last(issuesInTimeFrame).endTime < filterOptions.timeFrameTimes.to ?
            Math.min(allIssues.length, baseIndex + issuesInTimeFrame.length + 1) : baseIndex + issuesInTimeFrame.length;
        var expandedIssuesInTimeFrame = allIssues.slice(fromIndex, toIndex);

        _.each(expandedIssuesInTimeFrame, function (issue, i) {
            // Find the issues in the window
            var start = Math.max(0, fromIndex + i - windowDelta);
            var end = Math.min(allIssues.length, fromIndex + i + windowDelta + 1);
            var issuesInWindow = allIssues.slice(start, end);

            // Calculate mean and standard deviation on issues in the window
            var mean = d3.mean(issuesInWindow, durationAccessor);
            var meanInDays = mean / DAY;
            var stdDevInDays = standardDeviation(mean, issuesInWindow, durationAccessor) / DAY;
            result.rollingAverageLine.push([issue.endTime, meanInDays]);
            result.standardDeviationArea.push([issue.endTime, meanInDays, stdDevInDays]);
            result.maxStandardDeviation = Math.max(result.maxStandardDeviation, meanInDays + stdDevInDays);
        });

        // If the last point was within the selected timeframe,
        // extend the average line and std dev area to the edge of the selected timeframe.
        var lastRollingAverageLine = _.last(result.rollingAverageLine);
        var lastStandardDeviationArea = _.last(result.standardDeviationArea);
        if (lastRollingAverageLine[0] < filterOptions.timeFrameTimes.to) {
            result.rollingAverageLine.push([filterOptions.timeFrameTimes.to, lastRollingAverageLine[1]]);
            result.standardDeviationArea.push([filterOptions.timeFrameTimes.to, lastStandardDeviationArea[1], lastStandardDeviationArea[2]]);
        }

        return result;
    }

    function calculateMeanLine(mean, filterOptions) {
        mean /= DAY;
        return [
            [filterOptions.timeFrameTimes.from, mean],
            [filterOptions.timeFrameTimes.to, mean]
        ];
    }

    function calculateIssuesInTimeFrame(issues, filterOptions) {
        return _.filter(issues, function (issue) {
            // issue.endTime between timeframe from and to
            return filterOptions.timeFrameTimes.from <= issue.endTime && issue.endTime <= filterOptions.timeFrameTimes.to;
        });
    }

    function calculateSnapshotStats(issues, rollingAverageWindow) {
        return {
            mean: d3.mean(issues, durationAccessor),
            median: d3.median(issues, durationAccessor),
            min: d3.min(issues, durationAccessor),
            max: d3.max(issues, durationAccessor),
            rollingAverageWindow: rollingAverageWindow
        };
    }

    function calculateIssuePoints(issues) {
        return _.map(issues, function (issue) {
            return [issue.endTime, issue.duration / DAY];
        });
    }

    // TODO should be part of the control chart, not the data service
    function calculateBounds(issues, maxStandardDeviation, filterOptions) {
        var maxIssuePoint = Math.ceil(d3.max(issues, durationAccessor) / DAY);
        var maxPoint = Math.max(maxIssuePoint, maxStandardDeviation);
        var maxY = maxPoint * 1.1;

        return {
            x: [filterOptions.timeFrameTimes.from, filterOptions.timeFrameTimes.to],
            y: [0, maxY]
        };
    }

    /**
     * @contructor
     */
    function ControlChartDataService() {}

    // For testing
    ControlChartDataService._calculateRollingSeries = calculateRollingSeries;

    ControlChartDataService.prototype = {
        /**
         * Get control chart data with the specified filter options applied
         *
         * @param {number} rapidViewId
         * @param {object} [filterOptions]
         * @returns {Promise.<ControlChartData>}
         */
        get: function (rapidViewId, filterOptions) {
            filterOptions = filterOptions || {};

            return this._getData(rapidViewId, filterOptions).andThen(function (data) {
                var issues = processIssues(data, filterOptions);

                // Set timeFrame bounds based on the data, but clone the timeFrame to not cause side-effects
                var earliestTime = issues.length ? issues[0].endTime : data.currentTime;
                filterOptions.timeFrame = filterOptions.timeFrame.clone();
                filterOptions.timeFrame.setBounds(new Date(earliestTime), new Date(data.currentTime));

                // Simplified timeFrame info
                var timeFrameDates = filterOptions.timeFrame.getTimeFrameDates();
                filterOptions.timeFrameTimes = {
                    from: timeFrameDates.fromDate.getTime(),
                    to: timeFrameDates.toDate.getTime()
                };

                var issuesInTimeFrame = calculateIssuesInTimeFrame(issues, filterOptions);
                var rolling = calculateRollingSeries(issues, issuesInTimeFrame, filterOptions);
                var stats = calculateSnapshotStats(issuesInTimeFrame, rolling.rollingAverageWindow);
                var issuePoints = calculateIssuePoints(issuesInTimeFrame);

                return {
                    issues: issuesInTimeFrame,
                    series: {
                        meanLine: calculateMeanLine(stats.mean || 0, filterOptions),
                        rollingAverageLine: rolling.rollingAverageLine,
                        standardDeviationArea: rolling.standardDeviationArea
                    },
                    stats: stats,
                    bounds: calculateBounds(issuesInTimeFrame, rolling.maxStandardDeviation, filterOptions),
                    earliestTime: earliestTime,
                    currentTime: data.currentTime,
                    getClusteredIssueSeries: function (clusterThreshold, xScale, yScale) {
                        return getClusteredIssuePlotSeries(issuePoints, issuesInTimeFrame, clusterThreshold, xScale, yScale);
                    }
                };
            });
        },

        /**
         * Clear cached data
         */
        clearCache: function () {
            if (this._dataPromise) {
                this._dataPromise = null;
                this._dataRequestParams = null;
            }
        },

        /**
         * Gets raw data for given rapidViewId and filterOptions. Data could be retrieved from cache if available.
         *
         * @param {number} rapidViewId
         * @param {object} [filterOptions]
         * @returns {Promise}
         * @private
         */
        _getData: function (rapidViewId, filterOptions) {
            var requestParams = {
                rapidViewId: rapidViewId,
                swimlaneId: filterOptions.refinements.selected.swimlaneIds,
                quickFilterId: filterOptions.refinements.selected.quickFilterIds
            };
            // Only send a request if previous one cannot be used
            if (!this._dataPromise || this._dataPromise.state() === 'rejected' || !_.isEqual(this._dataRequestParams, requestParams)) {
                this._dataRequestParams = requestParams;
                this._dataPromise = fetchData(requestParams);
            }
            return this._dataPromise;
        }
    };

    return ControlChartDataService;
});
