/**
 * Represents the selection of refinements for the control chart.
 *
 * Unfortunately, this model has to be designed to work under 2 "stages":
 *  1. Before it has any knowledge of the available refinements (column, swimlane and quickFilter ids), due to
 *     the URL state management system requiring information about the refinements based on query params and localStorage.
 *  2. After it has received the available refinements, so that controls on the page can update the model based
 *     on user interaction.
 */
(function (_) {
    /**
     * @typedef {Object} RefinementModel
     * @property {number} id
     * @property {string} name
     */

    /**
     * @typedef {Object} RefinementsModel
     * @property {RefinementModel} columns
     * @property {RefinementModel} swimlanes
     * @property {RefinementModel} quickFilters
     */

    /**
     * @typedef {Object} RefinementsSelectionModel
     * @property {number[]} columnIds
     * @property {number[]} swimlaneIds
     * @property {number[]} quickFilterIds
     */

    /**
     * @constructor
     */
    function ControlChartRefinementsModel() {}

    ControlChartRefinementsModel.prototype = {
        /**
         * The available refinements to choose from.
         * Should be set by calling setAvailable().
         *
         * @type {RefinementsModel}
         */
        available: null,

        /**
         * The selected refinements.
         *
         * @type {RefinementsSelectionModel}
         */
        selected: null,

        setAvailable: function setAvailable(available) {
            this.available = available;
            // When the available refinements are first supplied,
            // it is possible to fill out any empty values.
            this.fromNormalized(this.selected);
            return this;
        },

        /**
         * The default refinements to select.
         *
         * @returns {RefinementsSelectionModel}
         */
        getDefaults: function getDefaults() {
            if (!this.available) {
                return null;
            }

            var availableColumnIds = _.pluck(this.available.columns, 'id');
            // Picks the default columns following the algorithm:
            // - the last column if less than 2 columns
            // - or else all columns except the first and last
            var defaultColumnIds = availableColumnIds.length < 3 ? [_.last(availableColumnIds)] : availableColumnIds.slice(1, -1);
            return {
                columnIds: defaultColumnIds,
                swimlaneIds: _.pluck(this.available.swimlanes, 'id'),
                quickFilterIds: []
            };
        },

        /**
         * Reset the model to the default state.
         *
         * @returns {ControlChartRefinementsModel}
         */
        resetToDefaults: function resetToDefaults() {
            this.selected = this.getDefaults();
            return this;
        },

        /**
         * Updates the model using the normalized representation of the model.
         * Missing refinements in the normalized model are filled with the defaults.
         *
         * @param {RefinementsSelectionModel} normalized
         * @returns {ControlChartRefinementsModel}
         */
        fromNormalized: function fromNormalized(normalized) {
            normalized = normalized ? _.pick(normalized, 'columnIds', 'swimlaneIds', 'quickFilterIds') : {};

            // Filter out any invalid ids if the available refinements have been supplied
            var selectedToAvailable = {
                columnIds: 'columns',
                swimlaneIds: 'swimlanes',
                quickFilterIds: 'quickFilters'
            };
            var available = this.available;
            if (available) {
                _.each(normalized, function (ids, key) {
                    var availableIds = _.pluck(available[selectedToAvailable[key]], 'id');
                    var validIds = _.intersection(normalized[key], availableIds);
                    if (validIds.length) {
                        normalized[key] = validIds;
                    } else {
                        // If all the ids are invalid, remove the refinement entirely
                        delete normalized[key];
                    }
                });
            }

            this.selected = _.defaults(normalized, this.getDefaults());
            return this;
        },

        /**
         * Convert model to a normalized representation.
         * Selected refinements are not included if they are equal to the default.
         *
         * @returns {RefinementsSelectionModel}
         */
        toNormalized: function toNormalized() {
            var defaults = this.getDefaults();
            var selected = this.selected;
            var normalized = {};

            if (!defaults) {
                return _.extend(normalized, selected);
            }

            function includeIfNotDefault(refinementName) {
                if (!_.isEqual(selected[refinementName].sort(), defaults[refinementName].sort())) {
                    normalized[refinementName] = selected[refinementName];
                }
            }

            includeIfNotDefault('columnIds');
            includeIfNotDefault('swimlaneIds');
            includeIfNotDefault('quickFilterIds');
            return normalized;
        },

        /**
         * Tests if the currently selected items on the model match the default state.
         *
         * @returns {boolean}
         */
        isDefault: function isDefault() {
            return _.isEmpty(this.toNormalized());
        }
    };

    GH.Reports.ControlChartRefinementsModel = ControlChartRefinementsModel;
})(_);