/* global Calendar, Date */
/**
 * Control Chart View. This view object manages the controls and the actual chart
 */
define('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-view', ['require'], function (require) {
    var ControlChartView = {};

    // REQUIRES
    var $ = require('jquery');
    var d3 = require('jira-agile/d3');
    var moment = require('jira/moment');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var ControlChartController;

    // GLOBALS... FIX ME
    var ChartView;
    var ControlChart;
    var ControlChartHowTo;
    var ControlChartLegend;
    var ControlChartRefinementsController;
    var Layout;
    var RapidViewConfig;
    var TimeFormat;
    var NumberFormat;
    var ChartTimeFrames;

    GlobalEvents.on('pre-initialization', function () {
        // circular dependency
        ControlChartController = require('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-controller');
        ControlChartRefinementsController = require('jira-agile/rapid/ui/chart/v2/controlchart/control-chart-refinements-controller');

        ChartView = GH.ChartView;
        ControlChart = GH.Reports.controlChart;
        ControlChartHowTo = GH.Reports.controlChartHowTo;
        ControlChartLegend = GH.Reports.controlChartLegend;
        Layout = GH.Layout;
        RapidViewConfig = GH.RapidViewConfig;
        TimeFormat = GH.TimeFormat;
        NumberFormat = GH.NumberFormat;
        ChartTimeFrames = GH.ChartTimeFrames;
    });

    ControlChartView.$timeWindowSelect = null;
    ControlChartView.$fromDateInput = null;
    ControlChartView.$toDateInput = null;
    ControlChartView.$timeFrameResetLink = null;

    ControlChartView.ControlChart = null;
    // Flag used by the view to check if the view is in the process of binding to the model. Changes to controls won't update the model.
    ControlChartView.bindingToModel = false;

    ControlChartView._firstRender = null;

    ControlChartView.init = function () {
        ControlChartView._firstRender = true;

        var swimlaneStrategy = RapidViewConfig.currentData.data.swimlaneStrategy;
        // render the containers
        ChartView.getChartContentElem(true).html(GH.tpl.reports.controlChart.renderChartBody({
            shouldShowSwimlaneRefine: swimlaneStrategy === 'custom'
        }));

        ChartView.getChartLegendD3(true).html(GH.tpl.reports.controlChart.renderLegend());
        $('#ghx-chart-header-secondary .ghx-chart-legend').show();

        // render the howto
        var howTo = new ControlChartHowTo();
        howTo(d3.select('.ghx-group-how'));

        // render the legend
        var legend = new ControlChartLegend();
        legend(d3.select('.ghx-chart-legend'));

        // render the controls
        ControlChartView.renderControls();
        ControlChartView.renderChartSelector();

        // setup the chart
        ControlChartView.ControlChart = new ControlChart();
        ControlChartView.bindToWindowResize();
    };

    ControlChartView.destroy = function () {
        ControlChartView.unbindToWindowResize();
    };

    /**
     * Renders the view.
     *
     * @param data
     * @param {GH.ChartTimeFrameModel} timeFrameModel
     */
    ControlChartView.render = function (data, timeFrameModel) {
        ControlChartView.bindModelToView(data, timeFrameModel);
        if (!ControlChartView._firstRender) {
            ChartView.showChartUpdatedToast();
        }
        ControlChartView._firstRender = false;
    };

    /**
     * Renders the chart
     */
    ControlChartView.renderChart = function () {
        ControlChartView.ControlChart(d3.select('#control-chart .ghx-svg-chart'));
    };

    ControlChartView.renderChartSelector = function () {
        ChartView.getChartSelector(true).html(GH.tpl.reports.controlChart.renderChartSelector());
    };

    /**
     * Binds the given ChartTimeFrameModel object to the chart header
     *
     * @param data
     * @param {GH.ChartTimeFrameModel} timeFrameModel
     */
    ControlChartView.bindModelToChartHeader = function (data, timeFrameModel) {
        // update the chart time window text
        var timeFrameDates = timeFrameModel.getTimeFrameDates();
        var dateFormat = 'LL';
        var timeWindowText = GH.tpl.reports.controlChart.renderChartTimeWindow({
            fromDate: moment(timeFrameDates.fromDate).format(dateFormat),
            toDate: moment(timeFrameDates.toDate).format(dateFormat),
            timeFrame: timeFrameModel.timeFrame
        });
        $('.js-chart-time-window').text(timeWindowText);

        var snapshotHtml = data.issues.length === 0 ? '' : GH.tpl.reports.controlChart.renderChartSnapshots({
            mean: TimeFormat.formatSensibleShortDuration(data.stats.mean),
            median: TimeFormat.formatSensibleShortDuration(data.stats.median),
            min: TimeFormat.formatSensibleShortDuration(data.stats.min),
            max: TimeFormat.formatSensibleShortDuration(data.stats.max),
            issueCount: NumberFormat.format(data.issues.length)
        });
        ChartView.getChartSnapshot(true).html(snapshotHtml);

        // update rolling average window length in legend
        var $rollingAverageWindow = $('.js-chart-legend-rollingaveragewindow');
        if (!data.stats.rollingAverageWindow) {
            $rollingAverageWindow.text(AJS.I18n.getText('gh.rapid.chart.legend.rollingaverage.notavailable'));
        } else {
            $rollingAverageWindow.text(AJS.I18n.getText('gh.rapid.chart.legend.rollingaverage.window', data.stats.rollingAverageWindow));
        }
    };

    /**
     * Binds the given ChartTimeFrameModel object to the chart's controls
     *
     * @param {GH.ChartTimeFrameModel} timeFrameModel
     */
    ControlChartView.bindModelToControls = function (timeFrameModel) {
        ControlChartView.bindingToModel = true;
        var timeFrameDates = timeFrameModel.getTimeFrameDates();
        var $resetBtn = ControlChartView.$timeFrameResetLink;

        // reset time window dropdown
        ControlChartView.$timeWindowSelect.val(timeFrameModel.getTimeFrame()).trigger('change');

        // set date values
        ControlChartView.updateCustomDatesFields(timeFrameDates.fromDate, timeFrameDates.toDate);

        // validate the dates
        ControlChartView.validateCustomDates();

        // toggle Reset link-button
        if (timeFrameModel.isDefault()) {
            $resetBtn.attr('aria-disabled', 'true').attr('title', AJS.I18n.getText('gh.rapid.timeframes.default'));
        } else {
            $resetBtn.removeAttr('aria-disabled title');
        }

        ControlChartView.bindingToModel = false;
    };

    /**
     * Binds the given ChartTimeFrameModel objet to the chart
     *
     * @param data
     */
    ControlChartView.bindModelToChart = function (data) {
        ControlChartView.ControlChart.data(data);
        ControlChartView.renderChart();
    };

    /**
     * Binds the model of the control chart to the view. This updates :
     *  - The chart's header with the active time frame
     *  - The chart's snapshot values
     *  - The time window select
     *  - The from and to date pickers
     *  - The chart itself
     *
     * @param data
     * @param {GH.ChartTimeFrameModel} timeFrameModel
     */
    ControlChartView.bindModelToView = function (data, timeFrameModel) {
        ControlChartView.bindModelToChart(data);
        ControlChartView.bindModelToChartHeader(data, timeFrameModel);
        ControlChartView.bindModelToControls(timeFrameModel);
    };

    ControlChartView.bindToWindowResize = function () {
        Layout.bindDelayedWindowResize();
        $(GH).bind(Layout.EVENT_DELAYED_RESIZE + '.' + ControlChartController.id, function () {
            ControlChartView.renderChart();
        });
    };

    ControlChartView.unbindToWindowResize = function () {
        $(GH).unbind(Layout.EVENT_DELAYED_RESIZE + '.' + ControlChartController.id);
    };

    /**
     * Shows a message in the chart area explaining the lack of data
     * @param {Object} filterOptions Current chart filters, used to customise message
     */
    ControlChartView.showBlankState = function (filterOptions) {
        var $message = $(GH.tpl.reports.controlChart.renderBlankStateMessage({
            canResetRefinements: !filterOptions.timeFrame.isDefault() || !filterOptions.refinements.isDefault()
        }));

        // Wire up in-message reset links
        $message.find('.js-options-reset').on('click', function (e) {
            ControlChartView.resetDefaultTimeWindow(e);
            ControlChartRefinementsController.resetToDefaults();
        });

        ChartView.showBlankStateMessage($message);
    };

    ControlChartView.hideBlankState = function () {
        ChartView.hideBlankStateMessage();
    };

    ControlChartView.showSpinner = function () {
        ChartView.showSpinner();
    };

    ControlChartView.hideSpinner = function () {
        ChartView.hideSpinner();
    };

    ControlChartView.handleTimeWindowChanged = function (event) {
        if (ControlChartView.bindingToModel) {
            return;
        }
        var timeWindow = ControlChartView.$timeWindowSelect.val();
        if (timeWindow === 'custom') {
            var customDates = ControlChartView.getCustomDates();
            ControlChartController.getFilterOptions().timeFrame.setSimpleDates(customDates.fromDate, customDates.toDate);
        } else {
            ControlChartController.getFilterOptions().timeFrame.setDays(+timeWindow);
        }
        ControlChartController.update();
    };

    ControlChartView.renderControls = function () {
        // Calendars trigger the change event on their respective input fields
        Calendar.setup({
            firstDay: 0,
            inputField: 'js-option-timeframe-custom-from',
            button: 'js-option-timeframe-custom-from_trigger_c',
            align: 'Tl',
            singleClick: true,
            cache: false,
            positionStyle: 'fixed',
            ifFormat: ChartTimeFrames.dateFormat,
            showsTime: false
        });
        Calendar.setup({
            firstDay: 0,
            inputField: 'js-option-timeframe-custom-to',
            button: 'js-option-timeframe-custom-to_trigger_c',
            align: 'Tl',
            singleClick: true,
            cache: false,
            positionStyle: 'fixed',
            ifFormat: ChartTimeFrames.dateFormat,
            showsTime: false
        });

        ControlChartView.$timeWindowSelect = $('#js-option-timeframe');
        ControlChartView.$fromDateInput = $('#js-option-timeframe-custom-from');
        ControlChartView.$toDateInput = $('#js-option-timeframe-custom-to');
        ControlChartView.$timeFrameResetLink = $('#js-option-timeframe-reset');

        ControlChartView.$timeWindowSelect.change(ControlChartView.handleTimeWindowChanged);
        ControlChartView.$fromDateInput.change(ControlChartView.handleCustomDateSelected);
        ControlChartView.$toDateInput.change(ControlChartView.handleCustomDateSelected);
        ControlChartView.$timeFrameResetLink.click(ControlChartView.resetDefaultTimeWindow);
    };

    ControlChartView.handleCustomDateSelected = function () {
        if (ControlChartView.bindingToModel) {
            return;
        }
        if (ControlChartView.validateCustomDates()) {
            ControlChartView.$timeWindowSelect.val('custom');
            var customDates = ControlChartView.getCustomDates();
            ControlChartController.getFilterOptions().timeFrame.setSimpleDates(customDates.fromDate, customDates.toDate);
            ControlChartController.update();
        }
    };

    /**
     * Returns the values of the two custom date pickers as Date objects or null if the values could not be parsed to a Date object.
     *
     * @return {{fromDate: Date, toDate: Date, fromText: String, toText: String}}
     */
    ControlChartView.getCustomDates = function () {
        var from = ControlChartView.$fromDateInput.val();
        var to = ControlChartView.$toDateInput.val();

        return {
            fromDate: Date.parseDate(from, ChartTimeFrames.dateFormat),
            toDate: Date.parseDate(to, ChartTimeFrames.dateFormat),
            fromText: from,
            toText: to
        };
    };

    ControlChartView.updateCustomDatesFields = function (from, to) {
        ControlChartView.$fromDateInput.val(from.print(ChartTimeFrames.dateFormat));
        ControlChartView.$toDateInput.val(to.print(ChartTimeFrames.dateFormat));
    };

    ControlChartView.resetDefaultTimeWindow = function (e) {
        if (ControlChartView.bindingToModel) {
            return;
        }
        ControlChartController.getFilterOptions().timeFrame.resetToDefault();
        ControlChartController.update();
        e.preventDefault();
    };

    /**
     * Validate the custom date values of the calendar controls.
     * Shows an error message if the dates are not valid.
     *
     * @returns {boolean} True if the dates are valid
     */
    ControlChartView.validateCustomDates = function () {
        var customDates = ControlChartView.getCustomDates();
        var result = true;
        var error = '';

        if (!customDates.fromDate || !customDates.toDate) {
            error = AJS.I18n.getText('gh.rapid.timeframes.error.invalid', ChartTimeFrames.getDateAsText(new Date()));
            result = false;
        } else if (!ChartTimeFrames.isValidDate(customDates.fromText) || !ChartTimeFrames.isValidDate(customDates.toText)) {
            error = AJS.I18n.getText('gh.rapid.timeframes.error.invalid', ChartTimeFrames.getDateAsText(new Date()));
            result = false;
        } else if (customDates.fromDate > customDates.toDate) {
            error = AJS.I18n.getText('gh.rapid.timeframes.error.reverse');
            result = false;
        }
        ControlChartView.setTimeFrameError(error);

        return result;
    };

    ControlChartView.setTimeFrameError = function (errorMessage) {
        $('.js-timeframes-error').html(errorMessage);
    };

    return ControlChartView;
});