if (!GH.Components) {
    GH.Components = {};
}

/**
 * Create a new QuickFilters component
 *
 * @param stateParamName the name to keep the selected filters state
 */
GH.Components.QuickFilters = function(stateParamName) {
    this.stateParamName = stateParamName;
};

GH.Components.QuickFilters.events = new GH.Events();

(function() {
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.Components.QuickFilters.analytics = new AnalyticsTracker('gh.rapidboard.quickFilters');
})();

GH.Components.QuickFilters.prototype.stateParamName = "activeQuickFilters";
GH.Components.QuickFilters.prototype.rapidViewConfig = {};

/**
 * Set the configuration of the current rapid view
 */
GH.Components.QuickFilters.prototype.setRapidViewConfig = function(rapidViewConfig) {
    this.rapidViewConfig = rapidViewConfig;

    // validate existing quick filters
    this.validateActiveFilters();
};

/**
 * Get the active quick filters
 */
GH.Components.QuickFilters.prototype.getActiveQuickFilters = function () {
    return GH.BoardState.getPerViewValue(this.stateParamName, []);
};

/**
 * Set the active quick filters
 */
GH.Components.QuickFilters.prototype.setActiveQuickFilters = function (activeQuickFilters) {
    GH.BoardState.setPerViewValue(this.stateParamName, activeQuickFilters);
};

/**
 * Validates the currently active quick filters and removes invalid filters
 */
GH.Components.QuickFilters.prototype.validateActiveFilters = function validateActiveFilters() {
    // fetch the active filters
    var activeFilters = this.getActiveQuickFilters();

    // filter out those that are invalid
    var self = this;
    var validFilters = _.filter(activeFilters, function(activeFilterId) {
        return _.find(self.rapidViewConfig.quickFilters, function(filter) {
            return activeFilterId == filter.id;
        });
    });

    // set the new values
    if (activeFilters.length != validFilters.length) {
        this.setActiveQuickFilters(validFilters);
        GH.RapidBoard.State.pushState();
    }
};

/**
 * Get all quick filters, including their active status.
 */
GH.Components.QuickFilters.prototype.getQuickFilters = function() {
    var activeFilters = this.getActiveQuickFilters();
    return _.map(this.rapidViewConfig.quickFilters, function(value) {
        value.isActive = _.contains(activeFilters, value.id);
        return value;
    });
};

GH.Components.QuickFilters.prototype.getInitiallyExpanded = function () {
    return GH.RapidBoard.State.getQuickFiltersExpanded();
};

GH.Components.QuickFilters.prototype.toggleInitiallyExpanded = function () {
    var expanded = !GH.RapidBoard.State.getQuickFiltersExpanded();
    GH.RapidBoard.State.setQuickFiltersExpanded(expanded);
};

GH.Components.QuickFilters.prototype.renderQuickFilters = function (mode) {
    var quickFilters = this.getQuickFilters();
    var html = GH.tpl.component.quickfilter.renderQuickFilters({
        quickFilters: quickFilters,
        canEdit: this.rapidViewConfig.canEdit,
        initiallyExpanded: this.getInitiallyExpanded(),
        thisMode: mode.mode
    });
    var elem = AJS.$(html);
    var self = this;
    var elemLabel = elem.find('#js-quickfilters-label');

    GH.Tooltip.tipsify({
        selector: elemLabel,
        gravity: 'w',
        className: 'ghx-tipsy-left'
    });

    var thisModeTrigger = '#js-'+ mode.mode + '-quickfilters-trigger';
    elem.find('.js-quickfilter-button').click(function(event) {
        var button = AJS.$(this);
        self.toggleQuickFilter(event, button);
    });

    var toggleFn = this.toggleInitiallyExpanded;
    elem.find(thisModeTrigger).click(function() {
        toggleFn();
    });
    elem.delegate(thisModeTrigger, 'aui-expander-expanded aui-expander-collapsed',GH.RapidBoard.ViewController.handleResizeEvent);

    return elem;
};
/**
 *
 * @fires GH.Components.QuickFilters.events#before:toggle as soon as a quickfilter button is clicked
 *      and the filter is about to toggle
 */
GH.Components.QuickFilters.prototype.toggleQuickFilter = function(event, button) {
    // prevent default, as href="#" would add a hash to the url/url change
    event.preventDefault();
    GH.Components.QuickFilters.events.trigger('before:toggle');
    var quickFilterId = parseInt(button.attr('data-filter-id'), 10);

    //Get the number encoded after the final dash in the button's id.
    this.toggleActiveQuickFilter(quickFilterId);
    button.toggleClass("ghx-active");

    // load the board again
    AJS.$(this).trigger('quickFilterChange');
    GH.Components.QuickFilters.analytics.trigger('toggle', 'activeFilterCount', this.getActiveQuickFilters().length);
};

GH.Components.QuickFilters.prototype.toggleActiveQuickFilter = function toggleActiveQuickFilter(quickFilterId) {
    // fetch the currently active quick filters
    if (quickFilterId) {
        var found = false;

        var activeFilters = this.getActiveQuickFilters();

        for (var i = 0, l = activeFilters.length; i < l; i++) {
            if (activeFilters[i] == quickFilterId) {
                activeFilters.splice(i, 1);
                found = true;
                break;
            }
        }
        if (!found) {
            activeFilters.push(quickFilterId);
        }

        // set the new values
        this.setActiveQuickFilters(activeFilters);
        GH.RapidBoard.State.pushState();
    }
};

GH.Components.QuickFilters.prototype.clearFilters = function clearFilters() {
    var currentFilters = this.getActiveQuickFilters();
    if (currentFilters.length == 0) {
        return false;
    } else {
        this.setActiveQuickFilters([]);
        AJS.$(".ghx-controls-filters .ghx-active").removeClass("ghx-active");
        return true;
    }
};

// Quick Filter expander toggle layout and visibilty
GH.Components.QuickFilters.prototype.quickExpanderLayout = function quickExpander(mode) {
    var modeExpanderId = '#js-'+ mode.mode +'-quickfilters';
    var $quickFilterContainer = AJS.$(modeExpanderId);
    var $quickFilterLabel = AJS.$($quickFilterContainer).find('dt');
    var $quickTrigger = $quickFilterContainer.find('.ghx-quickfilter-trigger');
    var $quickTriggerWidth = $quickTrigger.outerWidth();
    var $singleLineHeight = $quickTrigger.height() + parseInt($quickFilterLabel.css('marginBottom'),10) + 1;// magic number + 1 fixes Chrome calculation issue at 110%, safe elsewhere

    // remove the button and the space reserved for it
    $quickTrigger.css({'display':'none'});
    $quickFilterContainer.css({'padding-right':''});

    // check of the content of the expander container is bigger than one line
    if ($quickFilterContainer[0] && $quickFilterContainer[0].scrollHeight > $singleLineHeight){

        // some content is hidden - show the trigger
        $quickTrigger.css({'display':''});

        // if the container is closed the trigger needs space reserved for it
        if (!$quickFilterContainer.attr('aria-expanded') || $quickFilterContainer.attr('aria-expanded') == "false"){
            $quickFilterContainer.css({'padding-right':$quickTriggerWidth});
        }

    }
};
