/**
 * Detail fields edit model.
 *
 * Manages the field edit state for a single issue
 */
GH.DetailsFieldEditModel = function (issueId) {
    this.issueId = issueId;
    this.isLoading = false;
    this.editData = false;
    this.editInfos = [];
};

GH.DetailsFieldEditModel.prototype.getIssueId = function () {
    return this.issueId;
};

GH.DetailsFieldEditModel.prototype.setEditData = function (editData) {
    this.editData = editData;

    // process each field and potentially add an extra error handler
    _.each(editData.fields, function (field) {
        var errorSelector = GH.DetailsObjectFactory.getDetailsModel().getErrorSelectorForField(field.id);

        if (errorSelector) {
            field.errorSelector = errorSelector;
        }
    });
};

/**
 * Opens the editor for a given field
 */
GH.DetailsFieldEditModel.prototype.getFieldEditData = function (fieldId) {
    // find the field info inside the editData
    var fieldData = false;
    for (var i = 0; i < this.editData.fields.length; i++) {
        if (this.editData.fields[i].id == fieldId) {
            fieldData = this.editData.fields[i];
            break;
        }
    }
    return fieldData;
};

/**
 * Get the tab order for the fields contained in this model
 * @return {Array}
 */
GH.DetailsFieldEditModel.prototype.getFieldInTabOrder = function () {
    var fields = this.editData.fields;
    var addedIds = [];
    var orderedFields = [];

    // first push the summary field
    _.each(fields, function (field) {
        if (field.id == 'summary') {
            orderedFields.push(field);
            addedIds[field.id] = true;
        }
    });

    // then add the story points field if availabel
    _.each(fields, function (field) {
        if (field.id.indexOf("customfield_") === 0) {
            orderedFields.push(field);
            addedIds[field.id] = true;
        }
    });

    // then add the rest in current order
    _.each(fields, function (field) {
        if (!addedIds[field.id]) {
            orderedFields.push(field);
            addedIds[field.id] = true;
        }
    });
    return orderedFields;
};

//
// Currently editing fields information
//

/**
 * Get the number of currently edited fields
 */
GH.DetailsFieldEditModel.prototype.getEditingCount = function () {
    return this.editInfos.length;
};

/**
 * Get the number of currently edited fields
 */
GH.DetailsFieldEditModel.prototype.getEditingFieldIds = function () {
    if (this.getEditingCount() === 0) {
        return [];
    }

    return _.map(this.editInfos, function (editInfo) {
        return editInfo.fieldData.id;
    });
};

/**
 * Is the field edited?
 */
GH.DetailsFieldEditModel.prototype.isEditing = function (fieldId) {
    return this.getEditInfo(fieldId) !== false;
};

/**
 * Get the info object for a currently edited field
 */
GH.DetailsFieldEditModel.prototype.getEditInfo = function (fieldId) {
    for (var i = 0; i < this.editInfos.length; i++) {
        if (this.editInfos[i].fieldData.id == fieldId) {
            return this.editInfos[i];
        }
    }
    return false;
};

/**
 * Add an editInfo object
 */
GH.DetailsFieldEditModel.prototype.addEditInfo = function (editInfo) {
    this.editInfos.push(editInfo);
};

/**
 * Remove an editInfo object
 */
GH.DetailsFieldEditModel.prototype.removeEditInfo = function (editInfo) {
    this.editInfos = _.filter(this.editInfos, function (value) {
        return value.fieldData.id != editInfo.fieldData.id;
    });
};

/**
 * Clears the edit data
 */
GH.DetailsFieldEditModel.prototype.clearEditData = function () {
    this.editInfos = [];
};

GH.DetailsFieldEditModel.prototype.updateEditDataForField = function (fieldId, fieldData) {
    // update the edit data for the given field
    var editData = this.editData;
    if (editData) {
        for (var i = 0; i < editData.fields.length; i++) {
            if (editData.fields[i].id == fieldId) {
                editData.fields[i] = fieldData;
                break;
            }
        }
    }
};

/**
 * Tells all editors to submit their changes
 */
GH.DetailsFieldEditModel.prototype.completeEdits = function () {
    _.each(this.editInfos, function (editInfo) {
        editInfo.editor.submitChanges(editInfo);
    });
};

/**
 * Discards any ongoing edits
 */
GH.DetailsFieldEditModel.prototype.discardEdits = function () {
    _.each(this.editInfos, function (editInfo) {
        // TODO: editInfo should be a proper object
        editInfo.enabled = false;
    });
};