/* global _ */
/**
 * Details view component.
 */

(function () {
    GH.DetailsView = new GH.Events();

    GH.DetailsView.minWidth = 400;

    /**
     * Fired when the details pane is finished updating
     */
    GH.DetailsView.API_EVENT_DETAIL_VIEW_UPDATED = "GH.DetailView.updated";

    GH.DetailsView.EVENT_DETAIL_VIEW_CLOSED = "GH.DetailView.closed";

    GH.DetailsView.EVENT_DETAIL_VIEW_UPDATE_STARTED = "GH.DetailView.update.started";

    GH.DetailsView.selectedIssueKey = undefined;

    GH.DetailsView.rapidViewId = undefined;

    GH.DetailsView.rapidViewConfig = undefined;

    GH.DetailsView.containerSelector = undefined;

    GH.DetailsView.TAB_SUBTASKS = 'subtasks';

    GH.DetailsView.$lastSectionSpacer = undefined;

    /**
     * Options for this detail view
     */
    GH.DetailsView.opts = {
        canClose: true,
        showActionsCog: true,
        showSubtaskTab: true,
        showLogWorkTrigger: false,
        updateSizeHandler: function updateSizeHandler() {}
    };

    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    GH.DetailsView.analytics = new AnalyticsTracker('gh.issueaction.issuedetail');

    GH.DetailsView.triggerAnalyticsOpenDetailView = function () {
        // Trigger analytics event [SW-2193]
        AJS.trigger('analytics', { name: 'gh.issueaction.issuedetail.open' });
        GH.DetailsView.analytics.openDetailViewTime = Date.now();
    };

    GH.DetailsView.triggerAnalyticsCloseDetailView = function () {
        // Trigger analytics event [SW-2193]
        var openTime = GH.DetailsView.analytics.openDetailViewTime;
        var now = Date.now();
        // Open the detail view more than 3s
        if (!openTime || now - openTime > 3000) {
            AJS.trigger('analytics', { name: 'gh.issueaction.issuedetail.close' });
        } else {
            AJS.trigger('analytics', { name: 'gh.issueaction.issuedetail.dismiss' });
        }
        GH.DetailsView.analytics.openDetailViewTime = null;
    };

    GH.DetailsView.triggerAnalyticsEventForAction = function (action, data) {
        AJS.trigger('analytics', { name: 'gh.issueaction.issuedetail.' + action, data: data });
    };

    /**
     * initializes the detail view stuff
     */
    GH.DetailsView.init = function () {
        GH.DetailsView._initialisationPromise.done(function () {
            GH.DetailsObjectFactory.mergeDetailsViewTemplate();

            // tab selection
            AJS.$(document).bind("tabSelect", GH.DetailsView.handleTabSelection);

            // statistic editing might change when issue is updated
            AJS.$(GH).bind("issueUpdated", GH.DetailsView.handleIssueUpdated);

            AJS.$(document).delegate('.js-detailclose', "click", function (e) {
                if (GH.DetailsView.closeHandler) {
                    GH.DetailsView.closeHandler(e);
                    GH.DetailsView.triggerAnalyticsCloseDetailView();
                }
            });
            GH.DetailsView._postInit();

            // field edit functionality
            GH.DetailsFieldEdit.init();
        });
    };

    GH.DetailsView.getMode = function () {
        return GH.RapidBoard.ViewController.getMode();
    };

    GH.DetailsView.setContainerSelector = function (selector) {
        GH.DetailsView.containerSelector = selector;
        GH.DetailsView._initialisationPromise.done(function () {
            GH.DetailsView._onContainerSelectorSet(selector);
        });
    };

    GH.DetailsView._onContainerSelectorSet = function (selector) {};

    GH.DetailsView.setOptions = function (opts) {
        GH.DetailsView.opts = opts;
        GH.DetailsView.closeHandler = GH.DetailsView.opts.closeHandler;
    };

    /**
     * Get the container element for the detail view panel
     * @returns {jQuery}
     */
    GH.DetailsView.getContainer = function () {
        if (!GH.DetailsView.containerSelector) {
            // This error case happens on initial page load because ViewController calls hide()
            // on all modes except for the active one.
            // Both PlanController and WorkController will then attempt to hide the detail view.
            GH.log('tried to get container of DetailsView before selector was set');
            return AJS.$();
        }

        return AJS.$(GH.DetailsView.containerSelector);
    };

    /**
     * Get the container element for all content inside the detail view panel
     * @returns {jQuery}
     */
    GH.DetailsView._getContentContainer = function () {
        return GH.DetailsView.getContainer().find('.ghx-detail-contents');
    };

    /**
     * Get the root element representing the issue in detail view.
     * @returns {jQuery}
     */
    GH.DetailsView._getIssueElement = function () {
        return GH.DetailsView.getContainer().find('.ghx-detail-issue');
    };

    GH.DetailsView._replaceUpdatedFields = function ($issueElement, $newIssueElement, fieldId) {
        // Replace the field that was saved
        GH.DetailsView._replaceField($issueElement, $newIssueElement, fieldId);

        function isNotEditing(field) {
            return !_.contains(GH.DetailsView.getEditsInProgress(), field);
        }

        function isTimeTracking(field) {
            return _.contains(['timeestimate', 'timeoriginalestimate'], field);
        }

        if (fieldId === 'timeoriginalestimate' && isNotEditing('timeestimate')) {
            GH.DetailsView._replaceField($issueElement, $newIssueElement, 'timeestimate');
        }

        if (fieldId === 'timeestimate' && isNotEditing('timeoriginalestimate')) {
            GH.DetailsView._replaceField($issueElement, $newIssueElement, 'timeoriginalestimate');
        }

        if (isTimeTracking(fieldId)) {
            GH.DetailsView._replaceField($issueElement, $newIssueElement, 'aggregatetimeestimate');
        }
    };

    GH.DetailsView._replaceField = function ($oldIssueElement, $newIssueElement, fieldId) {
        var fieldSelector = GH.DetailsView._InlineEditor.getFieldSelector(fieldId);
        var $oldField = $oldIssueElement.find(fieldSelector);
        var $newField = $newIssueElement.find(fieldSelector);

        if ($newField.length === 0) {
            $oldField.closest('li, dl').remove();
        } else {
            $oldField.replaceWith($newField);
            GH.DetailsView._InlineEditor.triggerFieldUpdate(fieldId, $oldIssueElement, $newField);
        }
    };

    GH.DetailsView._setIssueElement = function ($newIssueElement, savedFieldId) {
        var $issueElement = GH.DetailsView._getIssueElement();

        if ($issueElement.length === 0) {
            GH.DetailsView._getContentContainer().prepend($newIssueElement);
            return;
        }

        if (savedFieldId && GH.DetailsView.hasEditsInProgress()) {
            // If there are edits in progress, we can't just replace the whole details view, because the user
            // will lose their unsaved changes. So we only replace the saved field and any other fields which we think may have updated.
            GH.DetailsView._replaceUpdatedFields($issueElement, $newIssueElement, savedFieldId);
        } else {
            $issueElement.replaceWith($newIssueElement);
        }
    };

    /**
     * Set the current rapid view id
     */
    GH.DetailsView.setRapidViewId = function (rapidViewId) {
        GH.DetailsView.rapidViewId = rapidViewId;
    };

    /**
     * Set the current rapid view config
     */
    GH.DetailsView.setRapidViewConfig = function (rapidViewConfig) {
        GH.DetailsView.rapidViewConfig = rapidViewConfig;

        // also update GH.DetailsView.opts to consider information from config
        // showLogWorkTrigger also depends on whether or not the tracking statistic is enabled for board
        // and it ALSO depends on whether or not the issue/user is allowed to log work, but that comes later
        if (GH.DetailsView.opts.showLogWorkTrigger) {
            GH.DetailsView.opts.showLogWorkTrigger = !_.isUndefined(rapidViewConfig.trackingStatistic) && rapidViewConfig.trackingStatistic.isEnabled;
        }
    };

    /**
     * Set the displayed issue
     */
    GH.DetailsView.setSelectedIssueKey = function (issueKey) {
        GH.DetailsView.selectedIssueKey = issueKey;
    };
    GH.DetailsView.clearSelectedIssueKey = function () {
        GH.DetailsView.selectedIssueKey = undefined;
    };

    /**
     * Renders the details view
     * @fires GH.DetailsView#before:render if the view panel is about to be rendered
     */
    GH.DetailsView.show = function (detailsViewLoadReason) {

        GH.DetailsView._initialisationPromise.done(function () {
            // finish any previous edits
            GH.DetailsFieldEdit.completeEdits();

            // TODO: this is incorrect! we delete current view data before having new view data around!
            // In case loading of new data fails we still show the current details, but the underlying data has gone.
            // ensure we don't keep stale data around
            if (!detailsViewLoadReason) {
                GH.DetailsObjectFactory.getDetailsModel().clearViewData();
            }

            // set up DOM
            var container = GH.DetailsView.getContainer();
            if (container.find('#ghx-detail-contents').length === 0) {
                var skeleton = GH.tpl.detailview.renderIssueDetailsSkeleton();
                container.empty().append(skeleton);
                container.width(GH.DetailsView.getWidth());
                GH.DetailsView.initTooltips();

                GH.DetailsView.triggerAnalyticsOpenDetailView();
            }

            if (!GH.DetailsView.selectedIssueKey) {
                GH.DetailsView._getIssueElement().remove();
                return;
            }

            GH.DetailsView.trigger('before:render');

            GH.DetailsView.load(null, detailsViewLoadReason);

            GH.RapidBoard.ViewController.handleResizeEvent();

            GH.DetailsView._postShow();
        });
    };

    GH.DetailsView.getWidth = function () {
        var storedWidthPercentage = GH.RapidBoard.State.getDetailViewWidth();
        var totalWidth = AJS.$("#ghx-backlog").width() + AJS.$("#ghx-detail-view").width();
        var calculatedWidth = storedWidthPercentage * totalWidth;

        var width = Math.max(calculatedWidth, GH.DetailsView.minWidth);

        return Math.min(GH.DetailsView.getMaxWidth(), width);
    };

    GH.DetailsView.initTooltips = function () {
        // for showing tooltip (Tipsy) on any element in the detail view.
        // how to use:
        //  - add the tooltip text in the element's DOM under attribute "data-tooltip"
        //  - Add the selector of that element into the array below
        // that's it
        //
        // The reason why we don't use the generic selector [data-tooltip] is: JIRA platform's editable fields may have
        // its tooltip (for example "status" field). If we use [data-tooltip] as a all-in-one selector, we will show a
        // duplicated tooltip
        var targetElementSelectors = ['.ghx-statistic-group [data-tooltip]'];

        GH.Tooltip.tipsify({
            selector: targetElementSelectors.join(', '),
            context: '#ghx-detail-contents',
            html: true
        });
    };

    /**
     * Hides the details view
     */
    GH.DetailsView.hide = function () {
        // abort any existing requests
        if (GH.DetailsView.currentPromise && GH.DetailsView.currentPromise.abort) {
            GH.DetailsView.currentPromise.abort();
        }
        // finish any ongoing edits
        GH.DetailsFieldEdit.completeEdits();

        // clear the container
        GH.DetailsView.getContainer().empty();
        AJS.$('#gh').removeClass('js-ghx-detail-loading').removeClass('js-ghx-detail-loaded');
        AJS.$(GH).trigger(GH.DetailsView.EVENT_DETAIL_VIEW_CLOSED);

        GH.RapidBoard.ViewController.handleResizeEvent();

        GH.DetailsView._initialisationPromise.done(function () {
            GH.DetailsView._postHide();
        });
    };

    GH.DetailsView.resizeLastSectionSpacer = function () {
        var $detailsNavContent = AJS.$('#js-detail-nav-content');
        var $detailsSections = $detailsNavContent.find('.ghx-detail-section');
        var $lastSectionSpacer;

        // add extra space in the last section if the view port is bigger than the content to allow to scroll to the bottom

        // get the current spacer height, if existing, to include it in the calculation
        var lastSectionSpacerPreviousHeight = 0;
        $lastSectionSpacer = $detailsNavContent.data('$lastSectionSpacer');
        if ($lastSectionSpacer) {
            lastSectionSpacerPreviousHeight = $lastSectionSpacer.height();
        }

        var viewPortHeight = $detailsNavContent.height();
        var $lastSection = $detailsSections.last();
        var lastSectionHeight = $lastSection.height() - lastSectionSpacerPreviousHeight;
        if (viewPortHeight > lastSectionHeight) {
            if (!$lastSectionSpacer) {
                $detailsNavContent.data('$lastSectionSpacer', $lastSectionSpacer = AJS.$('<div>', { css: { clear: 'both' } }).appendTo($lastSection));
            }

            // seems that IE calculates the size we want, but the others browser add an extra of 32px, we need to remove
            // that from the spacer height to not get extra scrolling space
            var empiricalExtraHeight = !AJS.$.browser.msie ? 40 : 0;

            $lastSectionSpacer.height(viewPortHeight - lastSectionHeight - empiricalExtraHeight);
        }
    };

    GH.DetailsView.selectNavItem = function (sectionId) {
        var triggerAnalyticsEventOnScrollEnd = _.debounce(function (sectionId) {
            // Trigger analytics event [SW-2194]
            GH.DetailsView.triggerAnalyticsEventForAction('scroll', { sectionId: sectionId });
        }, 150);

        var $detailsNavItems = AJS.$('.ghx-detail-nav-menu .ghx-detail-nav-item');

        var $selectedNavItem = $detailsNavItems.filter('.ghx-selected');
        if ($selectedNavItem.find("a").attr("href") !== '#' + sectionId) {
            triggerAnalyticsEventOnScrollEnd(sectionId);
        }

        $selectedNavItem.removeClass('ghx-selected');

        var $currentSection = AJS.$('#' + sectionId);

        // If tab specifies tabIconId, select by tabIconId instead
        var tabIconId = $currentSection.attr('ghx-tab-icon-id');
        if (tabIconId) {
            sectionId = tabIconId;
        }

        $detailsNavItems.find('a').filter(function () {
            return this.hash === '#' + sectionId;
        }).closest('.ghx-detail-nav-item').addClass('ghx-selected');
    };

    // fixedElement stuff start

    /**
     * Resize the column size.
     */
    GH.DetailsView.updateSize = function () {
        // delegate to our selected renderer
        GH.DetailsView.opts.updateSizeHandler();
        GH.DetailsView.resizeLastSectionSpacer();
        GH.DetailsView.getContainer().width(GH.DetailsView.getWidth());
        GH.BacklogView.checkBacklogWidth();
        // update max width for details view when sizes change
        AJS.$("#ghx-detail-view").resizable("option", "maxWidth", GH.DetailsView.getMaxWidth());
    };

    /**
     * Returns the fixed element data (the fixed element size is managed by the ViewController)
     */
    GH.DetailsView.getFixedElement = function () {
        return {
            element: GH.DetailsView.getContainer(),
            resizeCallback: GH.DetailsView.updateSize
        };
    };

    // fixedElement stuff end

    GH.DetailsView.reload = function (detailsViewLoadReason) {
        var model = GH.DetailsFieldEdit.model;
        // TODO: Figure out a smarter way to reload the details view
        // if we aren't in the middle of an edit
        if (!model || !model.getEditingCount()) {
            GH.DetailsView.show(detailsViewLoadReason);
        }
    };

    /**
     * Loads the detail view content, displaying a spinner while loading
     */
    GH.DetailsView.load = function (savedFieldId, loadReason) {
        AJS.$('#gh').addClass('js-ghx-detail-loading').removeClass('js-ghx-detail-loaded');

        if (!GH.Features.EDITABLE_DETAIL_VIEW_ENABLED.isEnabled() || !loadReason) {
            GH.DetailsView.loadContent(savedFieldId, GH.DetailsView.renderDetails);
        } else if (savedFieldId && GH.DetailsView.hasEditsInProgress()) {
            GH.DetailsView._Reloader.reload(loadReason, savedFieldId, GH.DetailsView.renderDetailsSingleField);
        } else {
            GH.DetailsView._Reloader.reload(loadReason);
        }
    };

    GH.DetailsView.hasEditsInProgress = function () {
        return GH.DetailsView.getEditsInProgress().length > 0;
    };

    GH.DetailsView.getEditsInProgress = function () {
        var edits = GH.DetailsView._InlineEditor && GH.DetailsView._InlineEditor.getEditsInProgress();
        return edits || [];
    };

    /**
     * Updates the issue details view.
     */
    GH.DetailsView.loadContent = function (savedFieldId, onSuccess) {
        // display a blanket with a spinner in the middle of Detail View
        // Unless this is a load due to a field being saved.
        if (!savedFieldId) {
            GH.DetailsView.showSpinner("load-content-spinner");
        }

        var requestData = {
            rapidViewId: GH.DetailsView.rapidViewId,
            issueIdOrKey: GH.DetailsView.selectedIssueKey,
            loadSubtasks: GH.DetailsView.opts.showSubtaskTab
        };

        if (GH.DetailsObjectFactory.isInlineEditableEnabled() && GH.DetailsView._InlineEditor.getIssueAttachmentQuery()) {
            requestData = _.extend(requestData, GH.DetailsView._InlineEditor.getIssueAttachmentQuery());
        }

        // issue an ajax request
        GH.DetailsView.currentPromise = GH.Ajax.get({
            url: '/xboard/issue/details.json',
            data: requestData
        }, 'detailView');

        GH.DetailsView.currentPromise.done(function (data) {
            // ignore content if it doesn't match expected key
            if (data.key != GH.DetailsView.selectedIssueKey) {
                // ignore
                return;
            }

            // setup the model
            GH.DetailsObjectFactory.getDetailsModel().setViewData(data);

            if (!_.isUndefined(onSuccess) && _.isFunction(onSuccess)) {
                onSuccess(savedFieldId);
            }

            GH.DetailsView.currentPromise = undefined;

            GH.DetailsView.makeResizable();

            GH.BacklogView.checkBacklogWidth();
        }).always(function () {
            // hide a blanket with a spinner in the middle of Detail View as long as no other code wants to take
            // ownership of the spinner.
            GH.DetailsView.hideSpinner("load-content-spinner");
        });
    };

    /**
     * @param spinnerOwnerId similar to event namespace, by providing a value, you could say: I will hide the spinner as long as
     *      it's still mine
     */
    GH.DetailsView.showSpinner = function (spinnerOwnerId) {
        AJS.$('.ghx-detail-view-blanket').show().spin("large").attr('spinnerOwnerId', !spinnerOwnerId ? '' : spinnerOwnerId);
    };

    /**
     * @param spinnerOwnerId when provided, the spinner can only be hidden if its id has the same value. If undefined, the spinner
     *      will be hidden anyway.
     */
    GH.DetailsView.hideSpinner = function (spinnerOwnerId) {
        var $blanket = AJS.$('.ghx-detail-view-blanket');
        if (!spinnerOwnerId || $blanket.attr('spinnerOwnerId') === spinnerOwnerId) {
            $blanket.hide().spin(false).attr('spinnerOwnerId', '');
        }
    };

    function getElementWidthOptimised($el) {
        // We are assuming that hidden element has 0 width. This is optimized to avoid layout trashing when rendering
        // big boards because normally jQuery will show and hide an element to measure its dimensions.
        if ($el.css('display') === 'none') {
            return 0;
        } else {
            return $el.width();
        }
    };

    GH.DetailsView.getBacklogElementWidth = function () {
        return getElementWidthOptimised(AJS.$('#ghx-backlog'));
    };

    GH.DetailsView.getDetailViewElementWidth = function () {
        return getElementWidthOptimised(AJS.$('#ghx-detail-view'));
    };

    GH.DetailsView.getMaxWidth = function () {
        return Math.max(GH.DetailsView.minWidth, (GH.DetailsView.getDetailViewElementWidth() + GH.DetailsView.getBacklogElementWidth()) / 2);
    };

    GH.DetailsView.makeResizable = function () {

        AJS.$("#ghx-detail-view").resizable("destroy");

        AJS.$("#ghx-detail-view").resizable({
            handles: {
                w: AJS.$("#js-sizer")
            },
            resize: GH.BacklogView.checkBacklogWidth,
            stop: GH.DetailsView.storeWidth,
            maxWidth: GH.DetailsView.getMaxWidth(),
            minWidth: GH.DetailsView.minWidth
        });

        GH.Tooltip.tipsify({
            selector: '#js-sizer',
            gravity: 'e'
        });
    };

    GH.DetailsView.storeWidth = function () {
        var detailViewWidth = GH.DetailsView.getDetailViewElementWidth();
        var backlogWidth = AJS.$("#ghx-backlog").width();

        var detailViewPercentage = detailViewWidth <= GH.DetailsView.minWidth ? 0 : detailViewWidth / (detailViewWidth + backlogWidth);
        GH.RapidBoard.State.setDetailViewWidth(detailViewPercentage);

        var isSmallWidth = function isSmallWidth(detailViewWidth) {
            return detailViewWidth >= GH.DetailsView.minWidth && detailViewWidth <= 500;
        };
        var isMediumWidth = function isMediumWidth(detailViewWidth) {
            return detailViewWidth > 500 && detailViewWidth <= 600;
        };
        var isLargeWidth = function isLargeWidth(detailViewWidth) {
            return detailViewWidth > 600;
        };

        // Update the scroll offsets and header width once when resizing is done
        GH.BacklogView.adjustStickyHeader();
        // Trigger analytics event [SW-2194]
        GH.DetailsView.triggerAnalyticsEventForAction('resize', { targetWidth: isSmallWidth(detailViewWidth) ? 'small' : isMediumWidth(detailViewWidth) ? 'medium' : isLargeWidth(detailViewWidth) ? 'large' : '' });
    };

    GH.DetailsView.renderDetailsSingleField = function (savedFieldId) {
        var viewData = GH.DetailsObjectFactory.getDetailsModel().viewData;

        // render the contents
        var detailContents = AJS.$(GH.tpl.detailview.renderIssueDetails({
            issueDetails: viewData,
            canClose: GH.DetailsView.opts.canClose,
            showActionsCog: GH.DetailsView.opts.showActionsCog && GH.UserData.hasUser() // only show cog to logged in users
        }));
        // render all fields
        GH.DetailsObjectFactory.getDetailsFieldRenderer().renderFields(viewData, detailContents, GH.DetailsView.opts);

        // replace the content
        GH.DetailsView._setIssueElement(detailContents, savedFieldId);
    };

    /**
     * Renders the details view given the provided issue data.
     * @fires GH.DetailsView#after:render
     */
    GH.DetailsView.renderDetails = function () {
        GH.DetailsView._initialisationPromise.done(function () {

            AJS.$(GH).trigger(GH.DetailsView.EVENT_DETAIL_VIEW_UPDATE_STARTED);

            var viewData = GH.DetailsObjectFactory.getDetailsModel().viewData;

            // render the contents
            var detailContents = AJS.$(GH.tpl.detailview.renderIssueDetails({
                issueDetails: viewData,
                canClose: GH.DetailsView.opts.canClose,
                showActionsCog: GH.DetailsView.opts.showActionsCog && GH.UserData.hasUser() // only show cog to logged in users
            }));

            // render all fields
            GH.DetailsObjectFactory.getDetailsFieldRenderer().renderFields(viewData, detailContents, GH.DetailsView.opts);

            // register events
            GH.DetailsView.registerAllListeners(detailContents);

            // replace the content
            GH.DetailsView._setIssueElement(detailContents);

            // remote issue links loading
            if (!GH.Features.EDITABLE_DETAIL_VIEW_ENABLED.isEnabled() && viewData.issueLinks.linkingEnabled) {
                GH.DetailsIssueLinks.loadRemoteLinks(viewData.issueLinks);
            }

            // image gallery support, only setup if we got image attachments
            if (viewData.attachments && !_.isEmpty(viewData.attachments.imageAttachments)) {
                GH.DetailFancyBox.initFancyBox();
            }

            // updating the size
            GH.DetailsView.updateSize();

            // iconfont for label's delete/close
            AJS.$('.js-epic-remove').find('.aui-icon-close').addClass('aui-icon aui-icon-small aui-iconfont-remove-label ghx-iconfont');

            // hide the loading
            AJS.$('#gh').removeClass('js-ghx-detail-loading').addClass('js-ghx-detail-loaded');

            GH.DetailsViewScrollTracker.init({
                onSelectedNavChanged: GH.DetailsView.selectNavItem
            });

            GH.DetailsViewScrollTracker.afterDetailsViewLoaded({
                issueId: viewData.id
            });

            GH.DetailsView.trigger('after:render');
            // tell third party tabs that the details are drawn and they
            // are free to make their content more "live"
            JIRA.trigger(GH.DetailsView.API_EVENT_DETAIL_VIEW_UPDATED, {
                issueId: viewData.id,
                issueKey: viewData.key,
                rapidViewId: GH.DetailsView.rapidViewId,
                mode: GH.DetailsView.getMode()
            });

            GH.DetailsView._postRenderDetails(detailContents);
        });
    };

    GH.DetailsView.registerAllListeners = function (detailContents) {
        // create drop down for issue actions cog
        if (detailContents.find("#ghx-detail-head .aui-list").length > 0) {
            var removeActionsDropdown = function removeActionsDropdown() {
                AJS.$(actionsDropdown).each(function (index, dropdown) {
                    dropdown.hide();
                    //remove the dropdown from the page
                    if (dropdown.layerController.$content) {
                        dropdown.layerController.$content.remove();
                    }
                });
                AJS.$(GH).unbind(GH.DetailsView.EVENT_DETAIL_VIEW_CLOSED, removeActionsDropdown);
                AJS.$(GH).unbind(GH.DetailsView.EVENT_DETAIL_VIEW_UPDATE_STARTED, removeActionsDropdown);
            };

            // Remove actions menu when details view is closed


            var actionsDropdown = AJS.Dropdown.create({
                trigger: detailContents.find("#ghx-detail-head .ghx-actions"),
                content: detailContents.find("#ghx-detail-head .aui-list"),
                alignment: AJS.RIGHT
            });

            AJS.$(GH).bind(GH.DetailsView.EVENT_DETAIL_VIEW_CLOSED, removeActionsDropdown);
            // Remove actions menu when we refresh the contents of the panel
            AJS.$(GH).bind(GH.DetailsView.EVENT_DETAIL_VIEW_UPDATE_STARTED, removeActionsDropdown);
        }

        // bind the "More Actions" item to show the Operations Dialog
        detailContents.find('#ghx-more-actions').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.showOperationsDialog();
        });

        // bind click for analytics
        detailContents.find('.js-issueaction-container-cog .js-issueaction').click(function (e) {
            var actionId = AJS.$(e.currentTarget).attr('id');
            GH.DetailsView.analytics.trigger(actionId); // SAFE
        });

        // bind create attachment actions
        detailContents.find('.js-create-attachment').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.attachFilesSelectedIssue();
        });

        GH.DetailsView.registerListenersForSubtask(detailContents);

        // bind link issue
        detailContents.find('.js-link-issue').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.linkSelectedIssue();
        });

        detailContents.find('.js-remove-issue-from-sprint').click(function (e) {
            e.preventDefault();
            // Do a thing
            var issueData = GH.DetailsObjectFactory.getDetailsModel().viewData;
            // TODO: fixes edge case where user navigates to a visible but deleted issue and then invokes the action.
            //       at that point the detail view has forgotten its issueData
            if (!issueData) {
                return;
            }
            var issueKey = issueData.key;

            GH.IssueActions.removeIssuesFromSprint(GH.DetailsView.rapidViewId, issueKey);
        });

        detailContents.find('.js-flag-issue').click(function (e) {
            e.preventDefault();
            GH.DetailsView.flagCurrentIssue(true, false);
        });

        detailContents.find('.js-flag-issue-and-comment').click(function (e) {
            e.preventDefault();
            GH.DetailsView.flagCurrentIssue(true, true);
        });

        detailContents.find('.js-unflag-issue').click(function (e) {
            e.preventDefault();
            GH.DetailsView.flagCurrentIssue(false, false);
        });

        detailContents.find('.js-unflag-issue-and-comment').click(function (e) {
            e.preventDefault();
            GH.DetailsView.flagCurrentIssue(false, true);
        });

        GH.DetailsView.registerHeaderFieldsListeners(detailContents);

        // Trigger analytics event [SW-2193]
        detailContents.find("#ghx-detail-head .ghx-actions").click(function () {
            AJS.trigger('analytics', { name: 'gh.issueaction.issuedetail.open.more.actions' });
        });

        GH.DetailsView._postRegisterAllListeners(detailContents);
    };

    GH.DetailsView.registerListenersForSubtask = function ($detailsViewContents) {
        $detailsViewContents.find('.js-create-subtask').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.createSubtask();
        });
        $detailsViewContents.find('.js-edit-subtask').click(function (e) {
            e.preventDefault();
            if (!GH.DetailsView.setSubtaskIssueOverwrite(AJS.$(this))) {
                return;
            }
            GH.IssueOperationShortcuts.editSelectedIssue();
        });
        $detailsViewContents.find('.js-delete-subtask').click(function (e) {
            e.preventDefault();
            if (!GH.DetailsView.setSubtaskIssueOverwrite(AJS.$(this))) {
                return;
            }
            GH.IssueOperationShortcuts.deleteSelectedIssue();
        });
    };

    /**
     * Register listeners for fields which appear in the Header. Since the header might be reloaded independently, we
     * need to be able to re-register listeners just for this section.
     */
    GH.DetailsView.registerHeaderFieldsListeners = function (detailContents) {
        // bind log work action
        detailContents.find('.js-log-work').click(function (e) {
            e.preventDefault();
            GH.IssueOperationShortcuts.logWorkOnSelectedIssue();
        });
    };

    GH.DetailsView.setSubtaskIssueOverwrite = function (actionElem) {
        var subtaskIssue = actionElem.parents('.js-subtask-issue');
        var issueKey = subtaskIssue.attr('data-issue-key');
        var issueId = parseInt(subtaskIssue.attr('data-issue-id'), 10);
        if (issueKey && issueId) {
            // set the overwrite
            GH.IssueOperationShortcuts.setOverrideSelectedIssue({
                id: issueId,
                key: issueKey
            });
            return true;
        } else {
            return false;
        }
    };

    GH.DetailsView.handleTabSelection = function (event, data) {
        // return if not correct tab group
        if (data.pane.parents('#ghx-detail-nav').length === 0) {
            return;
        }

        // clicked tab ID
        var tabId = data.pane.attr('id');
        GH.DetailsViewScrollTracker.setSelectedTab(tabId);
    };

    // set the Detail View tab
    GH.DetailsView.setTab = function (tabId) {
        var tab = AJS.$('a[href=#' + tabId + ']');
        if (tab.length > 0) {
            AJS.tabs.change(tab);
        }
    };

    /**
     * When an issue is updated, it could have come from inline edit of the primary statistic. In this case, we might
     * need to tweak the UI.
     */
    GH.DetailsView.handleIssueUpdated = function (event, data) {
        GH.log(event.type + " from source " + data.source + " handled", GH.Logger.Contexts.event);

        // ensure the updated issue is still the one shown currently. Ignore update otherwise
        var updatedIssueKey = GH.DetailsObjectFactory.getDetailsModel().getIssueKeyForId(data.issueId);
        if (!updatedIssueKey) {
            GH.log('Unable to find key for updated issue id ' + data.issueId + ". Ignoring update", GH.Logger.Contexts.event);
            return;
        }
        if (updatedIssueKey !== GH.DetailsView.selectedIssueKey) {
            return;
        }

        // reload the statistic container if the field updated was something that changes the header
        if (!GH.Features.EDITABLE_DETAIL_VIEW_ENABLED.isEnabled() && !_.isUndefined(data.fieldId)) {
            // check whether the edited field changes the header
            var headerChanged = GH.DetailsObjectFactory.getDetailsModel().isFieldInSection(data.fieldId, 'estimate') || GH.DetailsObjectFactory.getDetailsModel().isFieldInCategory(data.fieldId, 'timetracking');

            // if so reload the content and update the header
            if (headerChanged) {
                GH.DetailsView._handleHeaderChanged(data);
            }
        }
    };

    /**
     * Renders the details view given the provided issue data.
     */
    GH.DetailsView.renderDetailsHeader = function () {
        var viewData = GH.DetailsObjectFactory.getDetailsModel().viewData;

        // get the existing contents
        var detailContents = AJS.$('#ghx-detail-issue');

        // re-render only header fields
        GH.DetailsObjectFactory.getDetailsFieldRenderer().renderDetailsHeaderFields(viewData, detailContents, GH.DetailsView.opts);

        // register event listeners
        GH.DetailsView.registerHeaderFieldsListeners(detailContents);

        // updating the size
        GH.DetailsView.updateSize();

        // hide the loading
        AJS.$('#gh').removeClass('js-ghx-detail-loading').addClass('js-ghx-detail-loaded');
    };

    GH.DetailsView.hasCurrentError = function () {
        var detailContents = AJS.$('#ghx-detail-issue');
        return detailContents.find(".ghx-error").text().length > 0;
    };

    /**
     * Updates the flag of the currently displayed issue and trigger the "issueUpdated" event.
     *
     * @param {boolean} flag the flag value
     */
    GH.DetailsView.flagCurrentIssue = function (flag, withComment) {
        function updateView() {
            // will trigger the issue list referesh
            AJS.$(GH).trigger('issueUpdated', { issueId: issueData.id, source: 'detailView' });
            // reload the details view to update the menu, we don't have a way to be more granular currently
            GH.DetailsView.reload();
        }

        var issueData = GH.DetailsObjectFactory.getDetailsModel().viewData;
        if (issueData) {
            if (withComment) {
                GH.IssueFlagAndCommentAction.execute(issueData.key, flag, updateView);
            } else {
                GH.IssueActions.flag(issueData.key, flag).done(updateView);
            }
        }

        var eventName;
        if (flag) {
            eventName = withComment ? "gh.issue.ctx.menu.action.flag.with.comment" : "gh.issue.ctx.menu.action.flag";
            AJS.trigger("analytics", { name: eventName, data: { selectedIssueCount: 1 } });
        } else {
            eventName = withComment ? "gh.issue.ctx.menu.action.unflag.with.comment" : "gh.issue.ctx.menu.action.unflag";
            AJS.trigger("analytics", { name: eventName, data: { selectedIssueCount: 1 } });
        }
    };
})();

// Inline-edit dark feature switch
(function () {

    function addInlineEditCapabilities() {
        return require('jira-agile/rapid/ui/detail/detail-view-resources').done(function () {
            GH.DetailsView._InlineEditor = require('jira-agile/rapid/ui/detail/inlineedit/issue-detail-view-inline-editor');
            GH.DetailsView._Reloader = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reloader');

            GH.DetailsView._postInit = function () {};

            GH.DetailsView._postShow = function () {
                GH.DetailsView._InlineEditor.init({
                    $detailContainer: GH.DetailsView.getContainer(),
                    getRapidViewIdCallback: function getRapidViewIdCallback() {
                        return GH.DetailsView.rapidViewId;
                    },
                    loadIssueDetailCallback: GH.DetailsView.load,
                    getInlineEditableFieldsCallback: function getInlineEditableFieldsCallback() {
                        var detailsModel = GH.DetailsObjectFactory.getDetailsModel();
                        if (detailsModel.viewData && detailsModel.viewData.editable) {
                            return GH.DetailsObjectFactory.getDetailsModel().getInlineEditableFields();
                        }
                        return [];
                    },
                    getIssueIdCallback: function getIssueIdCallback() {
                        return GH.DetailsObjectFactory.getDetailsModel().viewData.id;
                    },
                    getIssueKeyCallback: function getIssueKeyCallback() {
                        return GH.DetailsObjectFactory.getDetailsModel().viewData.key;
                    },
                    getIssueSprintStatusCallback: function getIssueSprintStatusCallback() {
                        return GH.DetailsObjectFactory.getDetailsModel().viewData.sprint ? GH.DetailsObjectFactory.getDetailsModel().viewData.sprint.state : null;
                    }
                });
            };

            GH.DetailsView._postRegisterAllListeners = function () {};

            GH.DetailsView._postHide = function () {
                GH.DetailsView._InlineEditor.destroy();
            };

            GH.DetailsView._postRenderDetails = function ($detailContents) {
                GH.DetailsView._InlineEditor.fixUpDetailView($detailContents);

                GH.DetailsView._InlineEditor.loadIssueToEditor();
            };

            GH.DetailsView._handleHeaderChanged = function () {
                GH.DetailsView.load();
            };

            GH.DetailsView._onContainerSelectorSet = function (selector) {
                if (selector) {
                    AJS.$(selector).addClass('gh-editable-detail-view');
                }
            };

            GH.DetailsView.getAttachmentViewQuery = function () {
                return GH.DetailsView._InlineEditor.getIssueAttachmentQuery();
            };

            GH.DetailsView.registerListenersForTab = function (tabId, $detailsViewContents) {
                GH.DetailsView._InlineEditor.registerListenersForTab(tabId, $detailsViewContents);
                // Because we render subtasks by ourselves, not reuse from View Issue plugin.
                // Therefore, the InlineEditor should not have any knowledge about these listeners.
                if (tabId === 'sub_tasks') {
                    GH.DetailsView.registerListenersForSubtask($detailsViewContents);
                }
            };
        });
    }

    function addLegacyCapabilities() {
        GH.DetailsView.MORE_COMMENTS_COUNT = 5;

        GH.DetailsView._postInit = function () {
            // register inline dialog hacks
            GH.TimeTrackingInlineDialog.init();
        };

        GH.DetailsView._postShow = function () {};

        GH.DetailsView._postRegisterAllListeners = function ($detailContents) {
            // bind create comment actions
            $detailContents.find('.js-create-comment').click(function (e) {
                e.preventDefault();
                GH.IssueOperationShortcuts.commentSelectedIssue();
            });
        };

        GH.DetailsView._postHide = function () {};

        GH.DetailsView._postRenderDetails = function () {};

        GH.DetailsView._onContainerSelectorSet = function (selector) {
            if (selector) {
                AJS.$(selector).addClass('gh-fixed-detail-view');
            }
        };

        GH.DetailsView.getNumDisplayedComments = function () {
            return AJS.$('#ghx-detail-comments .ghx-comment').length;
        };

        GH.DetailsView._handleHeaderChanged = function (data) {
            GH.DetailsView.loadContent(null, function () {
                GH.DetailsView.handleHeaderChangeIssueUpdate(data.issueId);
            });
        };

        GH.DetailsView.handleHeaderChangeIssueUpdate = function (issueId) {
            // reload the data for detail view but only render the content for the header
            GH.DetailsView.renderDetailsHeader();

            // did we get asked to close the dialog while we were editing the field? If so, close the dialog
            if (GH.TimeTrackingInlineDialog.getShouldCloseLater()) {
                GH.TimeTrackingInlineDialog.getDialog().hide();
            }
            // reloading and re-rendering header will have closed the time tracking inline dialog if it was open
            // do we need to re-open the time tracking dialog?
            else if (GH.TimeTrackingInlineDialog.getIsOpen()) {
                    GH.TimeTrackingInlineDialog.getDialog().show();
                }

            // if we were editing a field in the header, we need to restore this field, as the original edit control would have been clobbered


            if (GH.DetailsFieldEdit.getEditingCount() > 0) {
                var editingFieldId = GH.DetailsFieldEdit.getEditingFieldIds()[0];

                if (GH.DetailsObjectFactory.getDetailsModel().isFieldInSection(editingFieldId, 'estimate') || GH.DetailsObjectFactory.getDetailsModel().isFieldInSection(editingFieldId, 'tracking') || GH.DetailsObjectFactory.getDetailsModel().isFieldInSection(editingFieldId, 'header')) {

                    // clear the editing fields
                    GH.DetailsFieldEdit.discardEdits();

                    // re-edit the field
                    GH.DetailsFieldEdit.editField(issueId, editingFieldId);
                }
            }
        };
        return AJS.$.Deferred().resolve().promise();
    }

    if (GH.DetailsObjectFactory.isInlineEditableEnabled()) {
        GH.DetailsView._initialisationPromise = addInlineEditCapabilities();
    } else {
        GH.DetailsView._initialisationPromise = addLegacyCapabilities();
    }
})();